// Fryers interface module as of 29 Jun 2000 by Michael Day
// Copyright 2000 Frye Electronics, Inc.
// Frye Instrument Packet Protocol interface program - FRYERS32.DLL
// This version is for BCB V1.0 and above for Windows
// NOTE - you *must* have the Fryers32.DLL loaded either in the
// Windows directory, or the directory where this program is located.

//---------------------------------------------------------------------------
#include <windows.h>
#ifdef __BORLANDC__
  #pragma anon_struct on
#endif  
#include "FryersC.h"

//========================== NOTES ====================================
//Example Object declaration. You declare a separate object for each
//com port you want to use with Fryers. Don't try to use two objects on
//one com port. Only one object can use a com port resource.
//To change the port number of an existing object, just call OpenPacketPort
//with the new parameters. It should be noted that it takes time to open a
//port under windows, as long as 50 to 100mS, so don't do that a lot.
//Also remember that if another porgram is using the com port, you will
//not be able to open the port with Fryers. Always shut down an MSDOS
//prompt window if i used the com port since the port won't be released
//by Windows until the MSDOS prompt window is closed.
//TFryers FryersObj;  //example object declaration.

// -- Usage --
//Declare a Fryers object variable - TFryers FryersObj;
//Call the FryersObj.OpenPacketPort() function of the object with the desired
//parameters for the port. Call FryersObj.ClosePacketPort() before closing the
//program to insure that Fryers is terminated.

//How to send a command to a Fonix instrument:
//1.Watch FryersObj.SendReady()==true to determine when you can send a command.
//2.Call FryersObj.SendCmd() with a pointer to the array containing the
//  command to be sent to the Fonix instrument.
//3.Watch FryersObj.SendReady()==true to determine when the command is complete.
//4.Call FryersObj.GetResponse() to read the response from the Fryers drive.
//5.If you need to release the instrument to do another measurment,
//  call FryersObj.QuickTerminate() to release the instrument.
//5.Verify the that the response is for the command you sent.

//Note: SendCmd, GetResponse, and QuickTerminate contain a callback that you
//can use to monitor the progress of the command for debugging, or
//if you wish to allow other Windows action to occur while waiting on a command.
//Use this this sparingly though. Remember that calling Windows will
//slow down the progress of the command. Also remember that Fryers is not
//re-entrant, so be careful that any code that is accessed from the callback
//doesn't accidently call a fryers function, which can disrupt the command
//in progress.

//Special notes:
//If you load Fryers32.dll by using the Win32API function
//"LoadLibrary", do not use the function "FreeLibrary" to detach it
//from your process. Using the "FreeLibrary" call causes the
//release of some globally allocated data while the threads that
//were created by the DLL are still running. This may cause
//exception errors such as "Access violation error". When your
//program quits running, the DLL will be detached by the system, so
//you don't need to call "FreeLibrary".

//Always try to terminate your process in normal ways. Forcing a
//process to terminate abruptly may cause some variables or
//pointers in the DLL to "hang around" which might cause
//exception errors.

//When using Visual Basic, do not call the "End" function since
//this function causes the process to terminate abruptly.

//Do not close the Fryers port twice in a row and do not use the
//Fryers calls above $FF00 once the port has been closed.

//The Send and Receive Data arrays that you pass to the Fryers driver
//do not have to be kept. Fryers copies the send data to it's own
//internal buffer so that you can release the SendData array immediately
//after calling the SendData function. When you call the GetResponse
//function, Fryers copies the received data to you array. Upon
//return from the GetResponse function, Fryers no longer needs
//the array, so you can do whatever you want with it.
//==============================================================

#define DEFAULT_POLL_TIME 100 //default poll timeout = 5.5 sec
#define DEFAULT_RSP_TIME 20   //default response timeout 1.1 second
#define DEFAULT_RCV_TIME 20   //default receive timeout 1.1 second

INT16 QTCmd[3] = {0x7fff,0,0};

//===============================================================
// Fryers32 DLL management code
//===============================================================

HINSTANCE FryersLibHandle = NULL;
tCallFryers CallFryers = NULL;  //this is __stdcall

bool __stdcall FryersLoaded(void) {
  return(FryersLibHandle != NULL);
};

bool __stdcall LoadFryers(void) {
  FryersLibHandle = LoadLibrary("fryers32.dll");
  if (FryersLibHandle != NULL) {
    CallFryers = (tCallFryers)GetProcAddress(FryersLibHandle, "CallFryers");
    if (CallFryers != NULL)
      return(TRUE);
  } return(FALSE);
}

//Don't call ReleaseFryers(). Windows will take care of it in a
//graceful manner when the program shuts down.
//bool __stdcall ReleaseFryers(void) {
//  bool fFreeResult = FreeLibrary(FryersLibHandle);
//  return(fFreeResult);
//}

//===============================================================
// FryersObj code
//===============================================================

//---------------------------------------------------------------------------------
//returns true if Fryers is loaded and open for business.
bool __stdcall TFryers::FryersReady(void) {
  if ((PortOpen == false) || (FryersLoaded() == false))
    return(false);
  else return(true);
};

//---------------------------------------------------------------------------------
//find out what the current selected baudrate is
int __stdcall TFryers::GetCurrentBaudrate(void) {
  if (FryersReady() == false)
    return(1);
  Regs.AX = 0x0ffff;  //get Fryers version number
  Regs.DX = ComPort;
  Regs.DI = 0;
  CallFryers(&Regs);
  if (Regs.DI == 0) return(9600);
   return(115200 / Regs.DI);
};

//-----------------------------------------------------------------
//Returns -1 if port not open or fryers not loaded
//else returns the current packet status Regs.AX flags
//See manual for definitions of the flags.
int __stdcall TFryers::PacketStatus(void) {
  if (FryersReady() == false)
    return(-1);
  Regs.AX = 0x0ff13;
  Regs.DX = ComPort;
  CallFryers(&Regs);
  return(Regs.AX);
};

//-----------------------------------------------------------------
//Checks on baudrate. If baud rate is wrong, tries to switch
//the baudrate (if V4.00 Fryers) to see if it will fix the problem.
//If you know the new baudrate, put the value in NewBaud.
//Otherwise use AUTO_BAUD as the value.
//Returns BaudSeek == true if busy looking for new baudrate,
//Returns Baudrate == current baudrate selection.
//Function call returns 0=no changes needed, 1=seeking new baudrate,
//-1=port not open, -2=packet mode not enabled, -3=baud baudrate given
//To fully update the baudrate, keep calling this until BaudSeek == false
//Note: if the baudrate is changed on the instrument, it can take time
//for the Fryers driver to notice the change. This is particularly true if
//the baudrate is set to 115200 when the previous baudrate was 9600.
//Fryers looks at the polls coming from the instrument to determine the
//baudrate. The Fryers autobaud routine will only update the baudrate if
//it determines that the current baudrate is wrong. If things are functioning
//normally, this call has no effect. Thus you can call it in the middle of
//a command status check routine to automatically update the baudrate if it
//changes. It should be noted that the routine looks for an error in the poll
//sequence to determine that the baudrate is incorrect. This is why it can take
//a while to determine that a new baudrate is needed since at least eight
//failed polls in a row must be received to initiate an autobaud seek, and
//at least three good polls in a row must be seen to settle into the newly
//selected baudrate.
int __stdcall TFryers::AutoBaudCheck(int NewBaud) {
  int tmp1,tmp2;

  if ((FryersReady() == false) || (FVersion < 400))
    return(false);
  Regs.DX = ComPort;
  Regs.BX = NewBaud; //0=autobaud
  Regs.AX = 0x0FF1B;
  CallFryers(&Regs);
  if ((Regs.CX & 0x0020) != 0)
    BaudSeek = true;
  else BaudSeek = false;
  tmp1 = Regs.AX;  //save status for return
  tmp2 = GetCurrentBaudrate();
  if (tmp2 > 100) Baudrate = tmp2;
  return(tmp1);
};

//----------------------------------------------------------
//Open a comport to begin packet communications through
//using the specified baudrate. ComPort is 1-4
//Baud is 9600, 19200, 38400, 57600, or 115200
//If Baud is invalid or 0, the default baudrate of 9600 is selected
//If Auto is true, autobaud is enabled while waiting on command status
//returns true if port is opened. If Fast is true, te fast transfer method
//is used to send commands. In most cases, you won't notice a difference
//in performance, but if you are pushing things, it can gain you an extra
//millisecond or so. If you have any problems, such as program crashes, or
//GPFs. use the slow transfer method which is safer.
//returns false if Fryers DLL not loaded, or Open failed.
bool __stdcall TFryers::OpenPacketPort(int IOPort, int Baud, bool Auto, bool Fast) {

  PortOpen = false;
  BaudSeek = false;
  if (FryersLoaded() == false) { //if Fryers is not loaded, load it now
    if (LoadFryers() == false)
      return(false);
  }

  if (IOPort < 1) IOPort = 1;
  if (IOPort > 15) IOPort = 1;
  ComPort = IOPort-1; //comport is the defined io port in 32bit mode
  AutoBaud = Auto;
  FastTransfer = Fast;

  FVersion = 0;
  Regs.AX = 0x0ffff;     //get Fryers version number
  Regs.DX = ComPort;
  CallFryers(&Regs);
  PacketError = NO_FRYERS;     //assume fryers not installed
  if (((Regs.DX & 0x0ffff) != 0x0ffff) || (Regs.AX < 50))
    return(false); //Win32 needs V5.00 or above
  Regs.BX = 0;
  Regs.SI = 0;
  Regs.DI = 0;
  Regs.AX = 0x0fffe;
  Regs.DX = ComPort;
  CallFryers(&Regs);
  FVersion = Regs.BX;

  Regs.AX = 0x0ff00;     //disable fryers interrupt procedure
  Regs.CX = 0x0ff00;     //this makes sure everything is kosher
  Regs.DX = ComPort;
  CallFryers(&Regs);

  Regs.AX = 0x0ff00;     //enable fryers interrupt procedure
  Regs.CX = 0x0ffff;
  Regs.DX = ComPort;
  CallFryers(&Regs);

  switch(Baud / 10) {
    case   960: Regs.AX = 0x00e3; //init to selected baudrate
                break;
    case  1920: Regs.AX = 0x0003; //no parity, and one stop bit
                break;
    case  2880: Regs.AX = 0x0023;
                break;
    case  3840: Regs.AX = 0x0043;
                break;
    case  5760: Regs.AX = 0x0063;
                break;
    case 11520: Regs.AX = 0x0083;
                break;
    default:    Regs.AX = 0x00e3; //default to 9600 if bad value given
  }//endswitch(Baudrate)
  Regs.DX = ComPort;   //init port baudrate
  CallFryers(&Regs);

  Regs.AX = 0x0ff10;     //enable fryers packet protocol
  Regs.CX = 0x0ffff;
  Regs.DX = ComPort;
  CallFryers(&Regs);

  Regs.AX = 0x0ff17;     //set poll timer
  Regs.CX = 0x00ff;
  Regs.BX = DEFAULT_POLL_TIME; //default poll timeout = 5.5 sec
  Regs.DX = ComPort;
  CallFryers(&Regs);

  Regs.AX = 0x0ff17;     //set response timer
  Regs.CX = 0x01ff;
  Regs.BX = DEFAULT_RSP_TIME; //20; //1.1 second
  Regs.DX = ComPort;
  CallFryers(&Regs);

  Regs.AX = 0x0ff17;     //set rcv timer
  Regs.CX = 0x02ff;
  Regs.BX = DEFAULT_RCV_TIME; //20; //1.1 second
  Regs.DX = ComPort;
  CallFryers(&Regs);

  PortOpen = true;
  Baudrate = GetCurrentBaudrate(); //update Baudrate to actual value
  if (Baudrate < 100) Baudrate = 9600; //if failed, force default baud
  if ((PacketStatus() & 0x01) == 0) { //verify that status works
    PacketError = BAD_FRYERS;
    return(false);
  }
  PacketError = SUCCESS;   //No error, so clear PacketError
  return(true);
};

//Close the specified packet communications com port
void __stdcall TFryers::ClosePacketPort(void) {
  if (FryersReady() == false) return;
  Regs.AX = 0x0ff10;     //disable fryers packet protocol
  Regs.CX = 0x0ff00;
  Regs.DX = ComPort;
  CallFryers(&Regs);

  Regs.AX = 0x0ff00;     //disable fryers interrupt procedure
  Regs.CX = 0x0ff00;
  Regs.DX = ComPort;
  CallFryers(&Regs);

  PortOpen = false;
};

//----------------------------------------------------
//Check to see if Fryers is ready for a command
bool __stdcall TFryers::SendReady(void) {
  int Status;

  if (FryersReady() == false) return(false);
  if (AutoBaud == true) AutoBaudCheck(AUTO_BAUD);
  Status = PacketStatus();
  if ((Status & 0x0001) != 0) return(true);
  else return(false);
};

//----------------------------------------------------
//make sure any old data is flushed
//This is Called in the send command procedure
bool __stdcall TFryers::ClearReceiveFlag(void) {
  if (FryersReady() == false) return(false);
  Regs.AX = (int)(0x0ff16);
  Regs.DX = ComPort;
  CallFryers(&Regs);
  return(true);
};

//----------------------------------------------------
//Waits for Fryers to be ready to accept a cmd
//returns false if port not opened or polls not being received
//Note: does not check for Autobaud. You should use SendReady()
//to make sure that everything is ready before calling SendCommand(0
//which calls this to make sure everything is ready.
//if Callback is not == NULL, the passed procedure will be called
//while waiting on the command status.
bool __stdcall TFryers::SendWait(PROCEDURE Callback) {
  int Status;
  if (FryersReady() == false) return(false);
  do {
    Status = PacketStatus();
    if ((Status & 0x0060) != 0) return(false);
    if ((Status & 0x0001) != 0) return(true);
    if (Callback != NULL) Callback();
  }while(true);
};

//-----------------------------------------------
//Sends a cmd to target via the rs232 port
//pData points to the data array containing the command to be sent.
//Load the command to be sent in in SendData[], then call this command.
//Note: if you do not want to hang around here waiting on the command,
//you should use the SendReady() function to wait until Fryers is ready
//for a commmand to be sent.
//if Callback is not == NULL, the passed procedure will be called
//while waiting on the command status.
bool __stdcall TFryers::SendCmd(INT16* pData, PROCEDURE Callback) {
  int i;

  if (SendWait(Callback) == false) return(false);
  ClearReceiveFlag(); //make sure any old data is flushed
  if (FastTransfer == true) {
    //Live fast and take risks!
    Regs.AX = 0x0ff11;
    Regs.CX = F_MAX_DATA_SIZE;
    Regs.DX = ComPort;
    Regs.EBX = (int)(pData); //Give fryers a pointer to the
    CallFryers(&Regs);       //command array then send it.
  }else{
    //this is a slightly slower but more reliable transfer method
    if (pData[1] > F_MAX_DATA_SIZE) return(false);
    for (i=0; i<pData[1]+2; i++) {
      Regs.AX = (int)(0x0ff23);
      Regs.DX = ComPort;
      Regs.BX = pData[i]; //load data to Fryers a word at a time
      Regs.CX = i;
      CallFryers(&Regs);
    }
    Regs.AX = int (0x0ff15); //Now send the command
    Regs.DX = ComPort;
    CallFryers(&Regs);
  }//endif(FastTransfer)
  return(true);
};

//--------------------------------------------
//Waits for response from target
//returns true when command is completed,
//returns false if command failed (low level Fryers comm failure).
//The failure is only if something went wrong at the Fryers level,
//you still must check the response from the instrument to see
//if the instrument accepted the command, or for a high level failure.
//if Callback is not == NULL, the passed procedure will be called
//while waiting on the command status.
bool __stdcall TFryers::RspWait(PROCEDURE Callback) {
  int Status;

  if (FryersReady() == false) return(false);
  while(true) {
    Status = PacketStatus();
    if ((Status & 0x00fc) != 0) return(false);
    if ((Status & 0x0001) != 0) return(true);
    if (AutoBaud == true) {
      do {
        PacketStatus();
        if (Callback != NULL) Callback();
        AutoBaudCheck(AUTO_BAUD);
      }while(BaudSeek==true);
    }else{
      if (Callback != NULL) Callback();
    }
  }
};

//--------------------------------------------------------
//Gets a response packet of integers from the rs232 port1.
//pData points to the data array where the response will be placed.
//Warning: Make sure that the array is big enough (see MAX_RCV_SIZE)
//if Callback is not == NULL, the passed procedure will be called
//while waiting on the command status.
bool __stdcall TFryers::GetResponse(INT16* pData, PROCEDURE Callback) {
  int i;
  int Size;

  if (RspWait(Callback) == false) return(false);
  if (FastTransfer == true) {
    //Live fast and take risks!
    Regs.AX = 0x0ff12;
    Regs.CX = F_MAX_DATA_SIZE;
    Regs.DX = ComPort;
    Regs.EBX = (int)(pData); //Give fryers a pointer to the
    CallFryers(&Regs);       //Response array and get the data.
  }else{
    //this is a slightly slower but more reliable transfer method
    Regs.AX = (int)(0x0ff26); //{how much data?}
    Regs.DX = ComPort;
    Regs.CX = 1;
    CallFryers(&Regs);
    Size = Regs.CX;
    if (Size > F_MAX_DATA_SIZE) return(false);
    for (i=0; i<Size+2; i++) {
      Regs.AX = (int)(0x0ff26);  //{get the data}
      Regs.DX = ComPort;
      Regs.CX = i;
      CallFryers(&Regs);
      pData[i] = (INT16)(Regs.DX);
    }
    Regs.AX = (int)(0x0ff16);  //{clear rcv flag}
    Regs.DX = ComPort;
    CallFryers(&Regs);
  }
  return(true);
};

//----------------------------------------------------
//Sends a regular quick terminate command
//returns false if port not ready
bool __stdcall TFryers::QuickTerminate(PROCEDURE Callback) {
  int i;

  if (SendWait(Callback) == false) return(false);
  if (FastTransfer == true) {
    //Live fast and take risks!
    Regs.AX = 0x0ff11;
    Regs.CX = 3;                 //total array size
    Regs.DX = ComPort;
    Regs.EBX = (int)(&QTCmd[0]); //Give fryers a pointer to the
    CallFryers(&Regs);           //command array then send it.
  }else{
    //this is a slightly slower but more reliable transfer method
    for (i=0; i<QTCmd[1]+2; i++) {
      Regs.AX = (int)(0x0ff23);
      Regs.DX = ComPort;
      Regs.BX = QTCmd[i]; //load data to Fryers a word at a time
      Regs.CX = i;
      CallFryers(&Regs);
    }
    Regs.AX = int (0x0ff15); //Now send the command
    Regs.DX = ComPort;
    CallFryers(&Regs);
  }//endif(FastTransfer)
  return(true);
};

//----------------------------------------------------
//Returns the current value of the no-poll timer
int __stdcall TFryers::GetPollTimeout(void) {
  if (FryersReady() == false) return(0);
  Regs.DX = ComPort;
  Regs.AX = 0x0FF17;
  Regs.CX = 0x0000;
  CallFryers(&Regs);
  return(Regs.AX);
};

//------------------------------------------------------------
//Set a new value for the no-poll timer
bool __stdcall TFryers::SetPollTimeout(int Value) {
  if (FryersReady() == false) return(false);
  Regs.AX = 0x0ff17;  //Select poll timer
  Regs.BX = Value;
  Regs.CX = 0x00FF;
  Regs.DX = ComPort;
  CallFryers(&Regs);
  return(true);
};

//----------------------------------------------------
//Returns the current value of the packet timer
int __stdcall TFryers::GetPacketTimer(void) {
  if (FryersReady() == false) return(0);
  Regs.DX = ComPort;
  Regs.AX = 0x0FF17;
  Regs.CX = 0x0300;
  CallFryers(&Regs);
  return(Regs.AX);
};

//------------------------------------------------------------
//clear the packet timer to zero
bool __stdcall TFryers::ClearPacketTimer(void) {
  if (FryersReady() == false) return(false);
  Regs.AX = 0x0ff17;  //Select packet timer
  Regs.BX = 0;
  Regs.CX = 0x03FF;
  Regs.DX = ComPort;
  CallFryers(&Regs);
  return(true);
};

//------------------------------------------------------------
//If AutoQT is set, each poll from the Fonix instrument will
//get a QT cmd if there is nothing to send. This is useful for
//capturing realtime curves by releasing the instrument from communication
//mode immediately after collecting the curve data so that the instrument can
//go do another measurement. You can achieve a similar effect by
//calling QuickTerminate yourself after getting the curve data,
//but it tends to be erratic due to the timing variences in Windows.
//It should be noted that Fryers can turn off the AutoQT
//if it sees a NAK in response to a QT command.
//Fryers does this to prevent a continuous stream of errors
//should the instrument not be able to handle a QT command.
//The AutoQT function is disabled when the packet mode is first
//turned on, a packet reset command is given , or a ILL response
//is received in response to the QT command
//(QT commands normally have no response).
//Warning: Do not repeatedly call the AutoQT. Each time you call it,
//the AutoQT function has to reinitialize the QT operation.
//This will cause it to miss a poll sequence. Only call AutoQT
//when you want to change it's state. You can determine the current
//state by using the CheckAT() function.
bool __stdcall TFryers::AutoQT(bool Enable) {
  if (FryersReady() == false) return(false);
  if (Enable == true)
    Regs.CX = 0x0ffff;
  else Regs.CX = 0x0ff00;
  Regs.DX = ComPort;
  Regs.AX = 0x0FF1A;
  CallFryers(&Regs);
  return(Enable);
};
//---------------------------------------
//This returns the current status of the autoQT function
//true=AutoQT is on
bool __stdcall TFryers::CheckQT(void) {
  if (FryersReady() == false) return(false);
  Regs.DX = ComPort;
  Regs.AX = 0x0FF1A;
  Regs.CX = 0x0000;
  CallFryers(&Regs);
  return((Regs.AX & 0x01) == 0x01);
};

//---------------------------------------
//Checks to see if we are communicating at all
//returns true if communicating ok, false if no poll being received
bool __stdcall TFryers::PollOK(void) {
  if ((PacketStatus() & 0x60) != 0x60) return(true);
  PacketError = NO_POLL;
  return(false);
};

//---------------------------------------
//this sets a new value in the response timer
bool __stdcall TFryers::SetRspTimer(int Value) {
  if (FryersReady() == false) return(false);
  Regs.AX = 0x0ff17;     //set response timer
  Regs.BX = Value;
  Regs.CX = 0x01ff;
  Regs.DX = ComPort;
  CallFryers(&Regs);
  return(true);
};

//---------------------------------------
//this returns the response timer's current value
int __stdcall TFryers::GetRspTimer(void) {
  if (FryersReady() == false) return(0);
  Regs.CX = 0x0100;
  Regs.DX = ComPort;
  CallFryers(&Regs);
  return(Regs.AX);
};

//==========================================================
// Example call to return Fryers version number
int GetFryersVersion(void) {
 F_RegsType R;
  if (FryersLibHandle == NULL)
    if (!LoadFryers()) return(-1);
  R.AX = 0x0FFFF;
  R.DX = 0;
  R.CX = 0;
  CallFryers(&R);
  if ((R.DX & 0x0FFFF) != 0x0FFFF)
    R.AL = 0;
  return(R.AL);
}

