Attribute VB_Name = "FryersB"

Rem FryersB.BAS  writen by Michael Day - 07/05/00
Rem Copyright 2000 Frye Electronics, Inc.
Rem Frye Instrument Packet Protocol interface program - FRYERS32.DLL
Rem This version is for Visual Basic for Windows
Rem NOTE - you *must* have the Fryers32.DLL loaded either in the
Rem Windows directory, or the directory where this program is located.

#If Win16 Then
  Declare Sub CallFryers Lib "Fryers16.DLL" Alias "#1" (FRegs As F_RegsType)
#End If
#If Win32 Then
  Declare Sub CallFryers Lib "Fryers32.DLL" Alias "#1" (FRegs As F_RegsType)
#End If

#If Win16 Then
  Public Type F_RegsType
    AX As Integer
    BX As Integer
    CX As Integer
    DX As Integer
    Di As Integer
    SI As Integer
    ES As Integer
    DS As Integer
  End Type
#End If
#If Win32 Then
  Public Type F_RegsType
    AX As Long
    BX As Long
    CX As Long
    DX As Long
    Di As Long
    SI As Long
  End Type
#End If

Public FRegs As F_RegsType

Function GetFryersVersion()
Dim rax, rdx, Size As Long
  GetFryersVersion = -1
   FRegs.AX = &HFFFF  'Rem check on version number
   FRegs.DX = 0
   CallFryers FRegs
   rax = CLng(FRegs.AX) And &HFFFF&
   rdx = CLng(FRegs.DX) And &HFFFF&
   If (rdx <> &HFFFF&) Or (rax < &H30&) Then
     Exit Function
   End If
  GetFryersVersion = rax
End Function

Rem //========================== NOTES ====================================
Rem //Example Object declaration. You declare a separate object for each
Rem //com port you want to use with Fryers. Don't try to use two objects on
Rem //one com port. Only one object can use a com port resource.
Rem //To change the port number of an existing object, just call OpenPacketPort
Rem //with the new parameters. It should be noted that it takes time to open a
Rem //port under windows, as long as 50 to 100mS, so don't do that a lot.
Rem //Also remember that if another porgram is using the com port, you will
Rem //not be able to open the port with Fryers. Always shut down an MSDOS
Rem //prompt window if i used the com port since the port won't be released
Rem //by Windows until the MSDOS prompt window is closed.

Rem // -- Usage --
Rem //Declare a Fryers object variable - TFryers FryersObj;
Rem //Create the object - FryersObj := Create.TFryers;
Rem //Call the FryersObj.OpenPacketPort() function of the object with the desired
Rem //parameters for the port. Call FryersObj.ClosePacketPort() before closing the
Rem //program to insure that Fryers is terminated.

Rem //How to send a command to a Fonix instrument:
Rem //1.Watch FryersObj.SendReady()==true to determine when you can send a command.
Rem //2.Call FryersObj.SendCmd() with a pointer to the array containing the
Rem //  command to be sent to the Fonix instrument.
Rem //3.Watch FryersObj.SendReady()==true to determine when the command is complete.
Rem //4.Call FryersObj.GetResponse() to read the response from the Fryers drive.
Rem //5.If you need to release the instrument to do another measurment,
Rem //  call FryersObj.QuickTerminate() to release the instrument.
Rem //5.Verify the that the response is for the command you sent.

Rem //Note: SendCmd, GetResponse, and QuickTerminate contain a callback that you
Rem //can use to monitor the progress of the command for debugging, or
Rem //if you wish to allow other Windows action to occur while waiting on a command.
Rem //Use this this sparingly though. Remember that calling Windows will
Rem //slow down the progress of the command. Also remember that Fryers is not
Rem //re-entrant, so be careful that any code that is accessed from the callback
Rem //doesn't accidently call a fryers function, which can disrupt the command
Rem //in progress.

Rem //Special notes:
Rem //If you load Fryers32.dll by using the Win32API function
Rem //"LoadLibrary", do not use the function "FreeLibrary" to detach it
Rem //from your process. Using the "FreeLibrary" call causes the
Rem //release of some globally allocated data while the threads that
Rem //were created by the DLL are still running. This may cause
Rem //exception errors such as "Access violation error". When your
Rem //program quits running, the DLL will be detached by the system, so
Rem //you don't need to call "FreeLibrary".

Rem //Always try to terminate your process in normal ways. Forcing a
Rem //process to terminate abruptly may cause some variables or
Rem //pointers in the DLL to "hang around" which might cause
Rem //exception errors.

Rem //When using Visual Basic, do not call the "End" function since
Rem //this function causes the process to terminate abruptly.

Rem //Do not close the Fryers port twice in a row and do not use the
Rem //Fryers calls above $FF00 once the port has been closed.

Rem //The Send and Receive Data arrays that you pass to the Fryers driver
Rem //do not have to be kept. Fryers copies the send data to it's own
Rem //internal buffer so that you can release the SendData array immediately
Rem //after calling the SendData function. When you call the GetResponse
Rem //function, Fryers copies the received data to you array. Upon
Rem //return from the GetResponse function, Fryers no longer needs
Rem //the array, so you can do whatever you want with it.
Rem //==============================================================

type INT16 = smallint;      //Fryers data is a 16 bit integer
const F_MAX_DATA_SIZE = 2100; //max size of Fryers data buffer
const AUTO_BAUD = 0;        //0=use let Fryers find the baudrate
const F_MAX_PORT = 15;
type tFryersDataArray = array[0..F_MAX_DATA_SIZE] of INT16;
type pFryersDataArray = ^tFryersDataArray;
type F_Callback = procedure;

//This defines the call to the FRYERS32.DLL for static dll load
// procedure CallFryers(var FRegs:F_RegsType); stdcall external 'Fryers32.DLL';
//However, we use the dynamic dll load method so that we can catch the Error
var CallFryers : procedure(var FIregs:F_RegsType); stdcall;
function FryersLoaded:boolean; //Use this to see if Fryers is loaded
function LoadFryers:boolean; //This is called automatically by OpenPacketPort

type TFryers = class
  private
    //void Constructor(void);
    //void Destructor(void);
    function SendWait(Callback:F_Callback):boolean;
    function RspWait(Callback:F_Callback):boolean;

  public
    Regs : F_RegsType;
    PortOpen:boolean;
    AutoBaud:boolean;
    BaudSeek:boolean;
    FastTransfer:boolean;
    FVersion:integer;
    ComPort:integer;
    Baudrate:integer;
    PacketError:integer;

    function FryersReady:boolean;
    function OpenPacketPort(IOPort:integer; Baud:integer; Auto:boolean; Fast:boolean):boolean;
    procedure ClosePacketPort;
    function ClearReceiveFlag:boolean;
    function PacketStatus:integer;
    function SendReady:boolean;
    function SendCmd(pData:pFryersDataArray; Callback:F_Callback):boolean;
    function GetResponse(pData:pFryersDataArray; Callback:F_Callback):boolean;
    function QuickTerminate(Callback:F_Callback):boolean;

    function GetCurrentBaudrate:integer;
    function AutoBaudCheck(NewBaud:integer):integer;
    function GetPollTimeout:integer;
    function SetPollTimeout(Value:integer):boolean;
    function GetPacketTimer:integer;
    function ClearPacketTimer:boolean;
    function AutoQT(Enable:boolean):boolean;
    function CheckQT:boolean;
    function PollOK:boolean;
    function SetRspTimer(Value:integer):boolean;
    function GetRspTimer:integer;
  end;

//Example send/rcv array declarations
//SendData : array[0..F_MAX_DATA_SIZE] of INT16;
//RcvData : array[0..F_MAX_DATA_SIZE] of INT16;

const SUCCESS = 0;
const RECEIVE_ERROR = 1;
const RECEIVE_OVERRUN = 2;
const SEND_OVERFLOW = 3;
const SEND_OVERRUN = 4;
const ILLEGAL_PACKET = 5;
const BAD_PACKET = 6;
const RESPONSE_ERROR = 7;
const NOGO_COMMAND = 8;
const NO_FRYERS = 9;
const NO_POLL = 10;
const BAD_FRYERS = 11;
const USER_ABORT = 255;

implementation
uses Windows; {,mmsystem;}

const DEFAULT_POLL_TIME = 100; //default poll timeout = 5.5 sec
const DEFAULT_RSP_TIME = 20;   //default response timeout 1.1 second
const DEFAULT_RCV_TIME = 20;   //default receive timeout 1.1 second

const QTCmd : array[0..2] of INT16 = ($7fff,0,0);

//===============================================================
// Fryers32 DLL management code
//===============================================================

const FryersLibHandle : integer = 0;

//returns true if Fryers DLL is loaded
function FryersLoaded:boolean;
begin
  FryersLoaded := FryersLibHandle <> NULL;
end;

{try to load fryers32.dll}
{returns false if cannot find Fryers}
{this is only used in 32 bit mode}
function LoadFryers:boolean;
begin
  LoadFryers := true;
  if FryersLibHandle > 0 then Exit;
  LoadFryers := false;
  {$IFDEF WIN32}
    FryersLibHandle := LoadLibrary('fryers32.dll');
  {$ELSE}
    FryersLibHandle := LoadLibrary('fryers16.dll');
  {$ENDIF}
  if FryersLibHandle < 32 then
  begin
    {showmessage('Unable to load fryers32.dll');}
    Exit;
  end;
  @CallFryers := GetProcAddress(FryersLibHandle,'CallFryers');
  if @CallFryers = nil then
  begin
   { showmessage('Failed to lookup CallFryers');}
    Exit;
  end;
  LoadFryers := true;
 end;

//Don't call ReleaseFryers(). Windows will take care of it in a
//graceful manner when the program shuts down.
function ReleaseFryers:boolean;
begin
   ReleaseFryers := false;
   if FryersLibHandle <= 0 then Exit;
   FreeLibrary(FryersLibHandle);
   ReleaseFryers := true;
end;

//===============================================================
// FryersObj code
//===============================================================

//---------------------------------------------------------------------------------
//This returns true if the Fryers drive is loaded and ready for business
function TFryers.FryersReady:boolean;
begin
  if not(Assigned(Self)) then FryersReady := false
  else FryersReady := ((PortOpen = true) and (FryersLoaded() = true));
end;

//---------------------------------------------------------------------------------
//find out what the current selected baudrate is
function TFryers.GetCurrentBaudrate:integer;
begin
  GetCurrentBaudrate := 1;
  if (FryersReady = false) then Exit;

  Regs.AX := $0ffff;  //get Fryers version number
  Regs.DX := ComPort;
  Regs.DI := 0;
  CallFryers(Regs);
  if (Regs.DI = 0) then GetCurrentBaudrate := 9600
  else GetCurrentBaudrate := 115200 div Regs.DI;
end;

//-----------------------------------------------------------------
//Returns -1 if port not open or fryers not loaded
//else returns the current packet status Regs.AX flags
//See manual for definitions of the flags.
function TFryers.PacketStatus:integer;
begin
  PacketStatus := -1;
  if (FryersReady = false) then Exit;
  Regs.AX := $0ff13;
  Regs.DX := ComPort;
  CallFryers(Regs);
  PacketStatus := Regs.AX;
end;

//-----------------------------------------------------------------
//Checks on baudrate. If baud rate is wrong, tries to switch
//the baudrate (if V4.00 Fryers) to see if it will fix the problem.
//If you know the new baudrate, put the value in NewBaud.
//Otherwise use AUTO_BAUD as the value.
//Returns BaudSeek == true if busy looking for new baudrate,
//Returns Baudrate == current baudrate selection.
//Function call returns 0=no changes needed, 1=seeking new baudrate,
//-1=port not open, -2=packet mode not enabled, -3=baud baudrate given
//To fully update the baudrate, keep calling this until BaudSeek == false
//Note: if the baudrate is changed on the instrument, it can take time
//for the Fryers driver to notice the change. This is particularly true if
//the baudrate is set to 115200 when the previous baudrate was 9600.
//Fryers looks at the polls coming from the instrument to determine the
//baudrate. The Fryers autobaud routine will only update the baudrate if
//it determines that the current baudrate is wrong. If things are functioning
//normally, this call has no effect. Thus you can call it in the middle of
//a command status check routine to automatically update the baudrate if it
//changes. It should be noted that the routine looks for an error in the poll
//sequence to determine that the baudrate is incorrect. This is why it can take
//a while to determine that a new baudrate is needed since at least eight
//failed polls in a row must be received to initiate an autobaud seek, and
//at least three good polls in a row must be seen to settle into the newly
//selected baudrate.
function TFryers.AutoBaudCheck(NewBaud:integer):integer;
var tmp1,tmp2:integer;
begin
  AutoBaudCheck := -1;
  if ((FryersReady = false) or (FVersion < 400)) then Exit;

  Regs.DX := ComPort;
  Regs.BX := NewBaud; //0=autobaud
  Regs.AX := $0FF1B;
  CallFryers(Regs);
  if ((Regs.CX and $0020) <> 0) then
    BaudSeek := true
  else BaudSeek := false;
  tmp1 := Regs.AX;  //save status for return
  tmp2 := GetCurrentBaudrate();
  if (tmp2 > 100) then Baudrate := tmp2;
  AutoBaudCheck := tmp1;
end;

//----------------------------------------------------------
//Open a comport to begin packet communications through
//using the specified baudrate. ComPort is 1-4
//Baud is 9600, 19200, 38400, 57600, or 115200
//If Baud is invalid or 0, the default baudrate of 9600 is selected
//If Auto is true, autobaud is enabled while waiting on command status
//returns true if port is opened. If Fast is true, te fast transfer method
//is used to send commands. In most cases, you won't notice a difference
//in performance, but if you are pushing things, it can gain you an extra
//millisecond or so. If you have any problems, such as program crashes, or
//GPFs. use the slow transfer method which is safer.
//returns false if Fryers DLL not loaded, or Open failed.
function TFryers.OpenPacketPort(IOPort:integer; Baud:integer; Auto:boolean; Fast:boolean):boolean;
begin
  OpenPacketPort := false;
  if not(Assigned(Self)) then Exit;

  PortOpen := false;
  BaudSeek := false;
  if (FryersLoaded() = false) then //if Fryers is not loaded, load it now
  begin
    if (LoadFryers() = false) then
      Exit;
  end;

  if (IOPort < 1) then IOPort := 1;
  if (IOPort > 15) then IOPort := 1;
  ComPort := IOPort-1; //comport is the defined io port in 32bit mode
  AutoBaud := Auto;
  FastTransfer := Fast;

  FVersion := 0;
  Regs.AX := $0ffff;     //get Fryers version number
  Regs.DX := ComPort;
  CallFryers(Regs);
  PacketError := NO_FRYERS;     //assume fryers not installed
  if (((Regs.DX and $0ffff) <> $0ffff) or (Regs.AX < 50)) then
    Exit; //Win32 needs V5.00 or above

  Regs.BX := 0;
  Regs.SI := 0;
  Regs.DI := 0;
  Regs.AX := $0fffe;
  Regs.DX := ComPort;
  CallFryers(Regs);
  FVersion := Regs.BX;

  Regs.AX := $0ff00;     //disable fryers interrupt procedure
  Regs.CX := $0ff00;     //this makes sure everything is kosher
  Regs.DX := ComPort;
  CallFryers(Regs);

  Regs.AX := $0ff00;     //enable fryers interrupt procedure
  Regs.CX := $0ffff;
  Regs.DX := ComPort;
  CallFryers(Regs);

  case (Baud div 10) of
      960: Regs.AX := $00e3; //init to selected baudrate
     1920: Regs.AX := $0003; //no parity, and one stop bit
     2880: Regs.AX := $0023;
     3840: Regs.AX := $0043;
     5760: Regs.AX := $0063;
    11520: Regs.AX := $0083;
     else  Regs.AX := $00e3; //default to 9600 if bad value given
  end;//endcase(Baudrate)
  Regs.DX := ComPort;   //init port baudrate
  CallFryers(Regs);

  Regs.AX := $0ff10;     //enable fryers packet protocol
  Regs.CX := $0ffff;
  Regs.DX := ComPort;
  CallFryers(Regs);

  Regs.AX := $0ff17;     //set poll timer
  Regs.CX := $00ff;
  Regs.BX := DEFAULT_POLL_TIME; //default poll timeout = 5.5 sec
  Regs.DX := ComPort;
  CallFryers(Regs);

  Regs.AX := $0ff17;     //set response timer
  Regs.CX := $01ff;
  Regs.BX := DEFAULT_RSP_TIME; //20; //1.1 second
  Regs.DX := ComPort;
  CallFryers(Regs);

  Regs.AX := $0ff17;     //set rcv timer
  Regs.CX := $02ff;
  Regs.BX := DEFAULT_RCV_TIME; //20; //1.1 second
  Regs.DX := ComPort;
  CallFryers(Regs);

  PortOpen := true;
  Baudrate := GetCurrentBaudrate(); //update Baudrate to actual value
  if (Baudrate < 100) then Baudrate := 9600; //if failed, force default baud
  if ((PacketStatus() and $01) = 0) then //verify that status works
  begin
    PacketError := BAD_FRYERS;
    Exit;
  end;
  PacketError := SUCCESS;   //No error, so clear PacketError
  OpenPacketPort := true;
end;

//Close the specified packet communications com port
procedure TFryers.ClosePacketPort;
begin
  if (FryersReady = false) then Exit;
  Regs.AX := $0ff10;     //disable fryers packet protocol
  Regs.CX := $0ff00;
  Regs.DX := ComPort;
  CallFryers(Regs);

  Regs.AX := $0ff00;     //disable fryers interrupt procedure
  Regs.CX := $0ff00;
  Regs.DX := ComPort;
  CallFryers(Regs);

  PortOpen := false;
end;

//----------------------------------------------------
//Check to see if Fryers is ready for a command
function TFryers.SendReady:boolean;
var Status:integer;
begin
  SendReady := false;
  if (FryersReady = false) then Exit;
  if (AutoBaud = true) then AutoBaudCheck(AUTO_BAUD);
  Status := PacketStatus;
  if ((Status and $0001) <> 0) then
    SendReady := true
  else SendReady := false;
end;

//----------------------------------------------------
//make sure any old data is flushed
//This is Called in the send command procedure
function TFryers.ClearReceiveFlag:boolean;
begin
  ClearReceiveFlag := false;
  if (FryersReady = false) then Exit;
  Regs.AX := $0ff16;
  Regs.DX := ComPort;
  CallFryers(Regs);
  ClearReceiveFlag := true;
end;

//----------------------------------------------------
//Waits for Fryers to be ready to accept a cmd
//returns false if port not opened or polls not being received
//Note: does not check for Autobaud. You should use SendReady()
//to make sure that everything is ready before calling SendCommand(0
//which calls this to make sure everything is ready.
//if Callback is not == NULL, the passed procedure will be called
//while waiting on the command status.
function TFryers.SendWait(Callback:F_Callback):boolean;
var Status:integer;
begin
  SendWait := false;
  if (FryersReady = false) then Exit;
  while (true) do
  begin
    Status := PacketStatus;
    if ((Status and $0060) <> 0) then
    begin
      SendWait := false;
      Exit;
    end;
    if ((Status and $0001) <> 0) then
    begin
      SendWait := true;
      Exit;
    end;
    Callback();
  end;
end;

//-----------------------------------------------
//Sends a cmd to target via the rs232 port
//pData points to the data array containing the command to be sent.
//Load the command to be sent in in SendData[], then call this command.
//Note: if you do not want to hang around here waiting on the command,
//you should use the SendReady() function to wait until Fryers is ready
//for a commmand to be sent.
//if Callback is not == NULL, the passed procedure will be called
//while waiting on the command status.
function TFryers.SendCmd(pData:pFryersDataArray; Callback:F_Callback):boolean;
var i:integer;
begin
  SendCmd := false;
  if (SendWait(Callback) = false) then Exit;
  ClearReceiveFlag(); //make sure any old data is flushed
  if (FastTransfer = true) then
  begin
    //Live fast and take risks!
    Regs.AX := $0ff11;
    Regs.CX := F_MAX_DATA_SIZE;
    Regs.DX := ComPort;
    pointer(Regs.EBX) := pData; //Give fryers a pointer to the
    CallFryers(Regs);       //command array then send it.
  end
  else
  begin
    //this is a slightly slower but more reliable transfer method
    if (pData[1] > F_MAX_DATA_SIZE) then Exit;
    for i:=0 to (pData[1]+1) do
    begin
      Regs.AX := $0ff23;
      Regs.DX := ComPort;
      Regs.BX := pData[i]; //load data to Fryers a word at a time
      Regs.CX := i;
      CallFryers(Regs);
    end;
    Regs.AX := $0ff15; //Now send the command
    Regs.DX := ComPort;
    CallFryers(Regs);
  end; //endif(FastTransfer)
  SendCmd := true;
end;

//--------------------------------------------
//Waits for response from target
//returns true when command is completed,
//returns false if command failed (low level Fryers comm failure).
//The failure is only if something went wrong at the Fryers level,
//you still must check the response from the instrument to see
//if the instrument accepted the command, or for a high level failure.
//if Callback is not == NULL, the passed procedure will be called
//while waiting on the command status.
function TFryers.RspWait(Callback:F_Callback):boolean;
var Status:integer;
begin
  RspWait := false;
  if (FryersReady = false) then Exit;
  while (true) do
  begin
    Status := PacketStatus();
    if ((Status and $00fc) <> 0) then
    begin
      RspWait := false;
      Exit;
    end;
    if ((Status and $0001) <> 0) then
    begin
      RspWait := true;
      Exit;
    end;
    if (AutoBaud = true) then
    begin
      repeat
        PacketStatus();
        Callback();
        AutoBaudCheck(AUTO_BAUD);
      until (BaudSeek = false);
    end
    else
    begin
      Callback();
    end;
  end; //EndWhile(true)
  RspWait := true;
end;

//--------------------------------------------------------
//Gets a response packet of integers from the rs232 port1.
//pData points to the data array where the response will be placed.
//Warning: Make sure that the array is big enough (see MAX_RCV_SIZE)
//if Callback is not == NULL, the passed procedure will be called
//while waiting on the command status.
function TFryers.GetResponse(pData:pFryersDataArray; Callback:F_Callback):boolean;
var i:integer;
var Size:integer;
begin
  GetResponse := false;
  if (RspWait(Callback) = false) then Exit;
  if (FastTransfer = true) then
  begin  //Live fast and take risks!
    Regs.AX := $0ff12;
    Regs.CX := F_MAX_DATA_SIZE;
    Regs.DX := ComPort;
    Regs.EBX := integer(pData); //Give fryers a pointer to the
    CallFryers(Regs);       //Response array and get the data.
  end
  else
  begin
    //this is a slightly slower but more reliable transfer method
    Regs.AX := $0ff26; //{how much data?}
    Regs.DX := ComPort;
    Regs.CX := 1;
    CallFryers(Regs);
    Size := Regs.CX;
    if (Size > F_MAX_DATA_SIZE) then Exit;
    for i:=0 to (Size+1) do
    begin
      Regs.AX := $0ff26;  //{get the data}
      Regs.DX := ComPort;
      Regs.CX := i;
      CallFryers(Regs);
      pData[i] := INT16(Regs.DX);
    end;
    Regs.AX := $0ff16;  //{clear rcv flag}
    Regs.DX := ComPort;
    CallFryers(Regs);
  end; //endif(FastXfer)
  GetResponse := true;
end;

//----------------------------------------------------
//Sends a regular quick terminate command
//returns false if port not ready
function TFryers.QuickTerminate(Callback:F_Callback):boolean;
var i:integer;
begin
  QuickTerminate := false;
  if (SendWait(Callback) = false) then Exit;
  if (FastTransfer = true) then
  begin //Live fast and take risks!
    Regs.AX := $0ff11;
    Regs.CX := 3;                 //total array size
    Regs.DX := ComPort;
    Regs.EBX := integer(@QTCmd[0]); //Give fryers a pointer to the
    CallFryers(Regs);           //command array then send it.
  end
  else
  begin
    //this is a slightly slower but more reliable transfer method
    for i:=0 to QTCmd[1]+1 do
    begin
      Regs.AX := $0ff23;
      Regs.DX := ComPort;
      Regs.BX := QTCmd[i]; //load data to Fryers a word at a time
      Regs.CX := i;
      CallFryers(Regs);
    end;
    Regs.AX := $0ff15; //Now send the command
    Regs.DX := ComPort;
    CallFryers(Regs);
  end;//endif(FastTransfer)
  QuickTerminate := true;
end;

//----------------------------------------------------
//Returns the current value of the no-poll timer
function TFryers.GetPollTimeout:integer;
begin
  GetPollTimeOut := 0;
  if (FryersReady = false) then Exit;
  Regs.DX := ComPort;
  Regs.AX := $0FF17;
  Regs.CX := $0000;
  CallFryers(Regs);
  GetPolltimeOut := Regs.AX;
end;

//------------------------------------------------------------
//Set a new value for the no-poll timer
function TFryers.SetPollTimeout(Value:integer):boolean;
begin
  SetPollTimeOut := false;
  if (FryersReady = false) then Exit;
  Regs.AX := $0ff17;  //Select poll timer
  Regs.BX := Value;
  Regs.CX := $00FF;
  Regs.DX := ComPort;
  CallFryers(Regs);
  SetPollTimeOut := true;
end;

//----------------------------------------------------
//Returns the current value of the packet timer
function TFryers.GetPacketTimer:integer;
begin
  GetPacketTimer := 0;
  if (FryersReady = false) then Exit;
  Regs.DX := ComPort;
  Regs.AX := $0FF17;
  Regs.CX := $0300;
  CallFryers(Regs);
  GetPacketTimer := Regs.AX;
end;

//------------------------------------------------------------
//clear the packet timer to zero
function TFryers.ClearPacketTimer:boolean;
begin
  ClearPacketTimer := false;
  if (FryersReady = false) then Exit;
  Regs.AX := $0ff17;  //Select packet timer
  Regs.BX := 0;
  Regs.CX := $03FF;
  Regs.DX := ComPort;
  CallFryers(Regs);
  ClearPacketTimer := true;
end;

//------------------------------------------------------------
//If AutoQT is set, each poll from the Fonix instrument will
//get a QT cmd if there is nothing to send. This is useful for
//capturing realtime curves by releasing the instrument from communication
//mode immediately after collecting the curve data so that the instrument can
//go do another measurement. You can achieve a similar effect by
//calling QuickTerminate yourself after getting the curve data,
//but it tends to be erratic due to the timing variences in Windows.
//It should be noted that Fryers can turn off the AutoQT
//if it sees a NAK in response to a QT command.
//Fryers does this to prevent a continuous stream of errors
//should the instrument not be able to handle a QT command.
//The AutoQT function is disabled when the packet mode is first
//turned on, a packet reset command is given , or a ILL response
//is received in response to the QT command
//(QT commands normally have no response).
//Warning: Do not repeatedly call the AutoQT. Each time you call it,
//the AutoQT function has to reinitialize the QT operation.
//This will cause it to miss a poll sequence. Only call AutoQT
//when you want to change it's state. You can determine the current
//state by using the CheckAT() function.
function TFryers.AutoQT(Enable:boolean):boolean;
begin
  AutoQT := false;
  if (FryersReady() = false) then Exit;
  if (Enable = true) then
    Regs.CX := $0ffff
  else Regs.CX := $0ff00;
  Regs.DX := ComPort;
  Regs.AX := $0FF1A;
  CallFryers(Regs);
  AutoQT := Enable;
end;
//---------------------------------------
//This returns the current status of the autoQT function
//true=AutoQT is on
function TFryers.CheckQT:boolean;
begin
  CheckQT := false;
  if (FryersReady() = false) then Exit;
  Regs.DX := ComPort;
  Regs.AX := $0FF1A;
  Regs.CX := $0000;
  CallFryers(Regs);
  CheckQT := ((Regs.AX and $01) = $01);
end;

//---------------------------------------
//Checks to see if we are communicating at all
//returns true if communicating ok, false if no poll being received
function TFryers.PollOK:boolean;
begin
  PollOK := true;
  if ((PacketStatus() and $60) <> $60) then Exit;
  PacketError := NO_POLL;
  PollOK :=false;
end;

//---------------------------------------
//this sets a new value in the response timer
function TFryers.SetRspTimer(Value:integer):boolean;
begin
  SetRspTimer := false;
  if (FryersReady() = false) then Exit;
  Regs.AX := $0ff17;     //set response timer
  Regs.BX := Value;
  Regs.CX := $01ff;
  Regs.DX := ComPort;
  CallFryers(Regs);
  SetRspTimer := true;
end;

//---------------------------------------
//this returns the response timer's current value
function TFryers.GetRspTimer:integer;
begin
  GetRspTimer := 0;
  if (FryersReady() = false) then Exit;
  Regs.CX := $0100;
  Regs.DX := ComPort;
  CallFryers(Regs);
  GetRspTimer := Regs.AX;
end;

//==========================================================
// Example call to return Fryers version number
function GetFryersVersion:integer;
var R : F_RegsType;
begin
  GetFryersVersion := -1;
  if (FryersLoaded()=false) then Exit;
  R.AX := $0FFFF;
  R.DX := 0;
  R.CX := 0;
  CallFryers(R);
  if ((R.DX and $0FFFF) <> $0FFFF) then
    R.AL := 0;
  GetFryersVersion := R.AL;
end;

end.
