
{Audiometer FIPP unit for communicating with FA1x audiometers}
{Copyright 1997 Frye Electronics  -- writen by Michael Day}
{V2.00  as of 04/08/97}
unit audfipp;
interface
{$I PLATFORM.INC}

{$IFDEF WIN16}
  uses DosCrt,Windos,AudSubs;
  type F_RegsType = tRegisters;
  type SmallInt = integer;
{$ENDIF}
{$IFDEF ISDOS}
  uses Dos,AudSubs,mouse,Ticker;
  type F_RegsType = Registers;
  type SmallInt = integer;
{$ENDIF}
{$IFDEF WIN32}
  uses Windows,AudSubs;
  {$I FRYEREGS.INC}
{$ENDIF}

{$UNDEF DEBUG}
{$I-,R+}

    {$IFDEF DEBUG}
const DEATHcmd  = $00000;   {0 kill machine - debug only}
    {$ENDIF}

const GETXPNcmd = $01001;   {1 get front panel data}
      SETXPNcmd = $01002;   {2 set front panel data}
      GETPRLcmd = $01003;   {3 get latched panel data}
      GETPLCcmd = $01004;   {4 get latched panel count}
      GETFPNcmd = $01005;   {5 Get front panel data (unlatched)}
      GETBPNcmd = $01006;   {6 get buffered panel data}
      GETBPCcmd = $01007;   {7 get buffered panel count}
      GETSSTcmd = $01008;   {8 get system status info}
      GETKEYcmd = $01009;   {9 get keyboard image}
      GETCFGcmd = $0100A;   {10 get cal flags}
      SETCFGcmd = $0100B;   {11 Set calibration flags}
      GETCALcmd = $0100C;   {12 get calibration data}
      SETCALcmd = $0100D;   {13 set calibration data}

      RDEERMcmd = $0100E;   {14 read eerom}
      WREERMcmd = $0100F;   {15 write eerom}

      GETWBDcmd = $01010;   {16 get warble data}
      SETWBDcmd = $01011;   {17 set warble data}
      GETWPCcmd = $01012;   {18 get warble pulse control}
      SETWPCcmd = $01013;   {19 set warble pulse control}

      GETSQDcmd = $01014;    {20 get seq data}
      SETSQDcmd = $01015;    {21 set seq data}
      GETSQIcmd = $01016;    {22 get seq index}
      SETSQIcmd = $01017;    {23 set seq index}
      GETSQEcmd = $01018;    {24 get seq enable}
      SETSQEcmd = $01019;    {25 set seq enable}
      GETSQCcmd = $0101A;    {26 get seq control}
      GETSQAcmd = $0101B;    {27 get seq activity}

      GETPIOcmd = $0101D;    {29 get port io regs}
      SETASCcmd = $0101E;    {30 set ascii mode}
      DOFPRLcmd = $0101F;    {31 do flush latched panel buffer}
      DOFBPNcmd = $01020;    {32 do flush buffered panel buffer}

      SETBSTcmd = $01025;   {37 set level boost enable}

      RESETcmd  = 38; {--  $01026;}   {reset the audiometer}
      STATUScmd = 33+$1000; {--$01021;} {33 -- $01021; }  {get command status}
      GETVERcmd = 28;               {get audiometer version/model}

      SetPDlycmd = 73;
      GetPDlycmd = 74;


      ACK    = $0FFFC;

      DoReset      = 1; {reset the audiometer}
      GetCmdStatus = 2; {get status of last command}
      GetPrlImg    = 3; {get panel latched data}
      GetXpnImg    = 4; {get panel data (unlatched)}
      SetXpnImg    = 5; {set panel data}
      GetBufPanImg = 6; {get buffered panel image}
      GetPLCount   = 7; {get front panel count}
      GetBPCount   = 8; {get buffered panel count}
      SetBstEnable = 9; {set level boost enable}
      GetKeyImg    = 10; {get keyboard image}
      GetSysStatus = 11; {get system status info}
      GetVersion   = 12; {get audiometer version/model}

      SetCalData   = 13;    {set new cal table values}
      GetCalData   = 14;    {get current cal table values}
      GetCalFlags  = 15;    {get current cal status flags}
      SetCalFlags  = 16;    {set new cal status flags}
      DoReadEEROM   = 17;   {read eerom}
      DoWriteEEROM  = 18;   {write eerom}

      GetWarbleData = 19;   {get warble data}
      SetWarbleData = 20;   {set warble data}
      GetWPcontrol  = 21;   {get warble pulse control}
      SetWPcontrol  = 22;   {set warble pulse control}

      GetSeqData     = 23;    {get seq data}
      SetSeqData     = 24;    {set seq data}
      GetSeqIndex    = 25;    {get seq index}
      SetSeqIndex    = 26;    {set seq index}
      GetSeqEnable   = 27;    {get seq enable}
      SetSeqEnable   = 28;    {set seq enable}
      GetSeqControl  = 29;    {get seq control}
      GetSeqActivity = 30;    {get seq activity}

      GetPortIORegs  = 31;    {get port io regs}
      SetAsciiMode   = 32;    {switch to ascii mode}
      GotoMonitor    = 33;    {kill aud and return to monitor}

      GetFpnImg      = 34; {get front panel data (unlatched)}
      DoBufPanFlush  = 35; {flush buffered panel buffer}
      DoPrlFlush     = 36; {flush latched panel buffer}

      SetPollDelay   = 37;
      GetPollDelay   = 38;

      MaxProc = 40;   { maxium number procedures in call list}

      NoFail  = 0;
      BadRsp  = 1;
      BadSend = 2;
      BadXfr  = 3;
      BadPoll = 4;
      BadLogic = 5;


type bytearray = array[0..65520] of byte;

type PnlRecType = packed Record
       PBSTAT : WORD;
       PBMISC : WORD;
       LFREQ  : SmallInt;
       LLEVEL : SmallInt;
       LSRC   : SmallInt;
       LDEST  : SmallInt;
       LSIM   : SmallInt;
       LLOFFS : SmallInt;
       RFREQ  : SmallInt;
       RLEVEL : SmallInt;
       RSRC   : SmallInt;
       RDEST  : SmallInt;
       RSIM   : SmallInt;
       RLOFFS : SmallInt;
     end;
     PnlRecTypePtr = ^PnlRectype;

     BufPnlRecType = packed Record
       INDEX  : WORD;
       PBSTAT : WORD;
       PBMISC : WORD;
       LFREQ  : SmallInt;
       LLEVEL : SmallInt;
       LSRC   : SmallInt;
       LDEST  : SmallInt;
       LSIM   : SmallInt;
       LLOFFS : SmallInt;
       RFREQ  : SmallInt;
       RLEVEL : SmallInt;
       RSRC   : SmallInt;
       RDEST  : SmallInt;
       RSIM   : SmallInt;
       RLOFFS : SmallInt;
     end;

     SSTRecType = packed record
       PBstat    : word; {1}
       PBmisc    : word; {2}
       MiscStat  : word; {3}
       CalStat   : word; {4}
       RS232Stat : word; {5}
       STSpare1  : word; {6}
       STSpare2  : word; {7}
       STSpare3  : word; {8}
     end;

     KeyRecType = packed record
       KEYPBD : WORD;
       ROTSWT : ARRAY[0..8] OF BYTE;
       PATSW  : BYTE;
       KEYERR : BYTE;
       KDUMMY : BYTE;
     end;

     VerRecType = packed record
       Version : word;
       Option  : longint;
       Custom  : longint;
       Model   : word;
       SubModel : word;
     end;

     SeqEnRecType = packed record
       Mask  : word;
       Flags : word;
     end;

     SeqIndexRecType = packed record
       Buffer : word;
       Index  : word;
     end;

     SeqRecType = packed record
       Hdr : packed record
         Buffer : word;
         Index  : word;
         Count  : word;
       end;
       Data   : array[0..256] of byte;
     end;

     SeqCtrlRecType = packed record
       Buffer : word;
       Data   : array[0..63] of byte;
     end;

     SeqActRecType = packed record
       Enables : byte;
       Spare   : byte;
       Index0  : byte;
       Ctrl0   : byte;
       Index1  : byte;
       Ctrl1   : byte;
     end;

     WarbleRecType = packed record
       Buffer : word;
       Count  : word;
       Data   : array[0..256] of byte;
     end;

const ArrayMax = 1000;
type AudComBuffer = array[0..1000] of word;
     AudComProc = procedure;

     AudComObj = object
       D : packed record
         Fversion      : word;
         TrueBaudRate  : longint;
         ComPort       : word;
         IRQnum        : array[0..15] of byte;
         RsErr         : WORD;
         Active        : WORD;
         ForeGround    : boolean;
         PortOpen      : boolean;
         NoPoll        : boolean;
         XfrErr        : boolean;
         CmdFailFlag   : boolean;
         PollDelayTime : word;

         SPI : array[0..MaxProc] of procedure;
         RPI : array[0..MaxProc] of procedure;
         sbuf,rbuf : AudComBuffer;

         CmdStatus : word;
         PrlRec    : BufPnlRecType;
         BufFpnRec : BufPnlRecType;
         XpnRec    : PnlRecType;
         FpnRec    : PnlRecType;
         SSTrec    : SSTRectype;
         KeyRec    : KeyRecType;
         VerRec    : VerRecType;
         PLCount   : word;
         BPCount   : word;
         BstEnable : word;
         FailError : word;
         EEROMTable: word;
         CalFlags  : word;
         CalRec    : array[0..127] of byte;
         SeqRec    : SeqRecType;
         SeqActRec : SeqActRecType;
         SeqEnRec  : SeqEnRecType;
         SeqIndexRec: SeqIndexRecType;
         SeqCtrlRec : SeqCtrlRecType;
         WarbleRec  : WarbleRecType;
         WPCrec     : array[0..9] of word;
         PortIOrec  : array[0..63] of byte;
         Regs : F_RegsType;
       end;
       constructor Init;
       destructor Done;

       {---------------------------------------}
       {public methods}
       function  InitRS232(IOport,IRQn:word):boolean; {open pacekt communications}
       procedure CloseRS232;
       function  BlindSend(Sp,Rp:pointer):boolean;
       procedure Cmd(What:word);
       function  Fail:boolean;
       function  FailMsg:str255;
       function  CmdCheck:boolean;
       procedure SetForeground(How:boolean);

       function  EnablePacketMode:boolean;
       function  EnableTerminalMode:boolean;
       procedure Terminal;
       function  ExitASCIImode:boolean;

       {---------------------------------------}
       {semi-private methods}
       procedure AsciiInit;
       function PacketInit:boolean;
       function QTon:boolean;
       procedure UpdateFryersInfo;
       procedure ClearRcvPacket;
       function SendReady:boolean;
       function Sendwait:boolean;
       function SendCmd(Data:pointer):boolean;
       function getresponse(Data:pointer):boolean;
       function GetPacketStatus:word;
       function GetPacketTimer:word;
       procedure ClearPacketTimer;

       procedure DoSendCmd(A,C,L:word; Data:pointer);
       procedure GetCmdRsp(C,L:word; Data:pointer);
       procedure ReadRsp;
       procedure CmdFail(How:word);

       function  Tsend(S:str255):boolean;
       procedure TWait(HowLong:word);

       {--------------------------------------}
       {non-fipp comm methods}
       function GetStat:boolean;
       function GetChar:char;
       function PutStat:boolean;
       function PutChar(What:char):byte;
       function SendStr(s:str255):boolean;

     end; {AudCom object}

var AudCom : ^AudComObj;

    StatAX : word;
    StatBX : word;
    StatCX : word;
    StatDX : word;

{$IFDEF WIN32}
  function LoadFryers:boolean;

  {we use the dynamic dll load method so that we can catch the Error}
  const LibHandle : integer = 0;
  var CallFryers : procedure(var FIregs:F_RegsType); stdcall;
{$ELSE}
  procedure CallFryers(var FRegs:F_RegsType);
{$ENDIF}

implementation

{$IFDEF WIN32}
  {this defines the call to the FRYERS32.DLL for static dll load }
  {however, we will use dynamic loading so as to catch the error}
//  procedure CallFryers(var FRegs:F_RegsType); stdcall external 'Fryers32.DLL';

    {this call is used for non-dll testing of the fryers dll}
//    procedure CallFryers(var FRegs:F_RegsType);
//    begin
//      ServiceFryers(FryerDef.F_RegsType(FRegs));
//    end;
{$ELSE}
  procedure CallFryers(var FRegs:F_RegsType);
  begin
    INTR($14,FRegs);
  end;
{$ENDIF}


{$IFDEF WIN32}
  {try to load fryers32.dll}
  {returns false if cannot find Fryers}
  {this is only used in 32 bit mode}
  function LoadFryers:boolean;
  begin
    LoadFryers := true;
    if LibHandle > 0 then Exit;
    LoadFryers := false;
    LibHandle := LoadLibrary('fryers32.dll');
    if LibHandle < 32 then
    begin
      {showmessage('Unable to load fryers32.dll');}
      Exit;
    end;
    @CallFryers := GetProcAddress(LibHandle,'CallFryers');
    if @CallFryers = nil then
    begin
     { showmessage('Failed to lookup CallFryers');}
      Exit;
    end;
    LoadFryers := true;
  end;
{$ENDIF}

{*******************************************************************}
{                       private procedures                          }
{*******************************************************************}

{-------}
function AudComObj.GetPacketTimer:word;
begin
  D.Regs.dx := D.comport;
  D.Regs.ax := $ff17;
  D.Regs.cx := $0300;
  CallFryers(D.Regs);
  GetPacketTimer := D.Regs.AX;
end;

{-------}
procedure AudComObj.ClearPacketTimer;
begin
  D.Regs.ax := $ff17;    {select packet timer}
  D.Regs.cx := $03ff;
  D.Regs.bx := 0;
  D.Regs.dx := D.comport;
  CallFryers(D.Regs);
end;


{*******************************************************************}
{                    semi-private methods                           }
{*******************************************************************}

function AudComObj.GetPacketStatus:word;
begin
  D.Regs.ax := $0ff13;
  D.Regs.dx := D.comport;
  CallFryers(D.Regs);
  StatAX := D.Regs.AX;
  StatBX := D.Regs.BX;
  StatCX := D.Regs.CX;
  StatDX := D.Regs.DX;
  GetPacketStatus := D.Regs.AX;
end;


function AudComObj.QTon:boolean;
begin
  D.Regs.dx := D.comport;
  D.Regs.cx := 0;
  D.Regs.ax := $ff1a;
  CallFryers(D.Regs);
  Qton := boolean(D.Regs.AX);
end;

{-------------------------------------------------------------------}
{check on current sending status}
function AudComObj.SendReady:boolean;
begin
  with D do
  begin
    SendReady := false;
    NoPoll := false;
    XfrErr := false;
    RSerr := GetPacketStatus;

    if (RSerr and $0040) = $0040 then
    begin
      SendReady := true;    {no poll means connection lost}
      NoPoll := true;       {so make ready to abort out of here}
      Exit;                 {it is up to the programmer to check}
    end;                    {the nopoll flag after this}

    if (RSerr and $0001) = 1 then
    begin
(*
      if ((RSerr and $0002) <> 2) and (Active > 0) then {check valid rsp}
        XfrErr := true;              {and mark error if bad}
*)
      Sendready := true;    {the systen is ready to send a new cmd}
      Exit;
    end;
  end;
end;

{------------------------------------------------------------------}
{ waits for int14 to be ready to accept send cmd }
{ returns false if cmd failed (no poll) }
{ returns true if ready to send new cmd (old cmd rsp was rcvd) }

function AudComObj.SendWait:boolean;
begin
   while true do
   begin
     {$IFNDEF WIN32}
       ReadMouse;            {keep mouse updated while we wait}
     {$ENDIF}
     if SendReady then
     begin
       if D.NoPoll then
         SendWait := false {if err or no poll ret false and exit}
       else
         SendWait := true;
       Exit;
     end
   end;
end;

{------------------------------------------------------------------}
{ clear anything still hanging around in the receive packet }
procedure AudComObj.ClearRcvPacket;
begin
  D.Regs.dx := D.comport;
  D.Regs.ax := $ff16;
  CallFryers(D.Regs);
  fillchar(D.Rbuf,sizeof(D.rbuf),0);
end;

{------------------------------------------------------------------}
{ sends a cmd to target via the rs232 port}

function AudComObj.SendCmd(Data:pointer):boolean;
type wordarray = array[0..100] of word;
VAR Serr,cp,value,i:word;
begin
   {$IFDEF MSDOS}
     cp := D.comport;
     asm
       push ds
       mov dx,[cp]
       and dx,1
       lds bx,[Data]
       mov ax,$ff11
       INT $14
       pop ds
       mov [Serr],ax
     end;
   {$ELSE}
     for i := 0 to succ(wordarray(Data^)[1]) do
     begin
       value := wordarray(Data^)[i];
       D.Regs.ax := $ff23;
       D.Regs.dx := D.comport;
       D.Regs.cx := i;
       D.Regs.bx := value;
       CallFryers(D.Regs);
     end;
     D.Regs.ax := $ff15;
     D.Regs.dx := D.comport;
     CallFryers(D.Regs);
     Serr := D.Regs.ax;
   {$ENDIF}

   D.RSerr := Serr;
   if Serr and $00FC = $0000 then  {if no transfer error we're ok}
     SendCmd := true
   else
     SendCmd := false;
end;


{-----------------------------------------------------------------}
{ gets a response packet of integers from the rs232 port }

function AudComObj.GetResponse(Data:pointer):boolean;
type wordarray = array [0..100] of word;
VAR Serr,Size,cp,value,i:word;
begin
  D.Regs.ax := $ff26;
  D.Regs.dx := D.comport;
  CallFryers(D.Regs);
  Size := D.Regs.cx;

  if Size < ArrayMax then  {check if data will fit}
  begin
    {$IFDEF MSDOS}
      cp := D.comport and 1;
      asm
        push ds
        mov ax,$ff12
        mov dx,[cp]
        and dx,1
        lds bx,[Data]
        INT $14
        pop ds
        mov [Serr],ax
      end;
    {$ELSE}
      for i := 0 to succ(Size) do
      begin
        D.Regs.ax := $ff26;
        D.Regs.dx := D.comport;
        D.Regs.cx := i;
        CallFryers(D.Regs);
        value := D.Regs.dx;
        wordarray(Data^)[i] := value;
      end;
      D.Regs.ax := $ff16;
      D.Regs.dx := D.comport;
      CallFryers(D.Regs);
      Serr := D.Regs.ax;
    {$ENDIF}
  end
  else
  begin
    Serr := $ffff;
  end;
  D.RSerr := Serr;
  if (Serr and $0080) <> 0 then
    GetResponse := false
  else
    GetResponse := true;
end;


procedure AudComObj.ReadRsp;
begin
   if not(GetResponse(@D.RBuf)) then
   begin
     CmdFail(BadRsp);
     Exit;
   end;
   if D.Active > MaxProc then
   begin
     CmdFail(BadLogic);
     Exit;
   end;
   D.RPI[D.Active];
   D.Active := 0;
end;


{--------------------------------------------------------}
{ C=true cmd number; L=length of data in bytes (must be even) }
{ A=Obj cmd number;  D=pointer to data to be sent }
procedure AudComObj.DoSendCmd(A,C,L:word; Data:pointer);
begin
  with D do
  begin
    if not(SendWait) then
    begin
      CmdFail(BadPoll);
      Exit;
    end;
    if Active <> 0 then
    begin
      if XfrErr then
      begin
        Active := 0;
        CmdFail(BadXfr);
        Exit;
      end
      else
        ReadRsp;
    end;

    ClearRcvPacket;
    Sbuf[0] := C;
    Sbuf[1] := L shr 1;
    if L > 0 then move(Data^,Sbuf[2],L);
    if not(SendCmd(@Sbuf)) then
    begin
      CmdFail(BadSend);
      Exit;
    end;

    Active := A;
    if ForeGround then
    begin
      if not(SendWait) or XfrErr then
      begin
        if XfrErr then
          CmdFail(BadXfr)
        else
          CmdFail(BadPoll);
        Exit;
      end;

      ReadRsp;
    end;
  end;
end;


procedure AudComObj.GetCmdRsp(C,L:word; Data:pointer);
begin
  if (D.Rbuf[0] and $7fff) = C then
    move(D.Rbuf[2],Data^,L)
  else
    CmdFail(BadRsp);
end;

{==========================================}
{Sends array of data pointed at by Sp}
{always happens as a foreground job}
{response is left in array pointed at by Rp.}
{returns true if sent ok. returns false on faliure}

function AudComObj.BlindSend(Sp,Rp:pointer):boolean;
begin
  with D do
  begin
    BlindSend := false;
    if not(SendWait) then
    begin
      CmdFail(BadPoll);
      Exit;
    end;
    if Active <> 0 then
    begin
      if XfrErr then
      begin
        Active := 0;
        CmdFail(BadXfr);
        Exit;
      end
      else
        ReadRsp;
    end;

    ClearRcvPacket;
    if not(SendCmd(Sp)) then
    begin
       CmdFail(BadSend);
       Exit;
    end;

    if not(SendWait) or XfrErr then
    begin
      if XfrErr then
        CmdFail(BadXfr)
      else
        CmdFail(BadPoll);
      Exit;
    end;

    if Sbuf[0] <> $7fff then  {ignore Quick Terminate command}
    begin
      if not(GetResponse(Rp)) then
      begin
        CmdFail(BadRsp);
        Exit;
      end;
    end;
    BlindSend := true;
  end;
end;


{=========================================================}
procedure AudComObj.CmdFail(How:word);
begin
  D.CmdFailFlag := true;
  D.Active := 0;
  D.FailError := How;
end;


{************************************************************}
{ These are private non-object procedures - this is done for }
{ speed and ease of interfacing via an array of procedures   }
{************************************************************}

procedure DummyProc; far;
begin
end;

{$IFDEF DEBUG}
procedure KillMeProc; far;     {Kill audiometer - return to monitor}
begin                          {for development use only}
  with AudCom^ do              {not available in production software}
    DoSendCmd(GotoMonitor,DEATHcmd,0,nil);
end;
{$ENDIF}

procedure DoResetProc; far;       {reset the audiometer}
begin
  with AudCom^ do
    DoSendCmd(DoReset,RESETcmd,0,nil);
end;

procedure GetCmdStatusProc; far;  {get status of last command}
begin
  with AudCom^ do
    DoSendCmd(GetCmdStatus,STATUScmd,0,nil);
end;

procedure GetPrlImgProc; far;     {get panel latched data}
begin
  with AudCom^ do
    DoSendCmd(GetPrlImg,GETPRLcmd,0,nil);
end;

procedure GetXpnImgProc; far;     {get panel data (unlatched)}
begin
  with AudCom^ do
    DoSendCmd(GetXpnImg,GETXPNcmd,0,nil);
end;

procedure GetBufPanImgProc; far;     {get panel data (buffered)}
begin
  with AudCom^ do
    DoSendCmd(GetBufPanImg,GETBPNcmd,0,nil);
end;

procedure SetXpnImgProc; far;     {set panel data}
begin
  with AudCom^ do
    DoSendCmd(SetXpnImg,SETXPNcmd,sizeof(D.XpnRec),@D.XpnRec);
end;

procedure GetSSTImgProc; far;     {get panel data (unlatched)}
begin
  with AudCom^ do
    DoSendCmd(GetSysStatus,GETSSTcmd,0,nil);
end;

procedure GetKeyImgProc; far;     {get keyboard image}
begin
  with AudCom^ do
    DoSendCmd(GetKeyImg,GETKEYcmd,0,nil);
end;

procedure GetPLCountProc; far;     {get panel latch counter}
begin
  with AudCom^ do
    DoSendCmd(GetPLCount,GETPLCcmd,0,nil);
end;

procedure GetBPCountProc; far;     {get buffered panel counter}
begin
  with AudCom^ do
    DoSendCmd(GetBPCount,GETBPCcmd,0,nil);
end;

procedure GetVersionProc; far;     {get version info}
begin
  with AudCom^ do
    DoSendCmd(GetVersion,GETVERcmd,0,nil);
end;


procedure GetCalDataProc; far;     {get calibration data}
begin
  with AudCom^ do
    DoSendCmd(GetCalData,GETCALcmd,0,nil);
end;

procedure GetCalFlagsProc; far;  {get calibration flags}
begin
  with AudCom^ do
    DoSendCmd(GetCalFlags,GETCFGcmd,0,nil);
end;

procedure SetCalDataProc; far;  {set cal data}
begin
  with AudCom^ do
    DoSendCmd(SetCalData,SETCALcmd,sizeof(D.CalRec),@D.CalRec);
end;

procedure SetCalFlagsProc; far;  {get calibration flags}
begin
  with AudCom^ do
    DoSendCmd(SetCalFlags,SETCFGcmd,sizeof(D.CalFlags),@D.CalFlags);
end;

procedure ReadEEROMProc; far;  {read eerom table}
begin
  with AudCom^ do
    DoSendCmd(DoReadEEROM,RDEERMcmd,sizeof(D.EEROMTable),@D.EEROMTable);
end;

procedure WriteEEROMProc; far;  {write eerom table}
begin
  with AudCom^ do
    DoSendCmd(DoWriteEEROM,WREERMcmd,sizeof(D.EEROMTable),@D.EEROMTable);
end;

procedure SetSeqDataProc; far;     {set seq data}
begin
  with AudCom^ do
    DoSendCmd(SetSeqData,SETSQDcmd,sizeof(D.SeqRec.Hdr)+succ(D.SeqRec.Hdr.Count),@D.SeqRec);
end;

procedure SetSeqIndexProc; far;  {set seq index}
begin
  with AudCom^ do
    DoSendCmd(SetSeqIndex,SETSQIcmd,sizeof(D.SeqIndexRec),@D.SeqIndexRec);
end;

procedure SetSeqEnableProc; far;  {set seq enable flags}
begin
  with AudCom^ do
    DoSendCmd(SetSeqEnable,SETSQEcmd,sizeof(D.SeqEnRec),@D.SeqEnRec);
end;


procedure GetSeqDataProc; far;     {get seq data}
begin
  with AudCom^ do
    DoSendCmd(GetSeqData,GETSQDcmd,sizeof(D.SeqRec.Hdr),@D.SeqRec.Hdr);
end;

procedure GetSeqIndexProc; far;  {get seq index}
begin
  with AudCom^ do
    DoSendCmd(GetSeqIndex,GETSQIcmd,
              sizeof(D.SeqIndexRec.Buffer),@D.SeqIndexRec.Buffer);
end;

procedure GetSeqEnableProc; far;  {get seq enable flags}
begin
  with AudCom^ do
    DoSendCmd(GetSeqEnable,GETSQEcmd,0,nil);
end;

procedure GetSeqControlProc; far;  {get seq control regs}
begin
  with AudCom^ do
    DoSendCmd(GetSeqControl,GETSQCcmd,sizeof(D.SeqCtrlRec.Buffer),@D.SeqCtrlRec.Buffer);
end;

procedure GetSeqActivityProc; far;  {get seq activity regs}
begin
  with AudCom^ do
    DoSendCmd(GetSeqActivity,GETSQAcmd,0,nil);
end;


procedure GetWarbleDataProc; far;     {get warble data}
begin
  with AudCom^ do
    DoSendCmd(GetWarbleData,GETWBDcmd,
              sizeof(D.WarbleRec.Buffer),@D.WarbleRec.Buffer);
end;

procedure SetWarbleDataProc; far;     {set warble data}
begin
  with AudCom^ do
    DoSendCmd(SetWarbleData,SETWBDcmd,(D.WarbleRec.Count+2)*2,@D.WarbleRec);
end;

procedure GetWPcontrolProc; far;     {get warble/pulse control regs}
begin
  with AudCom^ do
    DoSendCmd(GetWPcontrol,GETWPCcmd,0,nil);
end;

procedure SetWPcontrolProc; far;     {set warble data}
begin
with AudCom^ do
    DoSendCmd(SetWPcontrol,SETWPCcmd,sizeof(D.WPCrec),@D.WPCrec);
end;


procedure GetPortIOProc; far;     {get port IO regs}
begin
  with AudCom^ do
    DoSendCmd(GetPortIORegs,GETPIOcmd,0,nil);
end;

procedure SetAsciiModeProc; far;     {Switch to ASCII mode}
begin
  with AudCom^ do
    DoSendCmd(SetAsciiMode,SETASCcmd,0,nil);
end;

procedure GetFpnImgProc; far;     {get front panel data (unlatched)}
begin
  with AudCom^ do
    DoSendCmd(GetFpnImg,GETFPNcmd,0,nil);
end;

procedure DoBufPanFlushProc; far;     {flush buffered panel counter}
begin
  with AudCom^ do
    DoSendCmd(DoBufPanFlush,DOFBPNcmd,0,nil);
end;

procedure DoPrlFlushProc; far;     {flush latched panel counter}
begin
  with AudCom^ do
    DoSendCmd(DoPrlFlush,DOFPRLcmd,0,nil);
end;

procedure SetBstEnableProc; far;     {set the level boost enable flag}
begin
  with AudCom^ do
    DoSendCmd(SetBstEnable,SETBSTcmd,sizeof(D.BstEnable),@D.BstEnable);
end;

procedure SetPollDelayProc; far;  {set poll delay time}
begin
  with AudCom^ do
    DoSendCmd(SetPollDelay,SETPdlycmd,sizeof(D.PollDelayTime),@D.PollDelayTime);
end;

procedure GetPollDelayProc; far;  {get poll delay time}
begin
  with AudCom^ do
    DoSendCmd(GetPollDelay,GETPdlycmd,0,nil);
end;


{========================================================}

procedure AckRsp; far;
begin
  if AudCom^.D.Rbuf[0] <> ACK then
    AudCom^.CmdFail(BadRsp);
end;

procedure GetCmdStatusRsp; far;
begin
  with AudCom^ do
    GetCmdRsp(STATUScmd,sizeof(D.CmdStatus),@D.CmdStatus);
end;

procedure GetPrlImgRsp; far;
begin
  with AudCom^ do
    GetCmdRsp(GETPRLcmd,sizeof(D.PrlRec),@D.PrlRec);
end;

procedure GetXpnImgRsp; far;
begin
  with AudCom^ do
    GetCmdRsp(GETXPNcmd,sizeof(D.XpnRec),@D.XpnRec);
end;

procedure GetBufPanImgRsp; far;
begin
  with AudCom^ do
    GetCmdRsp(GETBPNcmd,sizeof(D.BufFpnRec),@D.BufFpnRec);
end;

procedure GetSSTImgRsp; far;
begin
  with AudCom^ do
    GetCmdRsp(GETSSTcmd,sizeof(D.SSTRec),@D.SSTRec);
end;

procedure GetKeyImgRsp; far;
begin
  with AudCom^ do
    GetCmdRsp(GETKEYcmd,sizeof(D.KeyRec),@D.KeyRec);
end;

procedure GetPLCountRsp; far;
begin
  with AudCom^ do
    GetCmdRsp(GETPLCcmd,sizeof(D.PLCount),@D.PLCount);
end;

procedure GetBPCountRsp; far;
begin
  with AudCom^ do
    GetCmdRsp(GETBPCcmd,sizeof(D.BPCount),@D.BPCount);
end;

procedure GetVersionRsp; far;
begin
  with AudCom^ do
    GetCmdRsp(GETVERcmd,sizeof(D.VerRec),@D.VerRec);
end;

procedure GetCalDataRsp; far;
begin
  with AudCom^ do
    GetCmdRsp(GETCALcmd,sizeof(D.CalRec),@D.CalRec);
end;

procedure GetCalFlagsRsp; far;
begin
  with AudCom^ do
    GetCmdRsp(GETCFGcmd,sizeof(D.CalFlags),@D.CalFlags);
end;

procedure GetSeqDataRsp; far;     {get seq data}
begin
  with AudCom^ do
    GetCmdRsp(GETSQDcmd,sizeof(D.SeqRec),@D.SeqRec);
end;

procedure GetSeqIndexRsp; far;  {get seq index}
begin
  with AudCom^ do
    GetCmdRsp(GETSQIcmd,sizeof(D.SeqIndexRec.Index),@D.SeqIndexRec.Index);
end;

procedure GetSeqEnableRsp; far;  {get seq enable flags}
begin
  with AudCom^ do
    GetCmdRsp(GETSQEcmd,sizeof(D.SeqEnRec.Flags),@D.SeqEnRec.Flags);
end;

procedure GetSeqControlRsp; far;  {get seq control regs}
begin
  with AudCom^ do
    GetCmdRsp(GETSQCcmd,sizeof(D.SeqCtrlRec),@D.SeqCtrlRec);
end;

procedure GetSeqActivityRsp; far;  {get seq activity regs}
begin
  with AudCom^ do
    GetCmdRsp(GETSQAcmd,sizeof(D.SeqActRec),@D.SeqActRec);
end;


procedure GetWarbleDataRsp; far;     {get warble data}
begin
  with AudCom^ do
    GetCmdRsp(GETWBDcmd,sizeof(D.WarbleRec),@D.WarbleRec);
end;

procedure GetWPcontrolRsp; far;     {get warble/pulse control regs}
begin
  with AudCom^ do
    GetCmdRsp(GETWPCcmd,sizeof(D.WPCrec),@D.WPCrec);
end;

procedure GetPortIORsp; far;     {get IO port regs}
begin
  with AudCom^ do
    GetCmdRsp(GETPIOcmd,sizeof(D.PortIOrec),@D.PortIOrec);
end;

procedure GetFpnImgRsp; far;
begin
  with AudCom^ do
    GetCmdRsp(GETFPNcmd,sizeof(D.FpnRec),@D.FpnRec);
end;

procedure GetPollDelayTimeRsp; far;
begin
  with AudCom^ do
    GetCmdRsp(GETpdlycmd,sizeof(D.PollDelayTime),@D.PollDelayTime);
end;


{*******************************************************************}
{                      public methods                               }
{*******************************************************************}

procedure AudComObj.Cmd(What:word);
begin
  D.SPI[What];
end;


function AudComObj.Fail:boolean;
begin
  Fail := D.CmdFailFlag;
  D.CmdFailFlag := false;
end;

procedure AudComObj.SetForeground(How:boolean);
begin
  D.Foreground := How;
end;


{cmd processing test - check on current condition of the cmd in progress}
{used mainly for background processing}
function AudComObj.CmdCheck:boolean;
begin
   CmdCheck := true;
   if D.Active = 0 then Exit;

   if not(SendReady) then
   begin
     CmdCheck := false;
     Exit;
   end;

   if D.NoPoll or D.XfrErr then
   begin
     CmdFail(BadPoll);
     Exit;
   end;
   ReadRsp;
end;

{-----------------------------------------------------------}
{returns true if char wating}
function AudComObj.GetStat:boolean;
begin
  D.Regs.ah := $03;
  D.Regs.dx := D.comport;
  CallFryers(D.Regs);
  GetStat := D.Regs.AH and 1 <> 0;
end;

function AudComObj.GetChar:char;
begin
  D.Regs.ah := $02;
  D.Regs.dx := D.comport;
  CallFryers(D.Regs);
  GetChar := char(D.Regs.AL);
end;

{returns NZ if ready for a char}
function AudComObj.PutStat:boolean;
begin
  D.Regs.ah := $03;
  D.Regs.dx := D.comport;
  CallFryers(D.Regs);
  PutStat := D.Regs.AH and $20 <> 0;
end;

function AudComObj.PutChar(What:char):byte;
begin
  D.Regs.ah := $01;
  D.Regs.al := byte(What);
  D.Regs.dx := D.comport;
  CallFryers(D.Regs);
  PutChar := D.Regs.AH and $80;
end;


function AudComObj.SendStr(s:str255):boolean;
var i,Ret : word;
var StartTime : longint;
begin
  SendStr := false;
  for i := 1 to length(s) do
  begin
     StartTime := GetTick;
     while not(PutStat) do
     begin
       if (GetTick-StartTime) > (20*55)  then Exit;
      (* if KeyPressed then {nop}; *)
     end;
     if s[i] <> #0 then
       Ret := PutChar(s[i]);
     if s[i] < #32 then
       ClkWait(8);      {wait 1/2 sec after control chars}
  end;
  SendStr := true;
end;

{===================================================================}
{ This procedure enables the RS232 port1 for use with terminal software. }

procedure AudComObj.ASCIIinit;
begin
  D.Regs.ax := $ff00; {disable fryers interrupt procedure}
  D.Regs.cx := $ff00; {this makes sure everything is kosher}
  D.Regs.dx := D.comport;
  CallFryers(D.Regs);

    D.Regs.ax := $ff00;
    D.Regs.cx := $ffff;
    D.Regs.dx := D.comport; {enable fryers interrupt procedure}
    CallFryers(D.Regs);
    D.Regs.ax := $00e3; {init to 9600 baud, no parity, 8 data bits}
    D.Regs.dx := D.comport;
    CallFryers(D.Regs);
    D.Regs.ax := $ff08;
    D.Regs.cx := $ff00;
    D.Regs.dx := D.comport; {disable CTS handshake}
    CallFryers(D.Regs);
    D.Regs.ax := $ff05;  {make sure dtr is on}
    D.Regs.cx := $ffff;
    D.Regs.dx := D.comport;
    CallFryers(D.Regs);
    D.Regs.ax := $ff04;  {make sure rts is on}
    D.Regs.cx := $ffff;
    D.Regs.dx := D.comport;
    CallFryers(D.Regs);
end;

{this returns the system to packet control}
function AudComObj.PacketInit:boolean;
begin
   PacketInit := false;
   UpdateFryersInfo;
   if not SendStr(#13'FIPP ON'#13) then Exit;
   D.Regs.ax := $ff10; {enable fryers packet protocol}
   D.Regs.cx := $ffff;
   D.Regs.dx := D.comport;
   CallFryers(D.Regs);
   PacketInit := true;
end;

procedure AudComObj.UpdateFryersInfo;
var RegAX,RegBX,RegDX,RegDI:word;
    P :pointer;
    S : str255;
begin
   D.Regs.di := 0;
   D.Regs.ax := $ffff; {test for fryers existance}
   D.Regs.dx := D.comport;
   CallFryers(D.Regs);
   RegAX := D.Regs.AX;
   RegDX := D.Regs.DX;
   RegDI := D.Regs.DI;

   if RegDI > 0 then
     D.TrueBaudRate := 115200 div RegDI
   else
     D.TrueBaudRate := 9600;

 {$IFDEF MSDOS}
   if lo(RegAX) < $27 then
   begin
     D.Fversion := ( (((RegAX and $f0) div 16) * 10) + (RegAX and $f) ) * 10;
   end
   else
   begin
     P := @S;
     asm
       mov bx,0
       mov si,word ptr [P+2]
       mov di,word ptr [P]
       mov ax,$fffe
       mov dx,0
       int $14
       mov [RegBX],bx
     end;
     D.Fversion := RegBX;
   end;
  {$ELSE}
   if lo(RegAX) < $30 then
   begin
     D.Fversion := ( (((RegAX and $f0) div 16) * 10) + (RegAX and $f) ) * 10;
   end
   else
   begin
     D.Regs.bx := 0;
     D.Regs.si := 0;
     D.Regs.di := 0;
     D.Regs.ax := $fffe;
     D.Regs.dx := 0;
     CallFryers(D.Regs);
     RegBX := D.Regs.bx;
     D.Fversion := RegBX;
   end;
 {$ENDIF}
end;


function AudComObj.InitRS232(IOport,IRQn:word):Boolean;
var TempI : byte;
var TempC,TempP : word;
begin
   InitRS232 := false;
   D.Regs.ax := $ffff; {test for fryers existance}
   D.Regs.dx := 0;
   CallFryers(D.Regs);

  {$IFDEF WIN32}
    if IOport > 15 then IOport := 1;
    if IOport < 1 then IOport := 1;
    D.comport := pred(IOport);
  {$ELSE}
    if IOport > 4 then IOport := 1;
    if IOport < 1 then IOport := 1;
    D.comport := pred(IOport) and 1;
  {$ENDIF}

   if (lo(D.Regs.AX) < $24) or (D.Regs.DX <> $FFFF) then Exit;
  {$IFDEF PMODE}
    if (lo(D.Regs.AX) < $40)  then Exit;
  {$ENDIF}
  {$IFDEF WIN32}
    if (lo(D.Regs.AX) < $50) then Exit;
  {$ENDIF}

   D.IRQnum[D.comport] := 0;
  {$IFNDEF WIN32}
    TempI := IRQn;
    TempP := IOport;
    TempC := D.comport;
    asm
      mov ax,$ff0a     {select the desired port to use}
      mov ch,$ff
      mov cl,[TempI]      {use the selected IRQ number}
      mov bx,[TempP]
      shr bx,1         {select the uart to use}
      and bx,1
      mov dx,[TempC]  {init the port}
      and dx,1
      INT $14
      mov [TempI],ah
    end;
    D.IRQnum[D.comport] := TempI;
  {$ENDIF}
   UpdateFryersInfo;
   AsciiInit;

   if not SendStr(#13'FIPP ON'#13) then Exit;
   D.Regs.ax := $ff10; {enable fryers packet protocol}
   D.Regs.cx := $ffff;
   D.Regs.dx := D.comport;
   CallFryers(D.Regs);

   D.PortOpen := true;
   InitRS232 := true;
end;


procedure AudComObj.CloseRS232;
begin
   D.Regs.ax := $ff00; {disable fryers interrupt procedure}
   D.Regs.cx := $ff00; {this makes sure everything is kosher}
   D.Regs.dx := D.comport;
   CallFryers(D.Regs);
   D.PortOpen := false;
end;

function AudComObj.EnableTerminalMode:boolean;
begin
   if not(D.PortOpen) then
   begin
     EnableTerminalMode := false;
     Exit;
   end;
   EnableTerminalMode := true;
   AsciiInit;
end;


    procedure GetEcho;
    var c : char;
    const i : byte = 0;
    begin
      c := #0;
      if AudCom^.GetStat then
        c := AudCom^.GetChar;
      if (c = #10) or (c = #13) then write(c);
      if (c > #$1f) and (c < #$7f) then write(c);
      if c = #$ff then
      begin
         write('#'); {probably a packet}
         inc(i);
         if i > 64 then
         begin
           i := 0;
           write(#13);
         end;
      end;
    end;


procedure AudComObj.Terminal;
var oc:char;
    ox : byte;
{const F10 = #196;
      EndKey = #207;}
const ExitKey = ^Z;
      CtrlXKey = ^X;
begin
  oc := #0;
  while true do
  begin
    GetEcho;

    if keywaiting then
    begin
      oc := getkey;
      write(oc);
      if (oc = ExitKey) or (oc = CtrlXKey) then Exit;
      if (oc <> #255) and (oc <> #0) then
        ox := PutChar(oc);
      oc := #0;
    end;
  end;
end;



function AudComObj.Tsend(S:str255):boolean;
var i : word;
    r : byte;
    C : char;
begin
  TSend := false;
  for i := 1 to length(S) do
  begin
    C := S[i];
    R := PutChar(C);
    if R and $80 = $80 then Exit;
    GetEcho;
  end;
  TSend := true;
end;

procedure AudComObj.TWait(HowLong:word);
var StartTime,When : longint;
begin
   When := HowLong*55;
   StartTime := GetTick;
   repeat
      GetEcho;
   until (GetTick-StartTime) > When;
end;


function AudComObj.ExitASCIImode:boolean;
begin
   ExitASCIImode := false;
   AsciiInit;       {make sure we are in ascii op}
   if not TSend(#3) then Exit;              {clear any pending command}
   Twait(4);
   if not TSend('FIPP ON'+#13) then Exit;   {try to turn on fipp mode}
   Twait(4);
   if not Tsend(#13) then Exit;
   Twait(4);
   if not TSend('FIPP ON'+#13) then Exit;   {try three times}
   Twait(4);
   if not TSend('FIPP ON'+#13) then Exit;
   TWait(4);  {wait 1/4 sec for things to stablize}
   ExitASCIImode := true;
end;

function AudComObj.EnablePacketMode:boolean;
begin
   EnablePacketMode := false;
   if not PacketInit then Exit;   {init back to packet operation}
   ClkWait(30);

   Cmd(GetCmdStatus);  {check if working}
   if Fail then
   begin
     Cmd(GetCmdStatus);  {try again if failed}
     if Fail then
     begin
       Cmd(GetCmdStatus);  {try one more time}
       if Fail then Exit;  {then give up}
     end;
   end;

   EnablePacketMode := true;
end;



function AudComObj.FailMsg:str255;
begin
   case D.FailError of
      NoFail  : FailMsg := 'No Failure';
      BadRsp  : FailMsg := 'Bad Response';
      BadSend : FailMsg := 'Bad Send';
      BadXfr  : FailMsg := 'Bad Transfer';
      BadPoll : FailMsg := 'No Poll';
      BadLogic: FailMsg := 'Bad Logic';
      else      FailMsg := 'Unknown Error';
   end;
end;



{*******************************************************************}
{                      internal use methods                         }
{*******************************************************************}

constructor AudComObj.Init;
var i:word;
begin
  with D do
  begin
    fillchar(D,sizeof(D),0);  {clear the vars}
    ForeGround := true;
    NoPoll := false;
    XfrErr := false;
    CmdFailFlag := false;
    for i := 0 to MaxProc do  {fill array with dummy procs}
      SPI[i] := DummyProc;

{$IFDEF DEBUG}
    SPI[GotoMonitor] := KillMeProc;
{$ENDIF}
    SPI[DoReset] := DoResetProc;
    SPI[GetCmdStatus] := GetCmdStatusproc;
    SPI[GetPrlImg] := GetPrlImgProc;
    SPI[GetXpnImg] := GetXpnImgProc;
    SPI[SetXpnImg] := SetXpnImgProc;
    SPI[GetBufPanImg] := GetBufPanImgProc;
    SPI[SetBstEnable] := SetBstEnableProc;

    SPI[GetSysStatus] := GetSSTImgProc;
    SPI[GetKeyImg] := GetKeyImgProc;
    SPI[GetPLCount] := GetPLCountProc;
    SPI[GetBPCount] := GetBPCountProc;
    SPI[GetVersion] := GetVersionProc;
    SPI[GetCalData] := GetCalDataProc;
    SPI[SetCalData] := SetCalDataProc;
    SPI[GetCalFlags] := GetCalFlagsProc;
    SPI[SetCalFlags] := SetCalFlagsProc;
    SPI[DoReadEEROM] := ReadEEROMProc;
    SPI[DoWriteEEROM] := WriteEEROMProc;
    SPI[GetWarbleData] := GetWarbleDataProc;
    SPI[SetWarbleData] := SetWarbleDataProc;
    SPI[GetWPcontrol] := GetWPcontrolProc;
    SPI[SetWPcontrol] := SetWPcontrolProc;
    SPI[GetPortIORegs] := GetPortIOProc;
    SPI[SetAsciiMode] := SetAsciiModeProc;

    SPI[GetSeqData] := GetSeqDataProc;
    SPI[SetSeqData] := SetSeqDataProc;
    SPI[GetSeqIndex] := GetSeqIndexProc;
    SPI[SetSeqIndex] := SetSeqIndexProc;
    SPI[GetSeqEnable] := GetSeqEnableProc;
    SPI[SetSeqEnable] := SetSeqEnableProc;
    SPI[GetSeqControl] := GetSeqControlProc;
    SPI[GetSeqActivity] := GetSeqActivityProc;
    SPI[GetPollDelay] := GetPollDelayProc;
    SPI[SetPollDelay] := SetPollDelayProc;

    SPI[GetFpnImg] := GetFpnImgProc;
    SPI[DoBufPanFlush] := DoBufPanFlushProc;
    SPI[DoPrlFlush] := DoPrlFlushProc;

    for i := 0 to MaxProc do  {fill array with dummy procs}
      RPI[i] := DummyProc;
    RPI[DoReset] := AckRsp;
    RPI[GetCmdStatus] := GetCmdStatusRsp;
    RPI[GetPrlImg] := GetPrlImgRsp;
    RPI[GetXpnImg] := GetXpnImgRsp;
    RPI[SetXpnImg] := AckRsp;
    RPI[GetFpnImg] := GetFpnImgRsp;

    RPI[GetBufPanImg] := GetBufPanImgRsp;
    RPI[SetBstEnable] := AckRsp;

    RPI[GetKeyImg] := GetKeyImgRsp;
    RPI[GetSysStatus] := GetSSTImgRsp;
    RPI[GetPLCount] := GetPLCountRsp;
    RPI[GetBPCount] := GetBPCountRsp;
    RPI[GetVersion] := GetVersionRsp;
    RPI[GetCalData] := GetCalDataRsp;
    RPI[SetCalData] := AckRsp;
    RPI[GetCalFlags] := GetCalFlagsRsp;
    RPI[SetCalFlags] := AckRsp;
    RPI[DoReadEEROM] := AckRsp;
    RPI[DoWriteEEROM] := AckRsp;
    RPI[GetWarbleData] := GetWarbleDataRsp;
    RPI[SetWarbleData] := AckRsp;
    RPI[GetWPcontrol] := GetWPcontrolRsp;
    RPI[SetWPcontrol] := AckRsp;
    RPI[GetPortIORegs] := GetPortIORsp;
    RPI[SetAsciiMode] := AckRsp;

    RPI[GetSeqData] := GetSeqDataRsp;
    RPI[SetSeqData] := AckRsp;
    RPI[GetSeqIndex] := GetSeqIndexRsp;
    RPI[SetSeqIndex] := AckRsp;
    RPI[GetSeqEnable] := GetSeqEnableRsp;
    RPI[SetSeqEnable] := AckRsp;
    RPI[GetSeqControl] := GetSeqControlRsp;
    RPI[GetSeqActivity] := GetSeqActivityRsp;
    RPI[SetPollDelay] := AckRsp;
    RPI[GetPollDelay] := GetPollDelayTimeRsp;

    RPI[DoBufPanFlush] := AckRsp;
    RPI[DoPrlFlush] := AckRsp;
  end;
end;

destructor AudComObj.Done;
begin
end;




{---------}
var i:word;
begin
  New(AudCom,Init);
  with AudCom^ do
  begin
    D.comport := 0;
    fillchar(D.IRQnum,sizeof(D.IRQnum),0);
  end;


{  OldRSPtr := nil;
  SaveProc := ExitProc;
  ExitProc := @RSExit; }
end.




