
{*******************************************************}
{                                                       }
{       Windows DOS CRT Simulator Unit                  }
{       Originally based on Borland's WINCRT unit       }
{       Massively rewritten by Michael Day              }
{       This release V1.30 as of 11-09-92               }
{       This program unit is public domain              }
{                                                       }
{       Note: Resource for program icon is assumed to   }
{       be located in main program file.                }
{       If a program resource icon is not supplied,     }
{       then the default windows icon will be used.     }
{                                                       }
{*******************************************************}

{$S-,R-}
unit DosCrt;

interface

uses WinTypes, WinProcs, WinDos;

const
  WindowOrg: TPoint =                       { program window origin }
    (X: cw_UseDefault; Y: cw_UseDefault);
  WindowSize: TPoint =                      { program window size }
    (X: cw_UseDefault; Y: cw_UseDefault);
  ScreenSize: TPoint = (X: 80; Y: 25);      { Screen buffer dimensions }
  Cursor: TPoint = (X: 0; Y: 0);            { Cursor location }
  Origin: TPoint = (X: 0; Y: 0);            { Client area origin }
  InactiveTitle: PChar = '(Inactive %s)';   { Inactive window title }
  AutoTracking: Boolean = True;             { Track cursor on Write? }
  CheckEOF: Boolean = False;                { Allow Ctrl-Z for EOF? }
  CheckBreak: Boolean = True;               { Allow Ctrl-C for break? }
  DosCrtWindow: HWnd = 0;                   { dos crt window handle }
  DosCrtFont: hFont = OEM_FIXED_FONT;       { dos crt font type }
  AllowPrintScreen : boolean = true;        { true = allow the screen buf print with ^P }
  LineDrawFonts : boolean = true;           { true = line draw chars available }
  TextWidth : word = 8;                     { width of fixed size text }
  TextHeight : word = 8;                    { height of fixed size text }
  TextAttr : byte = $f0;                    { text colors }

   RealMouseX : integer = 0;                   { current mouse position in client window}
   RealMouseY : integer = 0;
   ClickMouseX : integer = 0;                  { clicked mouse position in client window}
   ClickMouseY : integer = 0;
   MouseButton : word = 0;                  { dos mouse button simulation storage }
   MouseClickbutton : word = 0;             { where the sim dos mouse was when clicked }
   MouseClicked : boolean = false;          { was the sim dos mouse clicked? }
   LeftButton = 1;                          { dos mouse button definitions }
   RightButton = 2;
   MiddleButton = 4;

const GraphXY: TPoint = (X: 0; Y:0);           { graphics mode X/Y point }

type LineSettingsType = record
       LineStyle:word;
       Pattern  :word;
       Thickness:word;
     end;

type FillPatternType = array[1..8] of byte;

type string8 = string[8];


var
  dcwDC: HDC;                               { Global device context }
  WindowTitle: array[0..79] of Char;        { program window title }
  IconHandle :hicon;

Const GraphMaxColor = 15;
Const DosColor : array[0..GraphMaxColor] of longint = (
         0,$7f0000,$7f00,$7f7f00,$7f,$7f007f,$7f7f,$7f7f7f,
         $3f3f3f,$ff0000,$ff00,$ffff00,$ff,$ff00ff,$ffff,$ffffff);

Const Black = 0;
      Blue = 1;
      Green = 2;
      Cyan = 3;
      Red = 4;
      Magenta = 5;
      Brown = 6;
      LightGray = 7;
      DarkGray = 8;
      LightBlue = 9;
      LightGreen = 10;
      LightCyan = 11;
      LightRed = 12;
      LightMagenta = 13;
      Yellow = 14;
      White = 15;

const SetOn = true;
      SetOff = false;

procedure InitDosCrtWindow;
procedure DoneDosCrtWindow;

procedure WriteBuf(Buffer: PChar; Count: Word);
procedure WriteChar(Ch: Char);

function KeyPressed: Boolean;
function ReadKey: Char;
function ReadBuf(Buffer: PChar; Count: Word): Word;

procedure GotoXY(X, Y: Integer);
function WhereX: Integer;
function WhereY: Integer;
procedure ClrScr;
procedure ClrEol;

procedure CursorTo(X, Y: Integer);
procedure ScrollTo(X, Y: Integer);
procedure TrackCursor;
procedure SelectCursor(CurOn:boolean);

procedure AssignCrt(var F: Text);
procedure InitDeviceContext;
procedure DoneDeviceContext;

procedure Delay(T:longint);
procedure PrintScreen;

procedure TextColor(c:byte);
procedure TextBackground(c:byte);

function MouseClick:boolean;
procedure InitMouse;
procedure HideMouse;
procedure ShowMouse;
procedure ReadMouse;
Procedure MouseGraphicCursor(Shape:word);
function GetMx(X:word):word;
function GetMy(Y:word):word;


procedure DetectGraph(gd,gm:integer);
procedure InitGraph(gd,gm:integer; path:string);
procedure GraphSetColor(C:byte);
procedure GraphSetBkColor(C:byte);
procedure GraphSetFillPattern(Pattern:FillPatternType; C:word);
procedure GraphSetFillStyle(Pattern:Word; C:word);
procedure GraphMoveRel(X,Y:integer);
procedure GraphMoveTo(X,Y:integer);
procedure GraphRectangle(x1,y1,x2,y2:integer);
procedure GraphLine(X1,Y1,X2,Y2:integer);
procedure GraphLineTo(X,Y:integer);
procedure GraphSetLineStyle(LineStyle,Pattern,Thickness:word);
procedure GraphGetLineSettings(var GraphLineInfo:LineSettingsType);
procedure GraphPutPixel(X,Y:integer; C:word);
procedure GraphOutText(S:string);
procedure GraphOutTextXY(X,Y:integer; S:string);
function GraphGetX:integer;
function GraphGetY:integer;
function GraphGetMaxX:integer;
function GraphGetMaxY:integer;
function GraphTextWidth(s:string):integer;
function GraphTextHeight(s:string8):integer;
function GraphGetBkColor:integer;
procedure GraphArc(X,Y:integer; StAngle,EndAngle,Radius:word);
procedure GraphCircle(X,Y:integer; Radius:word);
procedure GraphEllipse(X,Y:integer; StAngle,EndAngle,XRadius,YRadius:word);
procedure GraphPieSlice(X,Y:integer; StAngle,EndAngle,XRadius,YRadius:word);
{procedure GraphSector; }

function Whex(W:word):string8;
function Lhex(L:longint):string8;


implementation

{ MinMaxInfo array }

type
  PMinMaxInfo = ^TMinMaxInfo;
  TMinMaxInfo = array[0..4] of TPoint;

{ Scroll key definition record }

type
  TScrollKey = record
    Key: Byte;
    Ctrl: Boolean;
    SBar: Byte;
    Action: Byte;
  end;

{ Dos Crt window procedure }

function DosCrtWinProc(Window: HWnd; Message, WParam: Word;
  LParam: Longint): Longint; export; forward;

{ dos crt window class }
const
  DosCrtClass: TWndClass = (
    style: cs_HRedraw + cs_VRedraw;
    lpfnWndProc: @DosCrtWinProc;
    cbClsExtra: 0;
    cbWndExtra: 0;
    hInstance: 0;
    hIcon: 0;
    hCursor: 0;
    hbrBackground: 0;
    lpszMenuName: nil;
    lpszClassName: 'TPPgmWin');


const
  FirstLine: Integer = 0;               { First line in circular buffer }
  KeyCount: Integer = 0;                { Count of keys in KeyBuffer }
  Created: Boolean = False;       	{ program window created? }
  Focused: Boolean = False;             { program window focused? }
  Reading: Boolean = False;             { Reading from program window? }
  CaretState: Boolean = False;          { Last known caret state on/off}
  CaretActive: Boolean = False;         { Caret currently active? }
  Painting: Boolean = False;            { Handling wm_Paint? }

  dcwPen : TlogPen = (                  { local pen style }
     lopnStyle : ps_Solid;
     lopnWidth : (x:1; y:1);
     lopnColor : $ffffff);

  dcwBrush : TlogBrush = (
     lbStyle : bs_Pattern;
     lbColor : $ffffff;
     lbHatch : bs_Solid );

  GraphColor : byte = white;            { local Dos graph drawing colors }
  GraphBkColor : byte = black;

const
   LineInfo : LineSettingsType =
    (LineStyle: 0;
     Pattern: 0;
     Thickness: 1);

const FillPattern : array[1..8] of word = (
           $ff, $ff, $ff, $ff, $ff, $ff, $ff, $ff);

var
  SaveExit: Pointer;                    { Saved exit procedure pointer }
  ScreenBuffer: PChar;                  { Screen buffer pointer }
  ScreenAttr: PChar;                    { Screen attributes pointer }
  ClientSize: TPoint;                   { Client area dimensions }
  Range: TPoint;                        { Scroll bar ranges }
  CharSize: TPoint;                     { Character cell size }
  CharAscent: Integer;                  { Character ascent }
  dcwPS: TPaintStruct;                  { Global paint structure }
  SaveFont: HFont;                      { Saved device context font }
  KeyBuffer: array[0..63] of Char;      { Keyboard type-ahead buffer }


{ Scroll keys table }

const
  ScrollKeyCount = 12;
  ScrollKeys: array[1..ScrollKeyCount] of TScrollKey = (
    (Key: vk_Left;  Ctrl: False; SBar: sb_Horz; Action: sb_LineUp),
    (Key: vk_Right; Ctrl: False; SBar: sb_Horz; Action: sb_LineDown),
    (Key: vk_Left;  Ctrl: True;  SBar: sb_Horz; Action: sb_PageUp),
    (Key: vk_Right; Ctrl: True;  SBar: sb_Horz; Action: sb_PageDown),
    (Key: vk_Home;  Ctrl: False; SBar: sb_Horz; Action: sb_Top),
    (Key: vk_End;   Ctrl: False; SBar: sb_Horz; Action: sb_Bottom),
    (Key: vk_Up;    Ctrl: False; SBar: sb_Vert; Action: sb_LineUp),
    (Key: vk_Down;  Ctrl: False; SBar: sb_Vert; Action: sb_LineDown),
    (Key: vk_Prior; Ctrl: False; SBar: sb_Vert; Action: sb_PageUp),
    (Key: vk_Next;  Ctrl: False; SBar: sb_Vert; Action: sb_PageDown),
    (Key: vk_Home;  Ctrl: True;  SBar: sb_Vert; Action: sb_Top),
    (Key: vk_End;   Ctrl: True;  SBar: sb_Vert; Action: sb_Bottom));


Const Hex : array[0..15] of char = '0123456789ABCDEF';

const DosLineStyle : array[0..6] of word =
         (ps_Solid,ps_Dot,ps_DashDot,ps_Dash,ps_DashDotDot,ps_Null,ps_InsideFrame);

const PIR = pi/180; {3.14159/180}   { Pi/180  to convert angle to radians}

const MaxFixedPattern = 12;
const FixedPattern : array[0..MaxFixedPattern] of FillPatternType = (
           ($ff, $ff, $ff, $ff, $ff, $ff, $ff, $ff),  {0}
           ($ff, $00, $ff, $00, $ff, $00, $ff, $00),  {1}
           ($ff, $ff, $ff, $ff, $ff, $ff, $ff, $ff),  {2}
           ($ff, $ff, $ff, $ff, $ff, $ff, $ff, $ff),  {3}
           ($ff, $ff, $ff, $ff, $ff, $ff, $ff, $ff),  {4}
           ($ff, $ff, $ff, $ff, $ff, $ff, $ff, $ff),  {5}
           ($ff, $ff, $ff, $ff, $ff, $ff, $ff, $ff),  {6}
           ($ff, $ff, $ff, $ff, $ff, $ff, $ff, $ff),  {7}
           ($ff, $ff, $ff, $ff, $ff, $ff, $ff, $ff),  {8}
           ($ff, $ff, $ff, $ff, $ff, $ff, $ff, $ff),  {9}
           ($ff, $ff, $ff, $ff, $ff, $ff, $ff, $ff),  {10}
           ($ff, $ff, $ff, $ff, $ff, $ff, $ff, $ff),  {11}
           ($ff, $ff, $ff, $ff, $ff, $ff, $ff, $ff)  {12}
          );



{--------------------------------------------------------------------}
{ Show caret }

procedure SelectCaret(CarOn:boolean);
begin
  if not(Focused) then Exit;
   
  If CarOn then
  begin
    if CaretActive then Exit;
    CaretActive := true;
    CreateCaret(DosCrtWindow, 0, CharSize.X, 2);
    SetCaretPos((Cursor.X - Origin.X) * CharSize.X,
      (Cursor.Y - Origin.Y) * CharSize.Y + CharAscent);
    ShowCaret(DosCrtWindow);
  end
  else
  begin
    if not(CaretActive) then Exit;
    CaretActive := false;
    DestroyCaret;
  end;
end;

{ Hide caret }
(*
procedure HideCursor;
begin
  DestroyCaret;
end;
*)
procedure SelectCursor(CurOn:boolean);
begin
    CaretState := CurOn;
    SelectCaret(CurOn);
end;

{ Allocate device context }

procedure InitDeviceContext;
begin
  SelectCaret(Off);
  if Painting then
    dcwDC := BeginPaint(DosCrtWindow, dcwPS)
  else
    dcwDC := GetDC(DosCrtWindow);
  SaveFont := SelectObject(dcwDC, GetStockObject(DosCrtFont));
end;

{ Release device context }

procedure DoneDeviceContext;
begin
  if CaretState = On then SelectCaret(On);
  DeleteObject(SelectObject(dcwDC, SaveFont));
  if Painting then
    EndPaint(DosCrtWindow, dcwPS) else
    ReleaseDC(DosCrtWindow, dcwDC);
end;


{--------------------------------------------------------------------}
{ Return the smaller of two integer values }

function Min(X, Y: Integer): Integer;
begin
  if X < Y then Min := X else Min := Y;
end;

{ Return the larger of two integer values }

function Max(X, Y: Integer): Integer;
begin
  if X > Y then Max := X else Max := Y;
end;

function Whex(W:word):string8;
var S:string8;
begin
  S[1] := hex[hi(W shr 4)];
  S[2] := hex[hi(W and $f)];
  S[3] := hex[lo(W shr 4)];
  S[4] := hex[lo(W and $f)];
  S[0] := #4;
  Whex := S;
end;

function Lhex(L:Longint):string8;
var S: string8;
begin
  S[1] := hex[hiword(hi(L shr 4))];
  S[2] := hex[hiword(hi(L and $f))];
  S[3] := hex[hiword(lo(L shr 4))];
  S[4] := hex[hiword(lo(L and $f))];

  S[5] := hex[loword(hi(L shr 4))];
  S[6] := hex[loword(hi(L and $f))];
  S[7] := hex[loword(lo(L shr 4))];
  S[8] := hex[loword(lo(L and $f))];

  S[0] := #8;
  Lhex := S;
end;

{delay for T milliseconds of time }

procedure Delay(T:longint);
var te : longint;
begin
   te := GetTickCount + T;

   if te < T then
   begin
     while GetTickCount > $ffff do
       keypressed;
     while (te < GetTickCount) do
       keypressed;
   end
   else

   begin
     while GetTickCount < te do
       keypressed;
   end;
end;

procedure TextColor(c:byte);
begin
  TextAttr := (TextAttr and $f0) or (c and $f);
end;

procedure TextBackground(c:byte);
begin
  TextAttr := (TextAttr and $8f) or ((c and $7) shl 4);
end;

procedure InitMouse;
begin
  {not needed under windows, just here for dos compatiblity}
end;

procedure HideMouse;
begin
  {not needed under windows, just here for dos compatiblity}
end;

procedure ShowMouse;
begin
  {not needed under windows, just here for dos compatiblity}
end;

procedure ReadMouse;
var Mxy : Tpoint;
begin
  GetCursorPos(Mxy);
  ScreenToClient(DosCrtWindow,Mxy);
  RealMouseX := Mxy.X;
  RealMouseY := Mxy.Y;
end;

function MouseClick:boolean;
begin
  MouseClick := MouseClicked;  {return value of mouseclicked var}
  MouseClicked := false;       {then clear it}
end;

Procedure MouseGraphicCursor(Shape:word);
begin
  SetCursor(LoadCursor(0, pchar(Shape))); 
end;

function GetMx(X:word):word;
begin
  if CharSize.X > 0 then
    GetMx := succ(X div CharSize.X)
  else
    GetMx := succ(X div 8);
end;

function GetMy(Y:word):word;
begin
  if CharSize.Y > 0 then
    GetMy := succ(Y div CharSize.Y)
  else            
    GetMy := succ(Y div 8);
end;

{----------------------------------------------------------------------------}
function GraphGetMaxX:integer;
begin
  GraphGetMaxX := ScreenSize.X * CharSize.X;
end;

function GraphGetMaxY:integer;
begin
  GraphGetMaxY := ScreenSize.Y * CharSize.Y;
end;

function GraphGetX:integer;
begin
  GraphGetX := GraphXY.X;
end;

function GraphGetY:integer;
begin
  GraphGetY := GraphXY.Y;
end;

function GraphTextWidth(s:string):integer;
begin
  GraphTextWidth := length(s)*CharSize.X;
end;

function GraphTextHeight(s:string8):integer;
begin
  GraphTextHeight := CharSize.Y
end;

function GraphGetMaxColor:integer;
begin
  GraphGetMaxColor := GraphMaxColor;
end;

function GraphGetBkColor:integer;
begin
  GraphGetBkColor := GraphBkColor;
end;

procedure GraphSetColor(C:byte);
begin
  dcwPen.lopnColor := DosColor[C];
  GraphColor := C;
end;

procedure GraphSetBkColor(C:byte);
begin
  GraphBkColor := C;
end;

procedure GraphSetFillStyle(Pattern:Word; C:word);
var i : word;
begin
  dcwBrush.lbColor := DosColor[C];
  if Pattern > MaxFixedPattern then Pattern := MaxFixedPattern;
  for i := 1 to 8 do
  begin
    FillPattern[i] := FixedPattern[Pattern][i];
  end;
end;

procedure GraphSetFillPattern(Pattern:FillPatternType; C:word);
var i : word;
begin
  dcwBrush.lbColor := DosColor[C];
  for i := 1 to 8 do
  begin
    FillPattern[i] := Pattern[i];
  end;
end;

procedure GraphRectangle(x1,y1,x2,y2:integer);
var OldPen:hPen;
begin
   InitDeviceContext;
   OldPen := SelectObject(dcwDC,CreatePenIndirect(dcwPen));
   WinProcs.moveto(dcwDC,x1,y1);
   WinProcs.lineto(dcwDC,x2,y1);
   WinProcs.lineto(dcwDC,x2,y2);
   WinProcs.lineto(dcwDC,x1,y2);
   WinProcs.lineto(dcwDC,x1,y1);
   GraphXY.X := X1;
   GraphXY.Y := Y1;
   DeleteObject(SelectObject(dcwDC,OldPen));
   DoneDeviceContext;
end;

procedure GraphLine(X1,Y1,X2,Y2:integer);
var OldPen:hPen;
begin
   InitDeviceContext;
   OldPen := SelectObject(dcwDC,CreatePenIndirect(dcwPen));
   WinProcs.moveto(dcwDC,x1,y1);
   WinProcs.lineto(dcwDC,x2,y2);
   GraphXY.X := X2;
   GraphXY.Y := Y2;
   DeleteObject(SelectObject(dcwDC,OldPen));
   DoneDeviceContext;
end;

procedure GraphLineTo(X,Y:integer);
var OldPen:hPen;
begin
   InitDeviceContext;
   OldPen := SelectObject(dcwDC,CreatePenIndirect(dcwPen));
   WinProcs.moveto(dcwDC,GraphXY.X,GraphXY.Y);
   WinProcs.lineto(dcwDC,x,y);
   GraphXY.X := X;
   GraphXY.Y := Y;
   DeleteObject(SelectObject(dcwDC,OldPen));
   DoneDeviceContext;
end;

procedure GraphSetLineStyle(LineStyle,Pattern,Thickness:word);
begin
   dcwPen.lopnStyle := DosLineStyle[LineStyle];
   dcwPen.lopnWidth.X := Thickness;
   dcwPen.lopnWidth.Y := Thickness;
   LineInfo.LineStyle := LineStyle;
   LineInfo.Pattern := Pattern;
   LineInfo.Thickness := Thickness;
end;

procedure GraphGetLineSettings(var GraphLineInfo:LineSettingsType);
begin
  GraphLineInfo := LineInfo;
end;

procedure GraphMoveTo(X,Y:integer);
begin
   InitDeviceContext;
   WinProcs.moveto(dcwDC,X,Y);
   GraphXY.X := X;
   GraphXY.Y := Y;
   DoneDeviceContext;
end;

procedure GraphMoveRel(X,Y:integer);
begin
   InitDeviceContext;
   WinProcs.moveto(dcwDC,GraphXY.X+X,GraphXY.Y+Y);
   GraphXY.X := GraphXY.X+X;
   GraphXY.Y := GraphXY.Y+Y;
   DoneDeviceContext;
end;

procedure GraphPutPixel(X,Y:integer; C:word);
begin
   InitDeviceContext;
   WinProcs.SetPixel(dcwDC,X,Y,DosColor[C]);
   DoneDeviceContext;
end;


procedure GraphOutText(S:string);
var OldBkColor : TcolorRef;
    OldTextColor : TcolorRef;
begin
    InitDeviceContext;
    OldBkColor := WinProcs.SetBkColor(dcwDC,DosColor[GraphBkColor]);
    OldTextColor := WinProcs.SetTextColor(dcwDC,DosColor[GraphColor]);
    TextOut(dcwDC, GraphXY.X, GraphXY.Y, @S[1], ord(S[0]));
    WinProcs.SetBkColor(dcwDC,OldBkColor);
    WinProcs.SetTextColor(dcwDC,OldTextColor);
    DoneDeviceContext;
end;

procedure GraphOutTextXY(X,Y:integer; S:string);
var OldBkColor : TcolorRef;
    OldTextColor : TcolorRef;
begin
    InitDeviceContext;
    OldBkColor := WinProcs.SetBkColor(dcwDC,DosColor[GraphBkColor]);
    OldTextColor := WinProcs.SetTextColor(dcwDC,DosColor[GraphColor]);
    TextOut(dcwDC, X, Y, @S[1], ord(S[0]));
    WinProcs.SetBkColor(dcwDC,OldBkColor);
    WinProcs.SetTextColor(dcwDC,OldTextColor);
    DoneDeviceContext;
end;

procedure GraphCircle(X,Y:integer; Radius:word);
var OldPen : hPen;
begin
  InitDeviceContext;
  OldPen := SelectObject(dcwDC,CreatePenIndirect(dcwPen));
  WinProcs.Ellipse(dcwDC,X-Radius,Y-Radius,X+Radius,Y+Radius);
  GraphXY.X := X;
  GraphXY.Y := Y;
  DeleteObject(SelectObject(dcwDC,OldPen));
  DoneDeviceContext;
end;

procedure GraphArc(X,Y:integer; StAngle,EndAngle,Radius:word);
var OldPen : hPen;
begin
  InitDeviceContext;
  OldPen := SelectObject(dcwDC,CreatePenIndirect(dcwPen));
  WinProcs.Arc(dcwDC,X-Radius,Y-Radius,X+Radius,Y+Radius,
                X+round(Radius*cos(PIR*StAngle)),Y-round(Radius*sin(PIR*StAngle)),
                X+round(Radius*cos(PIR*EndAngle)),Y-round(Radius*sin(PIR*EndAngle)));
  GraphXY.X := X;
  GraphXY.Y := Y;
  DeleteObject(SelectObject(dcwDC,OldPen));
  DoneDeviceContext;
(*
  gotoxy(1,1);
  write('Bx:',trunc(Radius*cos(PIR*StAngle)),' By:',trunc(Radius*sin(PIR*StAngle)),
        ' Ex:',trunc(Radius*cos(PIR*EndAngle)),' Ey:',trunc(Radius*sin(PIR*EndAngle)));

  write(' X:',X,' Y:',Y,' B:',StAngle,' E:',EndAngle,' Radius:',Radius,' piR:',PIR:4:4);

  gotoxy(1,2);
  write(' piR*St:',PIR*StAngle:4:4,' PiR*End:',PIR*EndAngle:4:4);
  gotoxy(1,3);
  write(' cos(PIR*StAngle):',cos(PIR*StAngle):4:4,
        ' sin(PIR*StAngle):',sin(PIR*StAngle):4:4 );
  gotoxy(1,4);
  write(' cos(PIR*EndAngle):',cos(PIR*EndAngle):4:4,
        ' sin(PIR*EndAngle):',sin(PIR*EndAngle):4:4 );

  GOTOXY(1,5);
  WRITE(X-Radius,' ',Y-Radius,' ',X+Radius,' ',Y+Radius,' ',
        X+round(Radius*cos(PIR*StAngle)),' ',Y-round(Radius*sin(PIR*StAngle)),' ',
        X+round(Radius*cos(PIR*EndAngle)),' ',Y-round(Radius*sin(PIR*EndAngle)));
*)
end;

procedure GraphEllipse(X,Y:integer; StAngle,EndAngle,XRadius,YRadius:word);
var OldPen : hPen;
begin
  InitDeviceContext;
  OldPen := SelectObject(dcwDC,CreatePenIndirect(dcwPen));
  WinProcs.Arc(dcwDC,X-XRadius,Y-YRadius,X+XRadius,Y+YRadius,
                X+round(XRadius*cos(PIR*StAngle)),Y-round(YRadius*sin(PIR*StAngle)),
                X+round(XRadius*cos(PIR*EndAngle)),Y-round(YRadius*sin(PIR*EndAngle)));
  GraphXY.X := X;
  GraphXY.Y := Y;
  DeleteObject(SelectObject(dcwDC,OldPen));
  DoneDeviceContext;
end;

procedure GraphPieSlice(X,Y:integer; StAngle,EndAngle,XRadius,YRadius:word);
var OldPen : hPen;
    OldBrush : hBrush;
begin
  InitDeviceContext;
  OldPen := SelectObject(dcwDC,CreatePenIndirect(dcwPen));
  dcwBrush.lbHatch := CreateBitMap(8,8,1,1,@FillPattern);
  OldBrush := SelectObject(dcwDC,CreateBrushIndirect(dcwBrush));
  WinProcs.Pie(dcwDC,X-XRadius,Y-YRadius,X+XRadius,Y+YRadius,
                X+round(XRadius*cos(PIR*StAngle)),Y-round(YRadius*sin(PIR*StAngle)),
                X+round(XRadius*cos(PIR*EndAngle)),Y-round(YRadius*sin(PIR*EndAngle)));
  GraphXY.X := X;
  GraphXY.Y := Y;
  DeleteObject(SelectObject(dcwDC,OldPen));
  DeleteObject(SelectObject(dcwDC,OldBrush));
  DeleteObject(dcwBrush.lbHatch);
  DoneDeviceContext;
end;

{ this is a emulation of the BGI init graph - it is here for compatibility purposes }
procedure InitGraph(gd,gm:integer; path:string);
begin
end;

procedure DetectGraph(gd,gm:integer);
begin
end;


procedure CloseGraph;
begin
end;



{------------------------------------------------------------------------------------}
{ Update scroll bars }

procedure SetScrollBars;
begin
  SetScrollRange(DosCrtWindow, sb_Horz, 0, Max(1, Range.X), False);
  SetScrollPos(DosCrtWindow, sb_Horz, Origin.X, True);
  SetScrollRange(DosCrtWindow, sb_Vert, 0, Max(1, Range.Y), False);
  SetScrollPos(DosCrtWindow, sb_Vert, Origin.Y, True);
end;

{ Terminate program window }

procedure Terminate;
begin
  SelectCaret(Off);
  Halt(255);
end;

{ Set cursor position }

procedure CursorTo(X, Y: Integer);
begin
  Cursor.X := Max(0, Min(X, ScreenSize.X - 1));
  Cursor.Y := Max(0, Min(Y, ScreenSize.Y - 1));
end;

{ Scroll window to given origin }

procedure ScrollTo(X, Y: Integer);
begin
  if Created then
  begin
    X := Max(0, Min(X, Range.X));
    Y := Max(0, Min(Y, Range.Y));
    if (X <> Origin.X) or (Y <> Origin.Y) then
    begin
      if X <> Origin.X then SetScrollPos(DosCrtWindow, sb_Horz, X, True);
      if Y <> Origin.Y then SetScrollPos(DosCrtWindow, sb_Vert, Y, True);
      ScrollWindow(DosCrtWindow,
	(Origin.X - X) * CharSize.X,
	(Origin.Y - Y) * CharSize.Y, nil, nil);
      Origin.X := X;
      Origin.Y := Y;
      UpdateWindow(DosCrtWindow);
    end;
  end;
end;

{ Scroll to make cursor visible }

procedure TrackCursor;
begin
  ScrollTo(Max(Cursor.X - ClientSize.X + 1, Min(Origin.X, Cursor.X)),
    Max(Cursor.Y - ClientSize.Y + 1, Min(Origin.Y, Cursor.Y)));
end;

{ Return pointer to location in screen buffer }

function ScreenPtr(X, Y: Integer): PChar;
begin
  Inc(Y, FirstLine);
  if Y >= ScreenSize.Y then Dec(Y, ScreenSize.Y);
  ScreenPtr := @ScreenBuffer[Y * ScreenSize.X + X];
end;

{ Return pointer to attribute location in screen attribute buffer }
function ScreenAttrPtr(X, Y: Integer): PChar;
begin
  Inc(Y, FirstLine);
  if Y >= ScreenSize.Y then Dec(Y, ScreenSize.Y);
  ScreenAttrPtr := @ScreenAttr[Y * ScreenSize.X + X];
end;

{ Update text on cursor line }

procedure ShowText(L, R: Integer);
var OldBkColor : TcolorRef;
    OldTextColor : TcolorRef;
begin
  if L < R then
  begin
    InitDeviceContext;
    OldBkColor := SetBkColor(dcwDC,DosColor[TextAttr shr 4]);
    OldTextColor := SetTextColor(dcwDC,DosColor[TextAttr and $f]);
    TextOut(dcwDC, (L - Origin.X) * CharSize.X,
      (Cursor.Y - Origin.Y) * CharSize.Y,
      ScreenPtr(L, Cursor.Y), R - L);
    SetBkColor(dcwDC,OldBkColor);
    SetTextColor(dcwDC,OldTextColor);
    DoneDeviceContext;
  end;
end;

{ Write text buffer to program window }

procedure WriteBuf(Buffer: PChar; Count: Word);
var
  L, R: Integer;

procedure NewLine;
begin
  ShowText(L, R);
  L := 0;
  R := 0;
  Cursor.X := 0;
  Inc(Cursor.Y);
  if Cursor.Y = ScreenSize.Y then
  begin
    Dec(Cursor.Y);
    Inc(FirstLine);
    if FirstLine = ScreenSize.Y then FirstLine := 0;
    FillChar(ScreenPtr(0, Cursor.Y)^, ScreenSize.X, ' ');
    FillChar(ScreenAttrPtr(0, Cursor.Y)^, ScreenSize.X, TextAttr);
    ScrollWindow(DosCrtWindow, 0, -CharSize.Y, nil, nil);
    UpdateWindow(DosCrtWindow);
  end;
end;

begin
  InitDosCrtWindow;
  L := Cursor.X;
  R := Cursor.X;
  while Count > 0 do
  begin
    case Buffer^ of
      #32..#255:
	begin
	  ScreenPtr(Cursor.X, Cursor.Y)^ := Buffer^;
	  ScreenAttrPtr(Cursor.X, Cursor.Y)^ := char(TextAttr);
	  Inc(Cursor.X);
	  if Cursor.X > R then R := Cursor.X;
	  if Cursor.X = ScreenSize.X then NewLine;
	end;
      #13:
	NewLine;
      #8:
	if Cursor.X > 0 then
	begin
	  Dec(Cursor.X);
	  ScreenPtr(Cursor.X, Cursor.Y)^ := ' ';
	  ScreenAttrPtr(Cursor.X, Cursor.Y)^ := char(TextAttr);
	  if Cursor.X < L then L := Cursor.X;
	end;
      #7:
        MessageBeep(0);
    end;
    Inc(Buffer);
    Dec(Count);
  end;
  ShowText(L, R);
  if AutoTracking then TrackCursor;
end;

{ Write character to program window }

procedure WriteChar(Ch: Char);
begin
  WriteBuf(@Ch, 1);
end;

{ Return keyboard status }

function KeyPressed: Boolean;
var
  M: TMsg;
begin
  InitDosCrtWindow;
  while PeekMessage(M, 0, 0, 0, pm_Remove) do
  begin
    if M.Message = wm_Quit then Terminate;
    TranslateMessage(M);
    DispatchMessage(M);
  end;
  KeyPressed := KeyCount > 0;
end;

{ Read key from program window }

function ReadKey: Char;
begin
  TrackCursor;
  if not KeyPressed then
  begin
    Reading := True;
{    if Focused and not(CursorActive) then ShowCursor; }
    repeat until KeyPressed;
{    if Focused and not(CursorActive) then HideCursor; }
    Reading := False;
  end;
  ReadKey := KeyBuffer[0];
  Dec(KeyCount);
  Move(KeyBuffer[1], KeyBuffer[0], KeyCount);
end;

{ Read text buffer from program window }

function ReadBuf(Buffer: PChar; Count: Word): Word;
var
  Ch: Char;
  I: Word;
begin
  I := 0;
  repeat
    Ch := ReadKey;
    case Ch of
      #8:
	if I > 0 then
	begin
	  Dec(I);
	  WriteChar(#8);
	end;
      #32..#255:
	if I < Count - 2 then
	begin
	  Buffer[I] := Ch;
	  Inc(I);
	  WriteChar(Ch);
	end;
    end;
  until (Ch = #13) or (CheckEOF and (Ch = #26));
  Buffer[I] := Ch;
  Inc(I);
  if Ch = #13 then
  begin
    Buffer[I] := #10;
    Inc(I);
    WriteChar(#13);
  end;
  TrackCursor;
  ReadBuf := I;
end;

{ Set cursor position }

procedure GotoXY(X, Y: Integer);
begin
  CursorTo(X - 1, Y - 1);
end;

{ Return cursor X position }

function WhereX: Integer;
begin
  WhereX := Cursor.X + 1;
end;

{ Return cursor Y position }

function WhereY: Integer;
begin
  WhereY := Cursor.Y + 1;
end;

{ Clear screen }

procedure ClrScr;
begin
  InitDosCrtWindow;
  FillChar(ScreenBuffer^, ScreenSize.X * ScreenSize.Y, ' ');
  FillChar(ScreenAttr^, ScreenSize.X * ScreenSize.Y, TextAttr);
  Longint(Cursor) := 0;
  Longint(Origin) := 0;
  SetScrollBars; 
  InvalidateRect(DosCrtWindow, nil, True);
  UpdateWindow(DosCrtWindow);
end;

{ Clear to end of line }

procedure ClrEol;
begin
  InitDosCrtWindow;
  FillChar(ScreenPtr(Cursor.X, Cursor.Y)^, ScreenSize.X - Cursor.X, ' ');
  FillChar(ScreenAttrPtr(Cursor.X, Cursor.Y)^, ScreenSize.X - Cursor.X, TextAttr);
  ShowText(Cursor.X, ScreenSize.X);
end;

{ send the full screen buffer to the printer }

procedure PrintScreen;
var Lst : Text;
    X,Y:word;
    S : string;
begin
  assign(Lst,'LPT1');
  rewrite(Lst);
  for y := 0 to pred(ScreenSize.Y) do
  begin
    for x := 0 to pred(ScreenSize.X) do
    begin
      S[succ(x)] := ScreenPtr(X,Y)^;
      S[0] := char(ScreenSize.X);
    end;
    while (S[0] > #0) and (ScreenPtr(pred(ord(S[0])),Y)^ = ' ') do
      dec(S[0]);
    writeln(Lst,s);
  end;
  close(Lst);
end;

{ wm_Create message handler }

procedure WindowCreate;
begin
  Created := True;
  GetMem(ScreenBuffer, ScreenSize.X * ScreenSize.Y);
  FillChar(ScreenBuffer^, ScreenSize.X * ScreenSize.Y, ' ');
  GetMem(ScreenAttr, ScreenSize.X * ScreenSize.Y);
  FillChar(ScreenAttr^, ScreenSize.X * ScreenSize.Y, TextAttr);
  if not CheckBreak then
    EnableMenuItem(GetSystemMenu(DosCrtWindow, False), sc_Close,
      mf_Disabled + mf_Grayed);
end;

{ wm_Paint message handler }

procedure WindowPaint;
var
  X1, X2, Y1, Y2, TX: Integer;
    OldBkColor : TcolorRef;
    OldTextColor : TcolorRef;
begin
  Painting := True;
  InitDeviceContext;
  X1 := Max(0, dcwPS.rcPaint.left div CharSize.X + Origin.X);
  X2 := Min(ScreenSize.X,
    (dcwPS.rcPaint.right + CharSize.X - 1) div CharSize.X + Origin.X);
  Y1 := Max(0, dcwPS.rcPaint.top div CharSize.Y + Origin.Y);
  Y2 := Min(ScreenSize.Y,
    (dcwPS.rcPaint.bottom + CharSize.Y - 1) div CharSize.Y + Origin.Y);
  while Y1 < Y2 do
  begin
    TX := X1;
    while TX < X2 do
    begin
      OldBkColor := SetBkColor(dcwDC,DosColor[ord(ScreenAttrPtr(TX, Y1)^) {TextAttr} shr 4]);
      OldTextColor := SetTextColor(dcwDC,DosColor[ord(ScreenAttrPtr(TX, Y1)^){TextAttr} and $f]);
      TextOut(dcwDC, (TX - Origin.X) * CharSize.X, (Y1 - Origin.Y) * CharSize.Y,
        ScreenPtr(TX, Y1), 1 {X2 - X1});
      SetBkColor(dcwDC,OldBkColor);
      SetTextColor(dcwDC,OldTextColor);
      inc(TX);
    end;
    Inc(Y1);
  end;
  DoneDeviceContext;
  Painting := False;
end;

{ wm_VScroll and wm_HScroll message handler }

procedure WindowScroll(Which, Action, Thumb: Integer);
var
  X, Y: Integer;

function GetNewPos(Pos, Page, Range: Integer): Integer;
begin
  case Action of
    sb_LineUp: GetNewPos := Pos - 1;
    sb_LineDown: GetNewPos := Pos + 1;
    sb_PageUp: GetNewPos := Pos - Page;
    sb_PageDown: GetNewPos := Pos + Page;
    sb_Top: GetNewPos := 0;
    sb_Bottom: GetNewPos := Range;
    sb_ThumbPosition: GetNewPos := Thumb;
  else
    GetNewPos := Pos;
  end;
end;

begin
  X := Origin.X;
  Y := Origin.Y;
  case Which of
    sb_Horz: X := GetNewPos(X, ClientSize.X div 2, Range.X);
    sb_Vert: Y := GetNewPos(Y, ClientSize.Y, Range.Y);
  end;
  ScrollTo(X, Y);
end;

{ wm_Size message handler }

procedure WindowResize(X, Y: Integer);
begin
  SelectCaret(Off);
  ClientSize.X := X div CharSize.X;
  ClientSize.Y := Y div CharSize.Y;
  Range.X := Max(0, ScreenSize.X - ClientSize.X);
  Range.Y := Max(0, ScreenSize.Y - ClientSize.Y);
  Origin.X := Min(Origin.X, Range.X);
  Origin.Y := Min(Origin.Y, Range.Y);
  SetScrollBars; 
  if CaretState = On then SelectCaret(On);
end;

{ wm_GetMinMaxInfo message handler }

procedure WindowMinMaxInfo(MinMaxInfo: PMinMaxInfo);
var
  X, Y: Integer;
  Metrics: TTextMetric;
begin
  InitDeviceContext;
  GetTextMetrics(dcwDC, Metrics);
  CharSize.X := Metrics.tmMaxCharWidth;
  CharSize.Y := Metrics.tmHeight + Metrics.tmExternalLeading;
  CharAscent := Metrics.tmAscent;
  X := Min(ScreenSize.X * CharSize.X + GetSystemMetrics(sm_CXVScroll),
    GetSystemMetrics(sm_CXScreen)) + GetSystemMetrics(sm_CXFrame) * 2;
  Y := Min(ScreenSize.Y * CharSize.Y + GetSystemMetrics(sm_CYHScroll) +
    GetSystemMetrics(sm_CYCaption), GetSystemMetrics(sm_CYScreen)) +
    GetSystemMetrics(sm_CYFrame) * 2;
  MinMaxInfo^[1].x := X;
  MinMaxInfo^[1].y := Y;
  MinMaxInfo^[3].x := CharSize.X * 16 + GetSystemMetrics(sm_CXVScroll) +
    GetSystemMetrics(sm_CXFrame) * 2;
  MinMaxInfo^[3].y := CharSize.Y * 4 + GetSystemMetrics(sm_CYHScroll) +
    GetSystemMetrics(sm_CYFrame) * 2 + GetSystemMetrics(sm_CYCaption);
  MinMaxInfo^[4].x := X;
  MinMaxInfo^[4].y := Y;
  TextWidth := CharSize.X;
  TextHeight := CharSize.Y;
  DoneDeviceContext;
end;

{ wm_Char message handler }

procedure WindowChar(Ch: Char);
begin
  if CheckBreak and (Ch = #3) then Terminate;
  if AllowPrintScreen and (Ch = ^P) then
    PrintScreen
  else
  begin
    if KeyCount < SizeOf(KeyBuffer) then
    begin
      KeyBuffer[KeyCount] := Ch;
      Inc(KeyCount);
    end;
  end;
end;

{ wm_KeyDown message handler }

procedure WindowKeyDown(KeyDown: Byte);
var
  CtrlDown: Boolean;
  I: Integer;
begin
  if CheckBreak and (KeyDown = vk_Cancel) then Terminate;
  if AllowPrintScreen and ((KeyDown = vk_SnapShot) or (KeyDown = vk_Print)) then
  begin
    PrintScreen;
    Exit;
  end;
  CtrlDown := GetKeyState(vk_Control) < 0;
  for I := 1 to ScrollKeyCount do
    with ScrollKeys[I] do
      if (Key = KeyDown) and (Ctrl = CtrlDown) then
      begin
	WindowScroll(SBar, Action, 0);
	Exit;
      end;
end;

{---------------------------------------------------------}
{ mouse button and position routines}

  procedure SetButtons(KeyState:word);  {set the mouse button flags}
  begin
    MouseButton := 0;
    if KeyState and mk_LButton <> 0 then
      MouseButton := MouseButton or LeftButton;
    if KeyState and mk_RButton <> 0 then
      MouseButton := MouseButton or RightButton;
    if KeyState and mk_MButton <> 0 then
      MouseButton := MouseButton or MiddleButton;
    MouseClickbutton :=  MouseButton;
  end;

procedure WindowLeftButtonDown(mX,mY:integer; KeyState:word);
begin
  ClickMouseX := mX;
  ClickMouseY := mY;
  RealMouseX := mX;
  RealMouseY := mY;
  SetButtons(KeyState);
  MouseClicked := true;
end;

procedure WindowRightButtonDown(mX,mY:integer; KeyState:word);
begin
  ClickMouseX := mX;
  ClickMouseY := mY;
  RealMouseX := mX;
  RealMouseY := mY;
  SetButtons(KeyState);
  MouseClicked := true;
end;

procedure WindowMiddleButtonDown(mX,mY:integer; KeyState:word);
begin
  ClickMouseX := mX;
  ClickMouseY := mY;
  RealMouseX := mX;
  RealMouseY := mY;
  SetButtons(KeyState);
  MouseClicked := true;
end;

{--------------------------------------------------------------}
{ wm_SetFocus message handler }

procedure WindowSetFocus;
begin
  Focused := True;
  if {Reading} CaretState = On then SelectCaret(On);
end;

{ wm_KillFocus message handler }

procedure WindowKillFocus;
begin
  SelectCaret(Off); {HideCursor;}
  Focused := False;
end;

{ wm_Destroy message handler }

procedure WindowDestroy;
begin
  FreeMem(ScreenBuffer, ScreenSize.X * ScreenSize.Y);
  FreeMem(ScreenAttr, ScreenSize.X * ScreenSize.Y);
  Longint(Cursor) := 0;
  Longint(Origin) := 0;
  PostQuitMessage(0);
  Created := False;
end;

{ DosCrt mainline window procedure }
function DosCrtWinProc(Window: HWnd; Message, WParam: Word;
  LParam: Longint): Longint;
begin
  DosCrtWinProc := 0;
  DosCrtWindow := Window;
  case Message of
    wm_Create: WindowCreate;
    wm_Paint: WindowPaint;
    wm_VScroll: WindowScroll(sb_Vert, WParam, loWord(LParam));
    wm_HScroll: WindowScroll(sb_Horz, WParam, loWord(LParam));
    wm_Size: WindowResize(loWord(LParam), hiWord(LParam));
    wm_GetMinMaxInfo: WindowMinMaxInfo(PMinMaxInfo(LParam));
    wm_Char: WindowChar(Char(WParam));
    wm_KeyDown: WindowKeyDown(Byte(WParam));
    wm_LButtonDown: WindowLeftButtonDown(loWord(LParam),hiword(LParam),WParam);
    wm_RButtonDown: WindowRightButtonDown(loWord(LParam),hiword(LParam),WParam);
    wm_MButtonDown: WindowMiddleButtonDown(loWord(LParam),hiword(LParam),WParam);
    wm_SetFocus: WindowSetFocus;
    wm_KillFocus: WindowKillFocus;
    wm_Destroy: WindowDestroy;
  else
    DosCrtWinProc := DefWindowProc(Window, Message, WParam, LParam);
  end;
end;

{ Text file device driver output via PgmWin }
function CrtOutput(var F: TTextRec): integer; far;
begin
  if F.BufPos <> 0 then
  begin
    Writebuf(Pchar(F.BufPtr), F.BufPos);
    F.BufPos := 0;
    KeyPressed;
  end;
  CrtOutput := 0;
end;

{ Text file device driver input via PgmWin }
function CrtInput(var F: TTextRec): integer; far;
begin
  F.BufEnd := ReadBuf(Pchar(F.BufPtr), F.BufSize);
  F.BufPos := 0;
  CrtInput := 0;
end;

{Text file device driver close function }
function CrtClose(var F: TTextRec): integer; far;
begin
  CrtClose := 0;
end;

{ Text file device driver open function }
function CrtOpen(var F: TTextRec): Integer; far;
begin
  if F.Mode = fmInput then
  begin
    F.InOutFunc := @CrtInput;
    F.FlushFunc := nil;
  end
  else
  begin
    F.Mode := fmOutput;
    F.InOutFunc := @CrtOutput;
    F.FlushFunc := @CrtOutput;
  end;
  F.CloseFunc := @CrtClose;
  CrtOpen := 0;
end;


{ Assign text file to PgmWin }
procedure AssignCrt(var F:text);
begin
  with TTextRec(F) do
  begin
    Handle := $ffff;
    Mode := fmClosed;
    BufSize := sizeof(Buffer);
    BufPtr := @Buffer;
    OpenFunc := @CrtOpen;
    Name[0] := #0;
  end;
end;

{ Create program window if required }
procedure InitDosCrtWindow;
begin
  if not(Created) then
  begin
    if DosCrtFont = OEM_FIXED_FONT then
      LineDrawFonts := true
    else
      LineDrawFonts := false;
    DosCrtWindow := CreateWindow(
      DosCrtClass.lpszClassName,
      WindowTitle,
      ws_OverlappedWindow + ws_HScroll + ws_VScroll,
      WindowOrg.X, WindowOrg.Y,
      WindowSize.X, WindowSize.Y,
      0,
      0,
      HInstance,
      nil);
    ShowWindow(DosCrtWindow, CmdShow);
    UpdateWindow(DosCrtWindow);
  end;
end;

{ Destroy program window if required }

procedure DoneDosCrtWindow;
begin
  if Created then DestroyWindow(DosCrtWindow);
 { Halt(0);}
end;

{ program window unit exit procedure }

procedure ExitDosCrtWin; far;
var
  P: PChar;
  Message: TMsg;
  Title: array[0..127] of Char;
begin
  CloseGraph;
  ExitProc := SaveExit;
  if Created and (ErrorAddr = nil) then
  begin
    P := WindowTitle;
    WVSPrintF(Title, InactiveTitle, P);
    SetWindowText(DosCrtWindow, Title);
    EnableMenuItem(GetSystemMenu(DosCrtWindow, False), sc_Close, mf_Enabled);
    CheckBreak := False;
    while GetMessage(Message, 0, 0, 0) do
    begin
      TranslateMessage(Message);
      DispatchMessage(Message);
    end;
  end;
end;

begin
  if HPrevInst = 0 then
  begin
    DosCrtClass.hInstance := HInstance;
  {  DosCrtClass.hIcon := LoadIcon(hInstance, idi_Application); }
    DosCrtClass.hIcon := LoadIcon(hInstance, 'PGM_ICON'); 
    DosCrtClass.hCursor := LoadCursor(0, idc_Arrow);
    DosCrtClass.hbrBackground := GetStockObject(White_Brush);
    RegisterClass(DosCrtClass);
  end;

  AssignCrt(Input);
  Reset(Input);
  AssignCrt(Output);
  ReWrite(Output);
  GetModuleFileName(HInstance, WindowTitle, SizeOf(WindowTitle));
  OemToAnsi(WindowTitle, WindowTitle);
  SaveExit := ExitProc;
  ExitProc := @ExitDosCrtWin;
end.

