
//FryeCom Error Message strings unit
//Copyright 2010 Frye Electronics, Inc.
//Written by Michael Day as of 9 Sept 2010
//You may use this code for any lawful purpose without fees or royalties.
//No warranties are express or implied in the use of this code.
//This will compile under any version of Borland Delphi (written using Delphi V3.00)
//----------------------------------------------------

unit FCErrMsg;

interface
uses FryeDefs, Fryers, FcomDefs,FryeComD, FryeStr;

//Errors returned to the operating system on program exit
//(if you return the error code).
//These are used for command line based console applications
//that use batch scripts.
const
     FCE_GoodPacket     = 0;
     FCE_ReceiveError   = 1;
     FCE_NakResponse    = 2;
     FCE_SendError      = 3;
     FCE_UnexpectedAck  = 4;
     FCE_IllegalPacket  = 5;
     FCE_BadPacket      = 6;
     FCE_NoResponse     = 7;
     FCE_CommandFail    = 8;
     FCE_NoFryers       = 9;
     FCE_NoPoll         = 10;
     FCE_BadConfig      = 11;
     FCE_UnknownError   = 12;
     FCE_NoPort         = 15;
     FCE_NoCmd          = 255; //no command given to send
const FCE_NoFileError = 200;
const FCE_FileReadError = 201;
const FCE_FileOverflowError = 202;

const FEM_NO_ERROR = 0;   //no error
const FEM_SOFT_ERROR = 1; //recoverable error
const FEM_HARD_ERROR = -1; //non-recoverable error

function FEM_GetErrorMsg(FippError:integer; PreError:boolean; var ErrorType:integer; var sMsg:string):integer;
//function FEM_GetErrorMsg(FippError:integer; var sMsg:string):integer;
function FEM_Msg(FippError:integer; var ErrorType:integer):string;

implementation


//---------------------------------------------------------------------------

//Returns the error message string for the indicated error number
//function result is the operating system return code (see note above).
//if PreError is true, adds "Error" to msg start if there is an error condition.
function FEM_GetErrorMsg(FippError:integer; PreError:boolean; var ErrorType:integer; var sMsg:string):integer;
var sName:string;
var sError:string;
begin
  if PreError = true then sError := 'Error ' else sError := '';
  case(FippError) of
    //FryeCom/Fryers Errors are -1000 series numbers
    FCOM_GENERIC_ERROR: // -1000   //unknown fryecom error - something bad happened
      begin
        sName := FS_Upper(FCOM_DLL_NAME);
        sMsg := sError + 'Unknown ' + sName + ' Access';
        ErrorType := FEM_HARD_ERROR;
        Result := FCE_NoFryers;
      end;
    FCOM_FILE_NOT_FOUND: // -1001  //device driver file not found
      begin
        sName := FS_Upper(FCOM_DLL_NAME);
        sMsg := sError + sName + ' Not Found';
        ErrorType := FEM_HARD_ERROR;
        Result := FCE_NoFryers;
      end;
    FCOM_ENTRY_NOT_FOUND: // -1002 //entry point to fryecom driver not found
      begin
        sName := FS_Upper(FCOM_DLL_NAME);
        sMsg := sError + sName + ' Procedure ';
        sName := FS_IntToStr(FCOM_ProcedureNumber);
        sMsg := sMsg + sName + ' Not Found';
        ErrorType := FEM_HARD_ERROR;
        Result := FCE_NoFryers;
      end;
    FCOM_BAD_DRIVER: // -1003      //unknown internal problem encountered in the Driver
      begin
        sName := FS_Upper(FCOM_DLL_NAME);
        sMsg := sError + sName + ' Invalid';
        Result := FCE_NoFryers;
      end;

    //common FCOM packet errors use the -9000 series
    FCOM_SUCCESS: // 0      //packet is ok (no errors)
      begin
        sMsg := 'OK';
        ErrorType := FEM_NO_ERROR;
        Result := FCE_GoodPacket;
      end;
    FCOM_PACKET_ERROR: // -9000      //generic unknown packet error
      begin
        sMsg := sError + 'Com Failure';
        Result := FCE_UnknownError;
      end;
    FCOM_PORT_RANGE_ERROR: // -9001  //not a valid comport selection (out of range)
      begin
        sMsg := sError + 'Invalid Port Selection';
        ErrorType := FEM_HARD_ERROR;
        Result := FCE_NoPort;
      end;
    FCOM_PORT_NOT_FOUND: // -9002    //Requested port not available
      begin
        sMsg := sError + 'Port Not Found';
        ErrorType := FEM_HARD_ERROR;
        Result := FCE_NoPort;
      end;
    FCOM_PORT_NOT_OPEN: // -9003     //Requested port not open for communication
      begin
        sMsg := sError + 'Port Not Open';
        ErrorType := FEM_HARD_ERROR;
        Result := FCE_NoPort;
      end;
    FCOM_AUTOBAUD_FAILED: // -9004   //could not find baudrate
      begin
        sMsg := sError + 'Could not find baudrate';
        ErrorType := FEM_HARD_ERROR;
        Result := FCE_NoPort;
      end;
    FCOM_NO_POLL: // -9010           //no polls from the instrument
      begin
        sMsg := sError + 'No Poll';
        ErrorType := FEM_SOFT_ERROR;
        Result := FCE_NoPoll;
      end;
    FCOM_SEND_OVERRUN: // -9011      //trying to send another command before old one done.
      begin
        sMsg := sError + 'Send Overrun';
        ErrorType := FEM_SOFT_ERROR;
        Result := FCE_SendError;
      end;
    FCOM_SEND_ERROR: // -9012        //error while trying to send the command
      begin
        sMsg := sError + 'Send Failure';
        ErrorType := FEM_SOFT_ERROR;
        Result := FCE_SendError;
      end;
    FCOM_NO_RESPONSE: // -9020       //no response received
      begin
        sMsg := sError + 'No Response';
        ErrorType := FEM_SOFT_ERROR;
        Result := FCE_ReceiveError;
      end;
    FCOM_RESPONSE_ERROR: // -9021    //error while receiveing the response
      begin
        sMsg := sError + 'Response Failure';
        ErrorType := FEM_SOFT_ERROR;
        Result := FCE_ReceiveError;
      end;
    FCOM_RESPONSE_OVERRUN: // -9022  //current response overwrote previous response
      begin
        sMsg := sError + 'Response Overrun';
        ErrorType := FEM_SOFT_ERROR;
        Result := FCE_ReceiveError;
      end;
    FCOM_NAK_RESPONSE: // -9023       //NAK rcvd as a respond
      begin
        sMsg := sError + 'NAK Response';
        ErrorType := FEM_SOFT_ERROR;
        Result := FCE_NakResponse;
      end;
    FCOM_ILL_RESPONSE: // -9024      //illegal command rcvd as a response
      begin
        sMsg := sError + 'ILL Response';
        ErrorType := FEM_SOFT_ERROR;
        Result := FCE_IllegalPacket;
      end;
    FCOM_POLL_RESPONSE: // -9025     //poll rcvd as a response
      begin
        sMsg := sError + 'Poll Response';
        ErrorType := FEM_SOFT_ERROR;
        Result := FCE_NoResponse;
      end;
    FCOM_CMD_FAIL: // -9026          //unexpected response to a command
      begin
        sMsg := sError + 'Command Failure';
        ErrorType := FEM_SOFT_ERROR;
        Result := FCE_CommandFail;
      end;
    FCOM_UNEXPECTED_ACK: // -9027    //got ACK response when wasn't supposed to
      begin
        sMsg := sError + 'Unexpected ACK';
        ErrorType := FEM_SOFT_ERROR;
        Result := FCE_UnexpectedAck;
      end;

    FCOM_INVALID_FORMAT: // -9999    //The passed format is invalid (missing pointer?)
      begin
        sMsg := sError + 'Invalid Command Format';
        ErrorType := FEM_SOFT_ERROR;
        Result := FCE_NoCmd;
      end;

    else
    begin
      sName := FS_IntToStr(FCOM_ProcedureNumber);
      sMsg := sError +  'Unknown (' + sName + ')';
      ErrorType := FEM_SOFT_ERROR;
      Result := FCE_UnknownError;
    end;
  end;//endswitch(Error)

end;


{-------------------------------------------------------------------}
//Calls message converter above, but just returns string,
//instead of exit error code. also inhibits "error" lead-in
function FEM_Msg(FippError:integer; var ErrorType:integer):string;
begin
  FEM_GetErrorMsg(FippError,false,ErrorType,Result);
end;

{-------------------------------------------------------------------}
{returns a string relating to the error encountered}
{old message format no longer supported - Use GetErrorMsg instead}
function fem_ErrorMsg(ErrorNumber:integer):str255;
var Emsg:string[20];
begin
    str(ErrorNumber,Emsg);
    case ErrorNumber of
      FCE_GoodPacket     : Result := '';
      FCE_ReceiveError   : Result := 'RECEIVE ERROR';
      FCE_NakResponse    : Result := 'RECEIVE OVERRUN';
      FCE_SendError      : Result := 'SEND ERROR';
      FCE_UnexpectedAck  : Result := 'UNEXPECTED ACK';
      FCE_IllegalPacket  : Result := 'ILLEGAL PACKET';
      FCE_BadPacket      : Result := 'BAD PACKET';
      FCE_NoResponse     : Result := 'NO RESPONSE';
      FCE_CommandFail    : Result := 'COMMAND ERROR';
      FCE_NoFryers       : Result := 'NO COMM DRIVER';
      FCE_NoPoll         : Result := 'NO POLL';
      FCE_BadConfig      : Result := 'BAD CMD FORMAT';
      FCE_NoPort         : Result := 'PORT NOT FOUND';
      FCE_NoCmd          : Result := 'NO CMD GIVEN';
    else
      Result := 'UNKNOWN ERROR: '+Emsg+' ';
    end;
end;



end.
