//---------------------------------------------------------------------------
//FippCore.CPP as of 15 Sep 2010 by Michael Day
//Frye Instrument Packet Protocol Core interface Copyright Frye Electronics 2010
//You may use this code for any lawful purpose without fees or royalties.
//This code is intended as an example showing how to communicate with Fonix
//equipment using the FryeCom.DLL device driver on the Windows operating system.
//No warranties are express or implied in the use of this code.
//---------------------------------------------------------------------------

//This is the Fipp Core interface unit that you can include in your own
//programs. This provides the minimal code interface to the FryeCom DLL
//to communicate with the Fonix Instruments.
//FippCore is object based so that you can include it as a part of your
//own object definition.
//
//You may also want to consider using the FippUnit file in your program as well.
//The FippUnit file contains the most common basic command calls used with
//the Fonix Instruments and provides a starting point for expanding
//the interface with your own Fipp commands. The FippUnit also shows how to
//include the FippCore object as a part of a higher level object.
//Normally you will not need to change the FippCore or FippUnit files,
//just include them as a part of your own object definition
//(which is the whole point for using C++...)

//---------------------------------------------------------------------------
//This software was written using the Borland C++ Builder compiler (ver 3.0)
//You should be able to recompile it using any other C++ compiler with
//little or no changes to the code.
//================================================================
//You will see many references to RS232 in the code and documentation.
//The Fryers communication protocol was originally developed for CP/M
//computers a long time ago. It has evolved over the years to the Windows
//version that you see here. The original protocol operated via an RS232
//Serial Com port and was based on a multi-cpu communications protocol.
//While much of the code has changed and morphed over the years,
//some of the names and structures have remained with only a few changes.
//The protocol is designed to be a simple transport mechanism that
//can function on minimal resource equipment. In addition, the
//protocol does not know what data is being transported.
//This allows the protocol to be stable and not require constant
//upgrades or changes as equipment and software changes over time.
//This has worked very well and has allowed migration of applications
//with little or no changes to the comunications interface over many years,
//many instruments and many operating systems.
//================================================================
//The original application references and drivers expected the programmer
//to manage the communication at a detailed level. While the low level
//Fryers32.dll driver (Fryers.COM for older MSDOS systems) managed the
//time critical aspects and communication handshaking between the instrument
//and the computer, it still required some bit-banging and knowledge of
//hardware communication to operate properly. Also over the years we've
//added newer functionality via some complex high-level calls such as
//Autobaud, OpenPort and FindPort which simply the high-level functionality.
//The FryeCom.DLL driver was designed to provide a more simplified programmer
//interface and to incorporate the messy Autobaud, OpenPort and FindPort calls.
//This makes the job a bit easier to communicate with Fonix instruments.
//This FIPP unit provides an example showing how to communicate with the
//FryeCom.dll driver. You can use this code as an example to write your
//own software, or you can modify this code for your own use, or
//use it as is if it works for you.
//Also see the Fonix programmers toolkit for other example application programs
//that comunicate with the Fonix equipment (most still use the older low-level
//driver Fryers calls, but the high level interface is still instructional
//on how to send specific instrument commands.

//================================================================

//Remember to call UnexpectedAck() after all Gets to check for unexpected
//ACK as a response. The 6500 is noted for inappropriate
//ACK responses to GET cmds.
//================================================================


#include "FryeDefs.h"
#include "FComDefs.h"
#include "FippCore.h"


//---------------------------------------------------------------------------------
//All of the calls in this unit can return FCOM driver errors (-1000 series) in
//addition to the FCOM packet errors (-9000 series) described in the routines below.
//See the FComDefs.h file for error code details.

//---------------------------------------------------------------------------------
INT16 fipp_QuickTerminateCmd[2] = { 0x7FFF, 0 };

//-------------------------------------------------------------------
// Implmentation for the FippCore/FryeCom class constructor
__fastcall TclsFippCore::TclsFippCore() {

  LookingForPort = false;    //not looking for new port
  PortInitialized = false;   //port has been initialized
  ThisComPort = 0;
  ActiveBaudrate = 0;        //current baudrate results (0=unknown)
  memset(&PortStatus,0,sizeof(PortStatus));
  PortStatus.SeekState = -1; //port not yet initialized
  FcomVersion = -1;
  FD_WhichDLL = FD_NO_DLL_LOADED;
};


//-------------------------------------------------------------------
//If the instrument cannot return an instrument id, returns false
//This command assumes that the Instrument Model information has been
//updated and is valid.
//-------------------------------------------------------------------
bool __fastcall TclsFippCore::PreIdInstrument(int InstrumentModel) {
  if ((InstrumentModel == FONIX_FP40)||
      (InstrumentModel == FONIX_6500)||
      (InstrumentModel == FONIX_6400)||
      (InstrumentModel == FONIX_FA10)||
      (InstrumentModel == FONIX_FA12)||
      (InstrumentModel == FONIX_FA18)
     ) return(true);
  else return(false);
};

//-------------------------------------------------------------------
//Figure out how to manage the custom option bits
//Errors: No errors, but will return 0 if no parameter pointer passed
//-------------------------------------------------------------------
DWORD __fastcall TclsFippCore::FixupCustomOptions(INT16 InstrumentType, DWORD CstOptions) {
  if ((InstrumentType <= 110)&&(InstrumentType >= 100)) InstrumentType = 100;
  switch (InstrumentType) {
    case 0,   //6500
         1,   //6400
        40,   //fp40
        100:  //100=fa10 102=fa12, 108=fa18
          //always invert these option bits, because they use the old 0xff standard
          return(!(CstOptions)); //invert cst options that we read
        
    default: //case(InstrumentType)
      //newer instruments use the custom option bits directly
      return(CstOptions); //cst options are same that we read
  }; //endswitch(InstrumentType)
};


//-------------------------------------------------------------------
//Find out if we can retry the command.
//This command looks at the FCOM error return code and determines
//if the failed command can be resent or if there is no point to even try.
//Returns true if you can retry the command, or false if no point in trying.
//-------------------------------------------------------------------
bool __fastcall TclsFippCore::CmdRetry(int Error) {
  if ((Error == FCOM_SUCCESS) ||           //no problems mate
      (Error == FCOM_FILE_NOT_FOUND) ||    //device driver file not found
      (Error == FCOM_ENTRY_NOT_FOUND) ||   //entry point to fryers driver not found
      (Error == FCOM_GENERIC_ERROR) ||     //unknown fryers driver error
      (Error == FCOM_BAD_DRIVER) ||        //bad version of Fryers Driver

      (Error == FCOM_PACKET_ERROR) ||      //Generic unknown FIPP packet xfer error
      (Error == FCOM_PORT_RANGE_ERROR) ||  //Requested port is not valid
      (Error == FCOM_PORT_NOT_FOUND) ||    //Requested port not available
      (Error == FCOM_PORT_NOT_OPEN) ||     //Requested port not open for communication
      (Error == FCOM_AUTOBAUD_FAILED) ||   //unable to find a valid baudrate
      (Error == FCOM_INVALID_FORMAT) ||    //bad FCOM command format
      (Error == FCOM_UNEXPECTED_ACK) ||    //unexpected ACK response

      //(Error == FCOM_NO_POLL) ||         //no polls from the instrument <- DO NOT RETRY ON THIS ERROR
      //(Error == FCOM_ILL_RSP) ||         //illegal command rcvd as a response <- DO NOT RETRY ON THIS ERROR
      (Error == FCOM_NAK_RESPONSE) ||      //NAK rcvd as a respond
      (Error == FIPP_IN_BOOTLOADER)) {     //tried to give non-BL cmd while in BL
    return(false); //don't retry on these errors
  }
  return(true); //try again on soft errors
};

//-------------------------------------------------------------------
//checks for ack as a response and returns error result if found
//-------------------------------------------------------------------
int __fastcall TclsFippCore::UnexpectedAck(int RspCmd) {
  if (RspCmd == FCOM_ACK) {
    return(FCOM_UNEXPECTED_ACK);
  }else{
    return(FCOM_SUCCESS);
  };
};

//==========================================================================
//FryeCom function calls
//==========================================================================

//-------------------------------------------------------------------
//returns FIPP_SUCCESS if all ok, or error if failed. Updates FcomVersion variable.
//Errors: FCOM driver error (-1000 series), FCOM_INVALID_FORMAT
//----------------------------------------------------------------------------
int __fastcall TclsFippCore::UpdateFcomVersion(void) {
  int Result = FCOM_GetFcomVersion(&FcomVersion);
  return(Result);
};

//----------------------------------------------------------------------------
//returns the highest available com port number
//Note: this is the max virtual com port not the actual max physical com port
//Currently the normal returned value is 99.
//MaxPort = 0->99 (Com1->Com100)
//Errors: FCOM driver error (-1000 series), FCOM_INVALID_FORMAT
//----------------------------------------------------------------------------
int  __fastcall TclsFippCore::GetMaxPort(int* pMaxPort) {
  int Result;
  Result = FCOM_MaxComPort(pMaxPort);
  return(Result);
};

//-------------------------------------------------------------------
//This command updates the local port status registers.
//The information can be used to display what is happening with the port.
//Errors: FCOM driver error (-1000 series), FCOM_INVALID_FORMAT
//May also return one of the following errors with the PortStatus record updated.
//FCOM_PORT_RANGE_ERROR, FCOM_PORT_NOT_OPEN
//-------------------------------------------------------------------
int __fastcall TclsFippCore::UpdatePortStatus(void) {
  int Result = FCOM_GetPortStatus(ThisComPort,&PortStatus);
  if (PortStatus.Baudrate > 0) ActiveBaudrate = PortStatus.Baudrate;
  return(Result);
};


//---------------------------------------------
//Sets the amount of time to wait for no poll in milliseconds
//(actual internal resolution is in 55ms ticks).
//Note: Setting the nopoll timeout will reset the timeout counter
//If the nopoll flag has not yet been set, then it will not be set
//until the new time has passed without a poll. If the flag is
//already set, it will remain set until a poll is received,
//at which time the new timeout period will take effect.
//This has no other affect on other fryers actions,
//even if in the middle of a packet transfer.
//The only way to clear a nopoll flag once set is to either get a
//valid poll, call ResetPacket, or disable then re-enable the packet mode.
//If the specified Timeout is out of range, the default poll timeout
//of 5 seconds (5000) will be used instead.
//Returns FCOM_SUCCESS if all ok, else returns error.
//Errors: FCOM driver error (-1000 series), FCOM_PORT_RANGE_ERROR
//FCOM_GENERIC_ERROR, FCOM_PORT_NOT_OPEN
int __fastcall TclsFippCore::SetNoPoll(int Timeout) {
  int Result;
  if (Timeout >= 0) {
    Result = FCOM_SetNoPollTimeout(ThisComPort,Timeout);
  }else{
    Result = FIPP_INVALID_FORMAT; //prohibit negatve values
  }
  return(Result);
};

//---------------------------------------------
//Gets the amount of time before a no poll condition is triggered in milliseconds.
//Returns FCOM_SUCCESS if all ok, else returns error.
//Errors: FCOM driver error (-1000 series), FCOM_PORT_RANGE_ERROR
//FCOM_GENERIC_ERROR, FCOM_PORT_NOT_OPEN
int __fastcall TclsFippCore::GetNoPoll(int* pTimeout) {
  int Result;
  Result = FCOM_GetNoPollTimeout(ThisComPort,pTimeout);
  return(Result);
};


//===========================================================================
//Open / Init / Check port functions.
//===========================================================================


//ok---------------------------------------------
//Closes the FIPP communications port if open.
//If the port is already closed, does nothing (no error).
//Returns FCOM_SUCCESS if closed properly (or already closed) else returns error.
//Errors: FCOM driver error (-1000 series), FCOM_PORT_RANGE_ERROR
//---------------------------------------------
int __fastcall TclsFippCore::ClosePort(void) {
  PortInitialized = false;    //port is no longer initialized
  int Result = FCOM_ClosePort(ThisComPort); //now close the port
  UpdatePortStatus();         //always update the port status info on exit (ignore errors)
  return(Result);
};

//ok---------------------------------------------------------------------------
//Try to open a comport and initiate communications
//(comport:0->99) 0=com1, 1=com2, 2=com3, 3=com4, etc...
//Tries to initialize the comport. If AutoPortseek is enabled, seeking starts
//with NewComPort going through to StopPort and starting over with StartPort.
//If AutoPortseek is disabled, only NewComPort is tried (StartPort and StopPort are ignored).
//If StartPort and StopPort are set to -1 and AutoPortSeek=enabled, all possible com ports are searched.
//If NewComPort is outside the valid port range, COM1 is used instead.
//MaxRetry defines the number of times we will retry each port.
//For AutoPortSeek=off, total retry attempts is MaxRetry.
//For AutoPortSeek=on, total port open attempts is: MaxRetry * (StopPort - StartPort + 1)
//If Init is successful, tries to read the machine information (FIPP cmd 28).
//If CallBack != NULL, calls the indicated callback procedure periodically.
//  Callback is performed once at the start of each port open attempt,
//  once every 100ms or so during port synchronization attempts,
//  or whenever activity is detected during port synchronization attempts.
// If the program passes back a FCOM_CANCEL from the callback, the port
// initialization will be canceled and the FCOM_CANCEL error result is returned.
//The callback also allows the application a chance to service Windows messages as needed.
//If NewComPort is already open, this command does nothing.
//If AutoPortSeek is used, any port that was previously opened (and not closed)
//will not be found (Windows prevents multiple-access to a port already in use).
//You must first close the port if you want to "find" it again.
//If AutoBaudSeek is enabled, the baudrate will be automatically detected
//and matched. If AutoBaudSeek is disabled, the specified StartBaud is used.
//Normally it is recommened that the AutoBaudSeek feature be used unless you
//have a reason not to use it.
//See: FCOM_NO_SEEK, FCOM_PORT_SEEK and FCOM_BAUD_SEEK in FComDefs.h
//AutoSeek:0 = disable AutoPortSeek, disabled AutoBaudSeek
//AutoSeek:1 = enable AutoPortSeek,  disabled AutoBaudSeek
//AutoSeek:2 = disable AutoPortSeek, enabled AutoBaudSeek
//AutoSeek:3 = enable AutoPortSeek,  enabled AutoBaudSeek
//(ie: BIT0=1=enabled AutoPortSeek, BIT1=1=enabled AutoBaudSeek)
//Returns FCOM_SUCCESS if port successfully opened else returns error.
//Errors: FCOM driver error (-1000 series), FCOM_PORT_RANGE_ERROR
// FCOM_PORT_NOT_FOUND, FCOM_PORT_NOT_OPEN, FCOM_GENERIC_ERROR,
// FCOM_CANCEL, FCOM_NO_POLL, FCOM_AUTOBAUD_FAILED
//---------------------------------------------------------------------------
int __fastcall TclsFippCore::InitPort(int AutoSeek, int NewComPort, int StartPort, int StopPort,
                                  int MaxRetry, int StartBaud, FCOM_TfcCallback Callback) {
  int i;
  int Result;
  int MaxPort;
  int RetryLimit;

  //If this fails it is because there is no driver available, so can't go on.
  Result = FCOM_MaxComPort(&MaxPort);
  if (Result != FCOM_SUCCESS) return(Result);
  //If comport parameters out of range, force to valid numbers.
  if ((NewComPort < 0)||(NewComPort > MaxPort)) NewComPort = 0;
  if ((StopPort > MaxPort)||(StopPort < 0)) StopPort = MaxPort;
  if ((StartPort < 0)||(StartPort > MaxPort)) StartPort = 0;
  if ((MaxRetry <= 0)||(MaxRetry > 100)) MaxRetry = 3; //validate MaxRetry.
  RetryLimit = MaxRetry * abs(StopPort - StartPort + 1);

  LookingForPort = true;     //mark that we are looking for the port
  ThisComPort = NewComPort;  //store this port to the FippObject
  if ((AutoSeek & FCOM_USE_PORT_SEEK) == 0) {
    //if autoportseek = 0, try to open the specified port
    for (i=0; i<MaxRetry; i++) {
      Result = FCOM_OpenPort(ThisComPort, AutoSeek, StartBaud, Callback);
      if ((Result == FIPP_CANCEL)||(Result == FIPP_SUCCESS)) {
        break;
      }
    }//endfor(i)

  }else{
    //auto-detect - if autoseek is true, try to find the port starting at NewComPort.
    for (i=0; i<RetryLimit; i++) {
      Result = FCOM_OpenPort(ThisComPort, AutoSeek, StartBaud, Callback);
      if ((Result == FCOM_CANCEL)||(Result == FCOM_SUCCESS)) {
        break;
      }else{
        ThisComPort = ThisComPort + 1;                       //if failed, try next port
        if (ThisComPort > StopPort) ThisComPort = StartPort; //wrap back to start port if overflow
      }
    };//endfor(i)
  }//endif(Autoseek)

  //If failed to find the comport, force the port not found error
  if (Result != FCOM_SUCCESS) {
    FCOM_ClosePort(ThisComPort);  //Make sure port is closed
    Result = FCOM_PORT_NOT_FOUND; //and return not found error
  }

  LookingForPort = false;                 //mark that we are done looking for the port
  UpdatePortStatus();                     //update the port status info
  if (Result == FIPP_SUCCESS) {           //now check on the results of the seek.
    PortInitialized = true;               //port is initialized
    SetNoPoll(FIPP_STD_POLL_TIMEOUT);     //set nopoll timeout value to default startup value
    return(Result);
  }else{
    PortInitialized = false;              //port is not initialized
    return(FCOM_PORT_NOT_FOUND);          //Mark that we did not find the port
  }
};

//ok---------------------------------------------
//This get the SendReady Status.
//If CallBack != NULL, calls the indicated callback procedure periodically.
//If the comport is open and synchronized to the instrument, the callback
//will not be performed (it is ignored). If Port synchronization is lost
//(due to baudrate change), the Callback is performed once every 100ms or so
//  during the port re-synchronization attempts, or whenever activity is
//  detected during the port re-synchronization attempts.
// If the program passes back a FCOM_CANCEL from the callback, the port
// operation will be canceled and the FCOM_CANCEL error result is returned.
//The callback also allows the application a chance to service Windows messages as needed.
//Returns FIPP_SUCCESS if ready to send a command or FIPP_NOT_READY if not ready.
//Can also return error failure codes.
//Errors: FCOM driver error (-1000 series), FCOM_INVALID_FORMAT
//FCOM_PORT_RANGE_ERROR, FCOM_GENERIC_ERROR, FCOM_PORT_NOT_OPEN,
//FCOM_AUTOBAUD_FAILED, FCOM_GENERIC_ERROR, FCOM_CANCEL,
//FCOM_NOT_READY, FCOM_NO_POLL
//---------------------------------------------
int __fastcall TclsFippCore::SendReady(FCOM_TfcCallback Callback) {
  int Result;

  Result = FCOM_SendReady(ThisComPort, Callback); //get Send ready status
  //Also update the port status after SendReady (but ignore the error return)
  FCOM_GetPortStatus(ThisComPort,&PortStatus);
  return(Result);
};

//ok---------------------------------------------
//Function to wait until the send ready flag is set or an error occurs.
//See the SendReady description for details of the callback functionality
//and errors that can be returned.
//Returns FCOM_SUCCESS if send is ready, or error condition if failed or canceled.
//If you use this call, you should use the callback procedure to
//allow your program to cancel the loop in case it never sees a ready condition.
//The callback also allows the application a chance to service Windows messages as needed.
//Can also return error failure codes.
//Errors: FCOM driver error (-1000 series), FCOM_INVALID_FORMAT
//FCOM_PORT_RANGE_ERROR, FCOM_GENERIC_ERROR, FCOM_PORT_NOT_OPEN,
//FCOM_AUTOBAUD_FAILED, FCOM_GENERIC_ERROR, FCOM_CANCEL, FCOM_NO_POLL
//---------------------------------------------
int __fastcall TclsFippCore::WaitForSendReady(FCOM_TfcCallback Callback) {
  int Result;

  while(FOREVER) {
    Result = FCOM_SendReady(ThisComPort, Callback);
    if (Result != FCOM_NOT_READY) break;
    UpdatePortStatus();
  }
  return(Result);
};

//ok-----------------------------------------------------------
// SENDS THE COMMAND AND ACCOMPANYING DATA TO THE INSTRUMENT
//The FIPP command array is passed in pCmdArray.
//NoPollTime is used to specify the amount of time (in milliseconds) before
// the no poll status flag is set. This is used to specify how long you expect
// the command to take so that alternate action can be taken if it takes too
// long (no response from the instrument). If you  set the NoPollTime period
// to zero it will be left at the current setting.
// Don't set the NoPollTime too short, or you may have problems with excessive
// FCOM_NO_POLL errors.
//Note: you can load the send command even if a no poll condition exists.
//If you try to send a new command before the previous command is finshed
//(see SendReady), the send command will fail with a FCOM_SEND_OVERRUN and
//the new command will not be sent (the existing command will run to completion).
//Returns FCOM_SUCCESS if send is successful.
//Can also return error failure codes.
//Errors: FCOM driver error (-1000 series), FCOM_INVALID_FORMAT
//FCOM_PORT_RANGE_ERROR, FCOM_GENERIC_ERROR, FCOM_PORT_NOT_OPEN,
//FCOM_GENERIC_ERROR, FCOM_SEND_OVERRUN, FCOM_SEND_ERROR
//---------------------------------------------
int __fastcall TclsFippCore::SendCmd(int NoPollTimeout, FCOM_TfcArray* pCmdArray) {
  int Result;
  Result = FCOM_SendCmdArray(ThisComPort,pCmdArray);
  if (NoPollTimeout > 0) SetNoPoll(NoPollTimeout);  //set no_poll timeout value after command is sent
  return(Result);
};


//ok---------------------------------------------
//Function to wait until the response ready flag is set or an error occurs.
//Returns FCOM_SUCCESS if the response is ready, or error condition if failed.
//Can also return error failure codes.
//Errors: FCOM driver error (-1000 series), FCOM_PORT_RANGE_ERROR,
//FCOM_GENERIC_ERROR, FCOM_PORT_NOT_OPEN, FCOM_GENERIC_ERROR,
//FCOM_NO_POLL, FCOM_SEND_OVERRUN, FCOM_SEND_ERROR, FCOM_NO_RESPONSE,
//FCOM_RESPONSE_ERROR, FCOM_RESPONSE_OVERRUN, FCOM_NAK_RESPONSE
//---------------------------------------------
int __fastcall TclsFippCore::RspReady(void) {
  int Result = FCOM_RspReady(ThisComPort);
  return(Result);
};

//ok---------------------------------------------
//Function to wait until the response ready flag is set or an error occurs.
//See the SendReady description for details of the callback functionality
//and errors that can be returned.
//Returns FCOM_SUCCESS if response is ready, or error condition if failed or canceled.
//If you use this call, you should use the callback procedure to
//allow your program to cancel the loop in case it never sees a ready condition.
//The callback also allows the application a chance to service Windows messages as needed.
//Can also return error failure codes.
//Errors: FCOM driver error (-1000 series), FCOM_PORT_RANGE_ERROR,
//FCOM_GENERIC_ERROR, FCOM_PORT_NOT_OPEN, FCOM_CANCEL,
//FCOM_NO_POLL, FCOM_SEND_OVERRUN, FCOM_SEND_ERROR, FCOM_NO_RESPONSE,
//FCOM_RESPONSE_ERROR, FCOM_RESPONSE_OVERRUN, FCOM_NAK_RESPONSE
int __fastcall TclsFippCore::WaitForRspReady(FCOM_TfcCallback Callback) {
  int Result;

  while(FOREVER) {
    Result = FCOM_RspReady(ThisComPort);
    if (Result != FCOM_NOT_READY) break;
    UpdatePortStatus();
    //We call the callback here because RspReady doesn't call it.
    //But we need it to be able to break out of this loop if requested.
    if (Callback != NULL) Result = Callback(ThisComPort);
    if (Result == FCOM_CANCEL) break;
  }
  return(Result);
};

//ok-----------------------------------------------------------
//This picks up the response from a previously sent command
//The FIPP response array is passed in pRspArray.
//If you try to get a response before the response ready, you will get the
// previous response data. There is no error condition for this situation.
// you should call RspReady() first to insure that the response is ready
// before calling GetResponse. If you do not read a response before
// sending another command, the new response will overwrite the unread
// response data (the new respojnse data will be returned in the pRspArray)
// and the FCOM_RESPONSE_OVERRUN error will be returned.
//The response data is placed in the pRspArray.
//Note: it is possible to get an ACK response when a data response is
//expected. This situation cannot be determined by this level of the
//program. It is up to the main application code to decide if an ACK
//is the appropriate response to the command that was sent.
//If an ACK is an acceptable response, set NoACK to zero.
//If an ACK is NOT an acceptable response, set NoACK to 1.
//Returns FCOM_SUCCESS if response read is successful.
//Can also return error failure codes.
//Errors: FCOM driver error (-1000 series), FCOM_INVALID_FORMAT
//FCOM_PORT_RANGE_ERROR, FCOM_GENERIC_ERROR, FCOM_PORT_NOT_OPEN,
//FCOM_GENERIC_ERROR, FCOM_SEND_OVERRUN, FCOM_SEND_ERROR,
//FCOM_RESPONSE_ERROR, FCOM_RESPONSE_OVERRUN, FCOM_NAK_RESPONSE,
//FCOM_ILL_RESPONSE, FCOM_POLL_RESPONSE, FCOM_INVALID_FORMAT,
//FCOM_UNEXPECTED_ACK, FCOM_NO_RESPONSE
int __fastcall TclsFippCore::GetResponse(int NoACK, FCOM_TfcArray* pRspArray) {
  int Result;

  Result = FCOM_GetRspArray(ThisComPort,pRspArray);
  if (pRspArray == NULL) return(Result); //if no array provided, we are done.
  if ((Result == FCOM_SUCCESS) && (NoACK != FCOM_ACK_OK)) {
    Result = UnexpectedAck(pRspArray->Cmd);
  }
  return(Result);
};

//-------------------------------------------------------------------
//DoCmd is encapsulation of the send and response FCOM commands
//to perfrom a complete coomand sequence.
//While this function can be used to send a FIPP command array to a
//Fonix instrument and get it's response, you should consider using
//a higher level approach where your application monitors and controls
//the command progress. Otherwise you may not be allowing Windows to
//function properly because this function will hog the cpu.
//If you do use this function, at least implement a minimal callback
//that can cancel the command if it gets hungup, and has a call in it
//to service the Windows Messages while waiting on the command.
//Errors: FCOM driver error (-1000 series), FCOM_INVALID_FORMAT
//FCOM_PORT_RANGE_ERROR, FCOM_GENERIC_ERROR, FCOM_PORT_NOT_OPEN,
//FCOM_SEND_OVERRUN, FCOM_SEND_ERROR, FCOM_CANCEL,
//FCOM_RESPONSE_ERROR, FCOM_RESPONSE_OVERRUN, FCOM_NAK_RESPONSE,
//FCOM_ILL_RESPONSE, FCOM_POLL_RESPONSE, FCOM_INVALID_FORMAT,
//FCOM_UNEXPECTED_ACK
//-------------------------------------------------------------------
int __fastcall TclsFippCore::DoCmd(int NoACK, int NoPollTime, FCOM_TfcArray* pCmdArray,
                               FCOM_TfcArray* pRspArray, FCOM_TfcCallback Callback) {
  int Result;
  Result = WaitForSendReady(Callback);
  if (Result != FIPP_SUCCESS) return(Result);
  Result = SendCmd(NoPollTime,pCmdArray);
  if (Result != FIPP_SUCCESS) return(Result);
  //All FIPP commands except Quick Terminate have a response
  if (pCmdArray->Cmd != FCOM_QTC) {
    Result = WaitForRspReady(Callback);
    if (Result != FIPP_SUCCESS) return(Result);
    Result = GetResponse(NoACK,pRspArray);
  }
  return(Result);
};


//===========================================================================
//Standard instrument commands.
//===========================================================================


//-------------------------------------------------------------------
//This function does a quick terminate command unless the previous
//result was a cmd failure. QuickTerminate is used to speed up
//calls to the instrument by telling it to immediately perfrom the
//command that was just sent rather than hanging around for more commands.
//Returns SendCmd errors.
//-------------------------------------------------------------------
int __fastcall TclsFippCore::QuickTerminate(FCOM_TfcCallback Callback) {
  int Result = DoCmd(FCOM_NO_ACK, 0, (FCOM_TfcArray*)fipp_QuickTerminateCmd, NULL, Callback);
  return(Result);
};


//<eof>


