//FonixID.CPP
//This file is used to manage the instrument ID information
//---------------------------------------------------------------------------
#include <vcl.h>
#pragma hdrstop

#include <stdlib.h>

#include "Fryedefs.h"
#include "FryeStr.h"
#include "TextFile.h"
#include "FonixID.h"

//---------------------------------------------------------------------------
//#pragma package(smart_init)

//---------------------------------------------------------------------------
//How to use this information.
//The purpose of this file is to help the program identify the instrument
//it is talking to. Call FID_LoadFonixID() at the start of your program.
//It checks for the file FONIXID.DAT. If the file exists, the Fonix ID
//info is loaded form the file. If the file is not found, we revert to the
//internal FID info noted below.
//The purpose of the FONIXID.DAT file is to allow adding new Fonix instrument
//names without having to hard code them into your program.
//Note: On newer instruments you can query the model string and number directly
//from the instrument. This method is mainly intended for older instruments
//without that feature.
//To get the instrument model number from the model string, use FID_GetInstrumentModel()
//To get the instrument model string from the model number and/or instrument type
//use FID_GetInstrumentName().
//GetInstrumentName uses the InstrumentModel as a reference first, If that is
//not available, it uses the InstrumentType and tries to figure out what the
//instrument is from that. Newer instruments all use the Model number scheme.
//Origonally the InstrumentType was supposed to be used to uniquely identify
//the instrument, but poor programming practics caused that protocol to be violated
//requiring the addition of the Model Number to provide the identification.

//---------------------------------------------------------------------------
enum {
  INDEX_6500 = 0,
  INDEX_6400 = 1,
  INDEX_M340 = 2,
  INDEX_FP35 = 3,
  INDEX_FP40 = 4,
  INDEX_FA10 = 5,
  INDEX_FA12 = 6,
  INDEX_FA18 = 7,
  INDEX_7000 = 8,
  INDEX_8000 = 9,
  INDEX_8001 = 10,
};
#define NUMBER_INSTRUMENTS 11 //number of defined instruments
INT16 InstrumentNumberList[NUMBER_INSTRUMENTS] = {
  0,   //0:6500
  1,   //1:6400
  6,   //2:M340
  35,  //3:FP35
  40,  //4:FP40
  100, //5:FA10
  102, //6:FA12
  108, //7:FA18
  7000,//8:7000
  8000,//9:8000
  8001,//10:8001
};
char* InstrumentNameList[NUMBER_INSTRUMENTS] = {
  "6500",  //0:0
  "6400",  //1:1
  "M340",  //2:6
  "FP35",  //3:35
  "FP40",  //4:40
  "FA10",  //5:100
  "FA12",  //6:102
  "FA18",  //7:108
  "7000",  //8:7000
  "8000",  //9:8000
  "8001",  //10:8001
};


typedef struct {
  int InstType;
  int SubType;
  int InstModel;
  AnsiString aName;  //four character name string
  AnsiString aModel; //full name string (8 characters)
}tFonixIDinfo;

tFonixIDinfo FonixID[MAX_FONIX_ID+2];

int FID_FileStatus = -1;  //status result from trying to read FonixID file
int fid_cnt = 0;
TTextFile* FID_Textfile;

AnsiString FID_aFilename = "FonixID.DAT";

//===========================================================================

//This tries to load the fonix instrument ID file.
//returns FID_NO_FILE if file is not found.
//returns FID_NO_DATA if no data found.
//returns FID_CORRUPT if data is corrupt.
//returns FID_OVERRUN if too many IDs in file (see MAX_FONIX_ID)
//returns FID_ERROR if unknown error encountered (exception).
//returns FID_SUCCESS if file read ok.
//If the file has not yet been read, FID_Status will have FD_NOT_READ in it.
//Lines with non-numeric character as the first character are ignored
//Empty lines are ignored.
//Leading and trailing spaces in the text name are truncated.
//Names should be exactly four characters, however it is not prohibited by this routine.
int FID_LoadFonixID(AnsiString* aFilename) {
  bool NumberOK;
  int i;
  int Result;
  int Index;
  int LineCount;
  AnsiString aTemp;
  char sTemp[9999];
  char sTemp2[9999];

  Index = 0;
try{
  FID_Textfile = new TTextFile();
  Result = FID_NO_FILE; //assume not found
  if (FID_Textfile->Exists(*aFilename) == true) {
    FID_Textfile->Open(*aFilename);
    Result = FID_NO_DATA;
    LineCount = FID_Textfile->LineCount();
    if (LineCount > 0) { //no data found
      for (i=0; i < LineCount; i++) {
        FID_Textfile->Readln();
        //read the first line - ignore if not a valid line
        if (FID_Textfile->cReadStr(&aTemp) >= 0) {
          NumberOK = FS_StringToInt(aTemp.c_str(),&FonixID[Index].InstType);
          if (NumberOK == true) { //only use the line if it starts with a number
            if (FID_Textfile->cReadInt(&FonixID[Index].SubType,0) != 0) {
              Result = FID_CORRUPT; //data file is corrupt
              break;
            }//endif(SubType)
            if (FID_Textfile->cReadInt(&FonixID[Index].InstModel,0) != 0) {
              Result = FID_CORRUPT; //data file is corrupt
              break;
            }//endif(SubType)
            if (FID_Textfile->cReadStr(&FonixID[Index].aName) != 0) {
              Result = FID_CORRUPT; //corrupt data file
              break;
            }else{
              FS_SCopy(sTemp,FonixID[Index].aName.c_str());
              FS_TrimLeadingSpaces(sTemp);
              FS_TrimTrailingSpaces(sTemp2,sTemp);
              FonixID[Index].aName = sTemp2;
              if (FonixID[Index].aName.Length() <= 0) {
                Result = FID_CORRUPT;
                break;
              }else{
                Index++;    //line ok, so increment the Index
                if (Index <= MAX_FONIX_ID) {
                  Result = FID_SUCCESS; //good result so far
                }else{
                  Result = FID_OVERRUN; //too many items in the file
                  break;
                }
              }
            }//endif(aName)
          }//endif(NumberOK)
        }//endif(InstrumentNumber)
      }//endfor(i)
    }//endif(LineCount)
    FID_Textfile->Close();
  }
  delete FID_Textfile;
  if (Result == 0) {
    fid_cnt = Index;
  }else{
    fid_cnt = 0;
  }
}catch(...){
  delete FID_Textfile;
  return(FID_ERROR); //generic error
}
  FID_FileStatus = Result;  //update FID status with result.
  return(Result);
};

//---------------------------------------------------------------------------
//This goes through the externally loaded fonix ID list (if available)
//to try to find the instrument.
//If Model Number provided, the string is determined frm that.
//if model number not available, tries to figure it out from InstType and SubType
//If found, returns the four character ID string in psName
int fid_CheckFonixID(INT16 InstType, INT16 SubType, INT16 InstModel, char* psName) {
  int i;
  //if file never loaded, can't use it
  if (FID_FileStatus < 0) return(FID_FileStatus);
  //if model Number provieed, do it this way
  if (InstModel > 0) {
    for (i=0; i<fid_cnt; i++) {
      if (FonixID[i].InstModel == InstModel) {
          FS_SCopy(psName,FonixID[i].aName.c_str()); //return the name
            return(FID_SUCCESS);
      }//endif(InstModel)
    }//endfor(i)
  }

  //fall through to here if model number not found
  for (i=0; i<fid_cnt; i++) {
    if (FonixID[i].InstType == InstType) {
      if (FonixID[i].SubType != 32768) {
        if (FonixID[i].SubType == SubType) {
          FS_SCopy(psName,FonixID[i].aName.c_str()); //return the name
          return(FID_SUCCESS);
        }//endif(SubType==SubType)
      }//endif(SubType==Valid)
    }//endif(Number)
  }//endfor(i)
  return(FID_NOT_FOUND); //instrument number not found
};

//---------------------------------------------------------------------------
//Given a number, returns the instrument name.
//if number not found in the list, returns false result with default string
//Note: 8000 is handled as a special case because Aaron refused to use
//the defined instrument identification number method. This means all software
//has to use special case hack to determine the instrument type.
//The 8000 is identified as a 7000 with a sub-type number of 1.
//The 7000 is identified as a 7000 with a sub-type number of 0.
//To help alieviate the problem in the future and to make the addition of
//newer instruments easier for production (not having to change this program),
//The instrument ID can be read from a file called FONIXID.DAT, which contains
//the instrument ID number, Sub-type, and four character Name string.
//Note: If the FonixID file contains 0x8000 as the sub-type, it means that
//the subtype is ignored. If the ID file contains a valid number, then
//the subtype is set to a valid number.
//Warning, a part if the problem with this is that when the instrument is
//in boot loader, the subtype is returned as -1, so there is no way to know
//what the instrument sub-type is in that situation.
//Returns SUCCESS if name was found, false if not.
//The FONIXID.DAT file has priority, but if it is not found, the internal list is used.
//The default condition is to convert the InstrumentModel number into a string name.
int FID_GetInstrumentName(INT16 InstType, INT16 SubType, INT16 InstModel, char* psName) {
  char Itmp[9999];
  int Result;
  int Which;
  Result = fid_CheckFonixID(InstType,SubType,InstModel,psName);
  //if FonixID.dat file not found, try for known instruments inside this program
  if (Result != FID_SUCCESS) {
    if (InstModel > 0) Which = InstModel; else Which = InstType;
    switch(Which) {
      case 0: FS_SCopy(psName, InstrumentNameList[INDEX_6500]); return(FID_SUCCESS);
      case 1: FS_SCopy(psName, InstrumentNameList[INDEX_6400]); return(FID_SUCCESS);
      case 6: FS_SCopy(psName, InstrumentNameList[INDEX_M340]); return(FID_SUCCESS);
      case 35: FS_SCopy(psName, InstrumentNameList[INDEX_FP35]); return(FID_SUCCESS);
      case 40: FS_SCopy(psName, InstrumentNameList[INDEX_FP40]); return(FID_SUCCESS);
      case 100: FS_SCopy(psName, InstrumentNameList[INDEX_FA10]); return(FID_SUCCESS);
      case 102: FS_SCopy(psName, InstrumentNameList[INDEX_FA12]); return(FID_SUCCESS);
      case 108: FS_SCopy(psName, InstrumentNameList[INDEX_FA18]); return(FID_SUCCESS);;
      case 8000: FS_SCopy(psName, InstrumentNameList[INDEX_8000]); return(FID_SUCCESS);
      case 8001: FS_SCopy(psName, InstrumentNameList[INDEX_8001]); return(FID_SUCCESS);
      case 7000: {
                   if (SubType == 0) {
                     FS_SCopy(psName, InstrumentNameList[INDEX_7000]); return(FID_SUCCESS);
                   }else if (SubType == 1) {
                     FS_SCopy(psName, InstrumentNameList[INDEX_8000]); return(FID_SUCCESS);
                   }
                 }
      default:
        itoa(InstType,Itmp,10);
        FS_SCopy3(psName,"?(",Itmp,")");
        return(Result); //return the FID Status result if instrument not found
     }//endswitch()
  }//endif(Result)
  return(FID_SUCCESS);
};

//---------------------------------------------------------------------------
//Takes a Fonix model string and returns the Fonix model number.
//returns FID_SUCCESS if model name found, with ModelNumber = instrument model
//If name not found, returns FID_NOT_FOUND and model number is not updated.
//Note: FID_Status is set by calling FID_LoadFonixID(). See notes at top of this file.
int FID_GetInstrumentModel(char* psName, INT16* ModelNumber) {
  int i;

  //If the FonixID file was found, use it for the information
  if (FID_FileStatus == FID_SUCCESS) {
    for (i=0; i<fid_cnt; i++) {
      if (CompareText(FonixID[i].aName.c_str(),psName) == 0) {
        *ModelNumber = (INT16)FonixID[i].InstModel;
        return(FID_SUCCESS);
      }
    }//endfor(I)
  }
  //Either the FonixID file was not found, or the FID was not found
  //in the file, so check our internal lis of known devices.
  for (i=0; i<NUMBER_INSTRUMENTS; i++) {
    if (CompareText(InstrumentNameList[i],psName) == 0) {
      *ModelNumber = InstrumentNumberList[i];
      return(FID_SUCCESS);
    }
  }
  return(FID_NOT_FOUND);
};

