//FryeCom Dynamic DLL interface unit
//Copyright 2010 Frye Electronics, Inc.
//Written by Michael Day as of 30 Aug 2010
//You may use this code for any lawful purpose without fees or royalties.
//This code is intended as an example showing how to communicate with Fonix
//equipment using the FryeCom.DLL device driver on the Windows operating system.
//No warranties are express or implied in the use of this code.
//This will compile under any version of Borland BCB (written using BCB V3.00)
//This code uses only C specific code so it should compile with most c compilers.
//Also see the FryeComS version of the FryeCom.DLL interface.
//The FryeComS version will statically load the DLL at program startup.
//Both methods work fine, but the dynamocally loaded version provide a little more
//control over the use of the DLL at the expense of having to explictly load
//the DLL file and the Procedure addreses (which we do for you in this unit).
//An advantage of the Dynamic loaded DLL is that you can catch any errors
//in your program and make your own decisions on how to handle them.
//With a Statically loaded DLL, the DLL is managed outside your program
//and Windows will halt the program with an exception window if anything
//goes wrong.
//----------------------------------------------------

#include "FryeComD.h"  //FryeCom DLL interface definitions

//-----------------------------------------------------------------------
//This method uses the dynamic DLL loading convention.
//If you use a static load (load the dll at application startup),
//Windows will refuse to start the application if the DLL is not found.
//Doing it this way allows the application to notify the user
//about the problem and/or ignore it if you don't need the DLL.
//You can call FCOM_DllLoaded() to attempt to pre-load the DLL.
//If the DLL load failed, the error indicating the cause is returned.
//If you don't preload the DLL, then it will be loaded the first time
//a call to the DLL is made.
//If you need direct access to the DLL, you can use the FCOM_FryersLibHandle.
//as the Windows reference handle (it is created at the time the DLL is loaded.)
//-----------------------------------------------------------------------
//exports
//  CallFryers       index 1,  //call thru to Fryers32
//  CallFryers_C     index 2,  //call thru to Fryers32 (not supported here)
//
//  OpenPort         index 3,  //Open the selected port
//  ClosePort        index 4,  //close the current port
//  GetPortStatus    index 5,  //get operational status of the port
//
//  SendReady        index 6,  //check if Ready to send
//  SendCmdArray     index 7,  //send a command array
//  RspReady         index 8,  //check if a response is available
//  GetRspArray      index 9,  //get a response array
//
//  SetNoPollTimeout index 10, //set new no_poll timeout value
//  GetNoPollTimeout index 11, //get current no_poll timeout value
//  GetFVersion      index 12, //*get FryeCom Driver version number
//  MaxComPort       index 13, //*get max possible available com port
//  GetDebugStatus   index 14; //get extended debug port status
//
//  SetPortMode index 15, //set new port mode value
//  GetPortMode index 16, //get current port mode value
//
//--> *Note: GetFVersion and MaxComPort do not require a ComPort handle.

//Old style Fryer call format
typedef void __stdcall (*tCallFryers)(F_RegsType* IRegs);   //export 1

//Open, close Fryers ports and Status info
typedef int __stdcall (*tOpenPort)(int ComPort, int PortControl, int Baudrate,
                                   FCOM_TfcCallback Callback);  //export 3
typedef int __stdcall (*tClosePort)(int ComPort);          //export 4

typedef int __stdcall (*tGetPortStatus)(int ComPort, FCOM_tPortStatusRec* pPortStatus); //export 5

//Fryers command transfers
typedef int __stdcall (*tSendReady)(int ComPort, FCOM_TfcCallback Callback);  //export 6
typedef int __stdcall (*tSendCmdArray)(int ComPort, FCOM_TfcArray* pData);    //export 7
typedef int __stdcall (*tRspReady)(int ComPort);                         //export 8
typedef int __stdcall (*tGetRspArray)(int ComPort, FCOM_TfcArray* pData);     //export 9

//Special maintance commands.
typedef int __stdcall (*tSetNoPollTimeout)(int ComPort, int Value);  //export 10
typedef int __stdcall (*tGetNoPollTimeout)(int ComPort, int* pValue); //export 11

typedef int __stdcall (*tGetFcomVersion)(int* pFVersion);      //export12
typedef int __stdcall (*tMaxComPort)(int* pMaxPort);           //export 13
typedef int __stdcall (*tGetDebugStatus)(int ComPort, FCOM_tDebugStatusRec* pDebugStatus); //export 14

typedef int __stdcall (*tSetPortMode)(int ComPort, int NewPortMode);      //export 15
typedef int __stdcall (*tGetPortMode)(int ComPort, int* CurrentPortMode); //export 16

bool FCOM_FryersLoaded = false;
HINSTANCE FCOM_FryersLibHandle = NULL;
char const FCOM_DLL_NAME[] = "fryecom.dll";
int FCOM_ProcedureNumber = -1;  //used for debugging to indicate which call failed

//These are pointers to the DLL procedures contained in FryeCom
tCallFryers     fcom_CallFryersProc    = NULL; //export 1

tOpenPort       fcom_OpenPortProc      = NULL; //export 3
tClosePort      fcom_ClosePortProc     = NULL; //export 4
tGetPortStatus  fcom_GetPortStatusProc = NULL; //export 5

tSendReady      fcom_SendReadyProc     = NULL; //export 6
tSendCmdArray   fcom_SendCmdArrayProc  = NULL; //export 7
tRspReady       fcom_RspReadyProc      = NULL; //export 8
tGetRspArray    fcom_GetRspArrayProc   = NULL; //export 9

tSetNoPollTimeout fcom_SetNoPollTimeoutProc = NULL; //export 10
tGetNoPollTimeout fcom_GetNoPollTimeoutProc = NULL; //export 11
tGetFcomVersion   fcom_GetFcomVersionProc   = NULL; //export 12
tMaxComPort       fcom_MaxComPortProc       = NULL; //export 13
tGetDebugStatus   fcom_GetDebugStatusProc   = NULL; //export 14

tSetPortMode fcom_SetPortModeProc = NULL; //export 15
tGetPortMode fcom_GetPortModeProc = NULL; //export 16

//-----------------------------------------------------------------------

//Initialize the DLL procedures
void fcom_InitDllProcs(void) {
  fcom_CallFryersProc = NULL;
  fcom_OpenPortProc = NULL;
  fcom_ClosePortProc = NULL;
  fcom_GetPortStatusProc = NULL;
  fcom_SendReadyProc = NULL;
  fcom_SendCmdArrayProc = NULL;
  fcom_RspReadyProc = NULL;
  fcom_GetRspArrayProc = NULL;
  fcom_SetNoPollTimeoutProc = NULL;
  fcom_GetNoPollTimeoutProc = NULL;
  fcom_GetFcomVersionProc = NULL;
  fcom_MaxComPortProc = NULL;
  fcom_GetDebugStatusProc = NULL;
  fcom_SetPortModeProc = NULL;
  fcom_GetPortModeProc = NULL;
};

//Check if the DLL has been loaded yet.
//If not, try to load it.
//returns error code if failed.
int FCOM_DllLoaded(void) {
  FCOM_FryersLoaded = true;
  if (FCOM_FryersLibHandle == NULL) {
    fcom_InitDllProcs();
    FCOM_FryersLibHandle = LoadLibrary(FCOM_DLL_NAME);
    if (FCOM_FryersLibHandle == NULL) {
      FCOM_ProcedureNumber = 0;
      FCOM_FryersLoaded = false;
      return(FCOM_FILE_NOT_FOUND);
    }
  }
  return(FCOM_SUCCESS);
};

//The following routine check of the procedure addresses have been loaded
//from the DLL. If not, they are loaded. If not found, an error is returned.
//This method means that only those procedures use in the DLL are checked.
//So if it doesn't exist and we don't use it, nobody gets upset.
//The downside is that if the procedure is not there and we do use it,
//an error occurs at the time of use. This could potentially be in the
//middle of doing something causing an unexpected error.

//check if we found the procedure we want
int fcom_CallFryersLoaded(void) {
  int Result = FCOM_DllLoaded();
  if (Result != FCOM_SUCCESS) return(Result);
  if (fcom_CallFryersProc == NULL) {
    fcom_CallFryersProc = (tCallFryers)GetProcAddress(FCOM_FryersLibHandle, "CallFryers");
    if (fcom_CallFryersProc == NULL) {
      FCOM_ProcedureNumber = 1;
      return(FCOM_ENTRY_NOT_FOUND);
    }
  }//endif(CallFryers==NULL)
  return(FCOM_SUCCESS);
};

int fcom_OpenPortLoaded(void) {
  int Result = FCOM_DllLoaded();
  if (Result != FCOM_SUCCESS) return(Result);
  if (fcom_OpenPortProc == NULL) {
    fcom_OpenPortProc = (tOpenPort)GetProcAddress(FCOM_FryersLibHandle, "OpenPort");
    if (fcom_OpenPortProc == NULL) {
      FCOM_ProcedureNumber = 3;
      return(FCOM_ENTRY_NOT_FOUND);
    }
  }//endif(OpenPortProc==NULL)
  return(FCOM_SUCCESS);
};

int fcom_ClosePortLoaded(void) {
  int Result = FCOM_DllLoaded();
  if (Result != FCOM_SUCCESS) return(Result);
  if (fcom_ClosePortProc == NULL) {
    fcom_ClosePortProc = (tClosePort)GetProcAddress(FCOM_FryersLibHandle, "ClosePort");
    if (fcom_ClosePortProc == NULL) {
      FCOM_ProcedureNumber = 4;
      return(FCOM_ENTRY_NOT_FOUND);
    }
  }//endif(ClosePortProc==NULL)
  return(FCOM_SUCCESS);
};

int fcom_GetPortStatusLoaded(void) {
  int Result = FCOM_DllLoaded();
  if (Result != FCOM_SUCCESS) return(Result);
  if (fcom_GetPortStatusProc == NULL) {
    fcom_GetPortStatusProc = (tGetPortStatus)GetProcAddress(FCOM_FryersLibHandle, "GetPortStatus");
    if (fcom_GetPortStatusProc == NULL) {
      FCOM_ProcedureNumber = 5;
      return(FCOM_ENTRY_NOT_FOUND);
    }
  }//endif(GetPortStatusProc==NULL)
  return(FCOM_SUCCESS);
};

int fcom_SendReadyLoaded(void) {
  int Result = FCOM_DllLoaded();
  if (Result != FCOM_SUCCESS) return(Result);
  if (fcom_SendReadyProc == NULL) {
    fcom_SendReadyProc = (tSendReady)GetProcAddress(FCOM_FryersLibHandle, "SendReady");
    if (fcom_SendReadyProc == NULL) {
      FCOM_ProcedureNumber = 6;
      return(FCOM_ENTRY_NOT_FOUND);
    }
  }//endif(SendReadyProc==NULL)
  return(FCOM_SUCCESS);
};

int fcom_SendCmdArrayLoaded(void) {
  int Result = FCOM_DllLoaded();
  if (Result != FCOM_SUCCESS) return(Result);
  if (fcom_SendCmdArrayProc == NULL) {
    fcom_SendCmdArrayProc = (tSendCmdArray)GetProcAddress(FCOM_FryersLibHandle, "SendCmdArray");
    if (fcom_SendCmdArrayProc == NULL) {
      FCOM_ProcedureNumber = 7;
      return(FCOM_ENTRY_NOT_FOUND);
    }
  }//endif(SendCmdArrayProc==NULL)
  return(FCOM_SUCCESS);
};

int fcom_RspReadyLoaded(void) {
  int Result = FCOM_DllLoaded();
  if (Result != FCOM_SUCCESS) return(Result);
  if (fcom_RspReadyProc == NULL) {
    fcom_RspReadyProc = (tRspReady)GetProcAddress(FCOM_FryersLibHandle, "RspReady");
    if (fcom_RspReadyProc == NULL) {
      FCOM_ProcedureNumber = 8;
      return(FCOM_ENTRY_NOT_FOUND);
    }
  }//endif(RspReadyProc==NULL)
  return(FCOM_SUCCESS);
};

int fcom_GetRspArrayLoaded(void) {
  int Result = FCOM_DllLoaded();
  if (Result != FCOM_SUCCESS) return(Result);
  if (fcom_GetRspArrayProc == NULL) {
    fcom_GetRspArrayProc = (tGetRspArray)GetProcAddress(FCOM_FryersLibHandle, "GetRspArray");
    if (fcom_GetRspArrayProc == NULL) {
      FCOM_ProcedureNumber = 9;
      return(FCOM_ENTRY_NOT_FOUND);
    }
  }//endif(GetRspArrayProc==NULL)
  return(FCOM_SUCCESS);
};

int fcom_SetNoPollTimeoutLoaded(void) {
  int Result = FCOM_DllLoaded();
  if (Result != FCOM_SUCCESS) return(Result);
  if (fcom_SetNoPollTimeoutProc == NULL) {
    fcom_SetNoPollTimeoutProc = (tSetNoPollTimeout)GetProcAddress(FCOM_FryersLibHandle, "SetNoPollTimeout");
    if (fcom_SetNoPollTimeoutProc == NULL) {
      FCOM_ProcedureNumber = 10;
      return(FCOM_ENTRY_NOT_FOUND);
    }
  }//endif(SetNoPollTimeoutProc==NULL)
  return(FCOM_SUCCESS);
};

int fcom_GetNoPollTimeoutLoaded(void) {
  int Result = FCOM_DllLoaded();
  if (Result != FCOM_SUCCESS) return(Result);
  if (fcom_GetNoPollTimeoutProc == NULL) {
    fcom_GetNoPollTimeoutProc = (tGetNoPollTimeout)GetProcAddress(FCOM_FryersLibHandle, "GetNoPollTimeout");
    if (fcom_GetNoPollTimeoutProc == NULL) {
      FCOM_ProcedureNumber = 11;
      return(FCOM_ENTRY_NOT_FOUND);
    }
  }//endif(GetNoPollTimeoutProc==NULL)
  return(FCOM_SUCCESS);
};

int fcom_GetFcomVersionLoaded(void) {
  int Result = FCOM_DllLoaded();
  if (Result != FCOM_SUCCESS) return(Result);
  if (fcom_GetFcomVersionProc == NULL) {
    fcom_GetFcomVersionProc = (tGetFcomVersion)GetProcAddress(FCOM_FryersLibHandle, "GetFcomVersion");
    if (fcom_GetFcomVersionProc == NULL) {
      FCOM_ProcedureNumber = 12;
      return(FCOM_ENTRY_NOT_FOUND);
    }
  }//endif(GetFcomVersionProc==NULL)
  return(FCOM_SUCCESS);
};

int fcom_MaxComPortLoaded(void) {
  int Result = FCOM_DllLoaded();
  if (Result != FCOM_SUCCESS) return(Result);
  if (fcom_MaxComPortProc == NULL) {
    fcom_MaxComPortProc = (tMaxComPort)GetProcAddress(FCOM_FryersLibHandle, "MaxComPort");
    if (fcom_MaxComPortProc == NULL) {
      FCOM_ProcedureNumber = 13;
      return(FCOM_ENTRY_NOT_FOUND);
    }
  }//endif(MaxComPortProc==NULL)
  return(FCOM_SUCCESS);
};


int fcom_GetDebugStatusLoaded(void) {
  int Result = FCOM_DllLoaded();
  if (Result != FCOM_SUCCESS) return(Result);
  if (fcom_GetDebugStatusProc == NULL) {
    fcom_GetDebugStatusProc = (tGetDebugStatus)GetProcAddress(FCOM_FryersLibHandle, "GetDebugStatus");
    if (fcom_GetDebugStatusProc == NULL) {
      FCOM_ProcedureNumber = 14;
      return(FCOM_ENTRY_NOT_FOUND);
    }
  }//endif(GetDebugStatusProc==NULL)
  return(FCOM_SUCCESS);
};

int fcom_SetPortModeLoaded(void) {
  int Result = FCOM_DllLoaded();
  if (Result != FCOM_SUCCESS) return(Result);
  if (fcom_SetPortModeProc == NULL) {
    fcom_SetPortModeProc = (tSetPortMode)GetProcAddress(FCOM_FryersLibHandle, "SetPortMode");
    if (fcom_SetPortModeProc == NULL) {
      FCOM_ProcedureNumber = 15;
      return(FCOM_ENTRY_NOT_FOUND);
    }
  }//endif(SetPortModeProc==NULL)
  return(FCOM_SUCCESS);
};

int fcom_GetPortModeLoaded(void) {
  int Result = FCOM_DllLoaded();
  if (Result != FCOM_SUCCESS) return(Result);
  if (fcom_GetPortModeProc == NULL) {
    fcom_GetPortModeProc = (tGetPortMode)GetProcAddress(FCOM_FryersLibHandle, "GetPortMode");
    if (fcom_GetPortModeProc == NULL) {
      FCOM_ProcedureNumber = 16;
      return(FCOM_ENTRY_NOT_FOUND);
    }
  }//endif(GetPortModeProc==NULL)
  return(FCOM_SUCCESS);
};


//----------------------------------------------------
//This is a wrapper to make the DLL call safe. It also eliminates the
//need to manage the DLL in your program since it is automatically managed here
//If this is a first time caller, the Fryers.DLL is searched for on the normal
//search path for the computer. If found, the driver is loaded into memory and
//the CallFryers interface is looked for. If found, the procedure pointer is
//updated and the call to the Fryers driver is made. Note: The call to initialize
//the Fryers dll will take up to 200ms to perform. Also if you change the baudrate
//there may be a 200ms delay as Windows does it's thing (I have no control over that).
//If you are doing an auto-seek for the port, it may take even longer as the
//drive will have to trundle thtough all the ports until a valid Frye Com port is found.
//Returns FRYERS_FILE_NOT_FOUND if the DLL was not found on the search path.
//returns FRYERS_ENTRY_NOT_FOUND if the CallFryers procedure call was not found.
//Note: the Fryers.dll on the first access will load the Fryers32.dll which it needs
//to provide the communication interface to Fonix equipment.
int FCOM_CallFryers(F_RegsType* pIRegs) {
  int Result;
  Result = fcom_CallFryersLoaded();
  if (Result != FCOM_SUCCESS) return(Result);

  //CallFryers does not return anything other than in IRegs (all other calls return a result)
  fcom_CallFryersProc(pIRegs);
  return(FCOM_SUCCESS);
};


//---------------------------------------------------------------------------
//Open this com port for use with FryeCom
//ComPort is 0->99 (com1->com100)
int FCOM_OpenPort(int ComPort, int PortControl, int Baudrate, FCOM_TfcCallback Callback) {
  int Result;
  Result = fcom_OpenPortLoaded();
  if (Result != FCOM_SUCCESS) return(Result);

  Result = fcom_OpenPortProc(ComPort,PortControl,Baudrate,Callback);
  return(Result);
};


//---------------------------------------------------------------------------
//Close this com port used by FryeCom
int FCOM_ClosePort(int ComPort) {
  int Result;
  Result = fcom_ClosePortLoaded();
  if (Result != FCOM_SUCCESS) return(Result);

  Result = fcom_ClosePortProc(ComPort);
  return(Result);
};


//---------------------------------------------------------------------------
//get the current status of the Fryers com port
int FCOM_GetPortStatus(int ComPort, FCOM_tPortStatusRec* pPortStatus) {
  int Result;
  Result = fcom_GetPortStatusLoaded();
  if (Result != FCOM_SUCCESS) return(Result);

  Result = fcom_GetPortStatusProc(ComPort, pPortStatus);
  return(Result);
};


//-------------------------------------------------
//Check if FryeCom is ready to allow command to be sent to instrument
int FCOM_SendReady(int ComPort, FCOM_TfcCallback Callback) {
  int Result;
  Result = fcom_SendReadyLoaded();
  if (Result != FCOM_SUCCESS) return(Result);

  Result = fcom_SendReadyProc(ComPort,Callback);
  return(Result);
};


//-------------------------------------------------
//Send a command array to instrument via FryeCom
int FCOM_SendCmdArray(int ComPort, FCOM_TfcArray* pData) {
  int Result;
  Result = fcom_SendCmdArrayLoaded();
  if (Result != FCOM_SUCCESS) return(Result);

  Result = fcom_SendCmdArrayProc(ComPort, pData);
  return(Result);
};

//-------------------------------------------------
//Check if instrument has returned a response
//If IgnoreNoPoll != 0, ignores the no-poll flag
int FCOM_RspReady(int ComPort) {
  int Result;
  Result = fcom_RspReadyLoaded();
  if (Result != FCOM_SUCCESS) return(Result);

  Result = fcom_RspReadyProc(ComPort);
  return(Result);
};


//-------------------------------------------------
//Get the response array returned form the instrument
int FCOM_GetRspArray(int ComPort, FCOM_TfcArray* pData) {
  int Result;
  Result = fcom_GetRspArrayLoaded();
  if (Result != FCOM_SUCCESS) return(Result);

  Result = fcom_GetRspArrayProc(ComPort, pData);
  return(Result);
};

//-------------------------------------------------
//set a new Poll timeout value
int FCOM_SetNoPollTimeout(int ComPort, int Value) {
  int Result;
  Result = fcom_SetNoPollTimeoutLoaded();
  if (Result != FCOM_SUCCESS) return(Result);

  Result = fcom_SetNoPollTimeoutProc(ComPort, Value);
  return(Result);
};

//-------------------------------------------------
//get the current Poll timeout value
int FCOM_GetNoPollTimeout(int ComPort, int* pValue) {
  int Result;
  Result = fcom_GetNoPollTimeoutLoaded();
  if (Result != FCOM_SUCCESS) return(Result);

  Result = fcom_GetNoPollTimeoutProc(ComPort, pValue);
  return(Result);
};

//-------------------------------------------------
//get the version number of the FryeCom Driver
int FCOM_GetFcomVersion(int* pFcomVersion) {
  int Result;
  Result = fcom_GetFcomVersionLoaded();
  if (Result != FCOM_SUCCESS) return(Result);

  Result = fcom_GetFcomVersionProc(pFcomVersion);
  return(Result);
};

//-------------------------------------------------
//get the maximum possible comport for the FruyeCom Driver
int FCOM_MaxComPort(int* pMaxPort) {
  int Result;
  Result = fcom_MaxComPortLoaded();
  if (Result != FCOM_SUCCESS) return(Result);

  Result = fcom_MaxComPortProc(pMaxPort);
  return(Result);
};

//---------------------------------------------------------------------------
//get the debug status of the Fryers com port
int FCOM_GetDebugStatusProc(int ComPort, FCOM_tDebugStatusRec* pDebugStatus) {
  int Result;
  Result = fcom_GetDebugStatusLoaded();
  if (Result != FCOM_SUCCESS) return(Result);

  Result = fcom_GetDebugStatusProc(ComPort, pDebugStatus);
  return(Result);
};

//-------------------------------------------------
//set a new Port Mode value
int FCOM_SetPortMode(int ComPort, int Mode) {
  int Result;
  Result = fcom_SetPortModeLoaded();
  if (Result != FCOM_SUCCESS) return(Result);

  Result = fcom_SetPortModeProc(ComPort, Mode);
  return(Result);
};

//-------------------------------------------------
//get the current port mode value
int FCOM_GetPortMode(int ComPort, int* pMode) {
  int Result;
  Result = fcom_GetPortModeLoaded();
  if (Result != FCOM_SUCCESS) return(Result);

  Result = fcom_GetPortModeProc(ComPort, pMode);
  return(Result);
};


//<eof>


