
//----------------------------------------------------
//FryeCom Dynamic DLL interface unit
//Copyright 2010 Frye Electronics, Inc.
//Written by Michael Day as of 30 Aug 2010
//You may use this code for any lawful purpose without fees or royalties.
//This code is intended as an example showing how to communicate with Fonix
//equipment using the FryeCom.DLL device driver on the Windows operating system.
//No warranties are express or implied in the use of this code.
//This will compile under any version of Borland Delphi (written using Delphi V3.00)
//----------------------------------------------------

unit FryeComD;

interface

uses Windows,   //used to get the DLL calling API from Windows
     FryeDefs,  //Common Frye definitions
     Fryers,    //Common Fryers DLL defs
     FcomDefs;  //FryeCom definitions


type HINSTANCE = integer;

//--------------------------------
//These three items are not normally used, they are available for debugging.
var FCOM_ProcedureNumber:integer = -1;    //used for debugging to indicate which call failed
var FCOM_FryersLibHandle : HINSTANCE = 0; //Windows handle for the DLL
var FCOM_FryersLoaded:boolean = false;    //true=dll was loaded, false=not yet loaded
const FCOM_DLL_NAME : PChar = 'fryecom.dll';

//--------------------------------------------------------------------------
function FCOM_DllLoaded:integer;  //export 1
function FCOM_CallFryers(var IRegs:F_RegsType):integer; //export 2

function FCOM_OpenPort(ComPort:integer; PortControl:integer; StartBaud:integer; Callback:FC_tCallback):integer; //export 3
function FCOM_ClosePort(ComPort:integer):integer;  //export 4
function FCOM_GetPortStatus(ComPort:integer; var PortStatus:FC_tPortStatusRec):integer; //export 5

function FCOM_SendReady(ComPort:integer; Callback:FC_tCallback):integer;  //export 6
function FCOM_SendCmdArray(ComPort:integer; var Data:FC_tCmdArray):integer; //export 7
function FCOM_RspReady(ComPort:integer):integer;  //export 8
function FCOM_GetRspArray(ComPort:integer; var Data:FC_tCmdArray):integer; //export 9

function FCOM_SetNoPollTimeout(ComPort:integer; Value:integer):integer; //export 10
function FCOM_GetNoPollTimeout(ComPort:integer; var Value:integer):integer; //export 11
function FCOM_GetFcomVersion(var FcomVersion:integer):integer; //export 12
function FCOM_MaxComPort(var MaxPort:integer):integer; //export 13
function FCOM_GetDebugStatus(ComPort:integer; var DebugStatus:FC_tDebugStatusRec):integer; //export 14
function FCOM_SetPortMode(ComPort:integer; NewPortMode:integer):integer; //export 15
function FCOM_GetPortMode(ComPort:integer; var CurrentPortMode:integer):integer; //export 16

//--------------------------------

implementation


//-----------------------------------------------------------------------
//This method uses the dynamic DLL loading convention.
//If you use a static load (load the dll at application startup),
//Windows will refuse to start the application if the DLL is not found.
//Doing it this way allows the application to notify the user
//about the problem and/or ignore it if you don't need the DLL.
//You can call FCOM_DllLoaded() to attempt to pre-load the DLL.
//If the DLL load failed, the error indicating the cause is returned.
//If you don't preload the DLL, then it will be loaded the first time
//a call to the DLL is made.
//If you need direct access to the DLL, you can use the FCOM_FryersLibHandle.
//as the Windows reference handle (it is created at the time the DLL is loaded.)
//-----------------------------------------------------------------------
//exports
//  CallFryers       index 1,  //call thru to Fryers32
//  CallFryers_C     index 2,  //call thru to Fryers32 (not supported here)
//
//  OpenPort         index 3,  //Open the selected port
//  ClosePort        index 4,  //close the current port
//  GetPortStatus    index 5,  //get operational status of the port
//
//  SendReady        index 6,  //check if Ready to send
//  SendCmdArray     index 7,  //send a command array
//  RspReady         index 8,  //check if a response is available
//  GetRspArray      index 9,  //get a response array
//
//  SetNoPollTimeout index 10, //set new no_poll timeout value
//  GetNoPollTimeout index 11, //get current no_poll timeout value
//  GetFcomVersion   index 12, //*get FryeCom Driver version number
//  MaxComPort       index 13, //*get max possible available com port
//  GetDebugStatus   index 14, //get extended debug port status
//  SetPortMode      index 15, //set communication mode driver is to use
//  GetPortMode      index 16, //get communication mode driver is using
//
//--> *Note: GetFcomVersion and MaxComPort do not require a ComPort handle.

//=============================================================================
//Direct DLL interface api - Normally you would use the higher level calls here
//--------------------------------

//Old style Fryer call format
var fcdll_CallFryers : function (var FIregs:F_RegsType):integer; stdcall;

//Open, close Fryers ports and Status info
var fcdll_OpenPort : function(ComPort:integer; PortControl:integer; StartBaud:integer; Callback:FC_tCallback):integer; stdcall;  //index 3
var fcdll_ClosePort : function(ComPort:integer):integer; stdcall; //index 4
var fcdll_GetPortStatus : function(ComPort:integer; var PortStatus:FC_tPortStatusRec):integer; stdcall; //index 5

//Fryers command transfers
var fcdll_SendReady : function(ComPort:integer; Callback:FC_tCallback):integer; stdcall; //index 6
var fcdll_SendCmdArray : function(ComPort:integer; var Data:FC_tCmdArray):integer; stdcall; //index 7
var fcdll_RspReady : function(ComPort:integer):integer; stdcall;  //index 8
var fcdll_GetRspArray : function(ComPort:integer; var Data:FC_tCmdArray):integer;  stdcall;  //index 9

//Special maintance commands.
var fcdll_SetNoPollTimeout : function(ComPort:integer; Value:integer):integer; stdcall; //index 10
var fcdll_GetNoPollTimeout : function(ComPort:integer; var Value:integer):integer;  stdcall; //index 11
var fcdll_GetFcomVersion : function(var FcomVersion:integer):integer; stdcall; //inndex 12
var fcdll_MaxComPort : function(var MaxPort:integer):integer; stdcall;  //index 13
var fcdll_GetDebugStatus : function(ComPort:integer; var DebugStatus:FC_tDebugStatusRec):integer; stdcall;  //index 14
var fcdll_SetPortMode : function(ComPort:integer; NewPortMode:integer):integer; stdcall; //export 15
var fcdll_GetPortMode : function(ComPort:integer; var CurrentPortMode:integer):integer; stdcall; //export 16

//-----------------------------------------------------------------------

//Initialize the DLL procedures
procedure fcom_InitDllProcs;
begin
  fcdll_CallFryers := NIL;
  fcdll_OpenPort := NIL;
  fcdll_ClosePort := NIL;
  fcdll_GetPortStatus := NIL;
  fcdll_SendReady := NIL;
  fcdll_SendCmdArray := NIL;
  fcdll_RspReady := NIL;
  fcdll_GetRspArray := NIL;
  fcdll_SetNoPollTimeout := NIL;
  fcdll_GetNoPollTimeout := NIL;
  fcdll_GetFcomVersion := NIL;
  fcdll_MaxComPort := NIL;
  fcdll_GetDebugStatus := NIL;
  fcdll_SetPortMode := NIL;
  fcdll_GetPortMode := NIL;
end;

//Check if the DLL has been loaded yet.
//If not, try to load it.
//returns error code if failed.
function FCOM_DllLoaded:integer;
begin
  FCOM_FryersLoaded := true;
  if (FCOM_FryersLibHandle = 0) then
  begin
    fcom_InitDllProcs();
    FCOM_FryersLibHandle := LoadLibrary(FCOM_DLL_NAME);
    if (FCOM_FryersLibHandle = 0) then
    begin
      FCOM_ProcedureNumber := 0;
      //FD_ErrorFilename := FCOM_DLL_NAME; //,1,255,);
      FCOM_FryersLoaded := false;
      Result := FCOM_FILE_NOT_FOUND;
      Exit;
    end;
  end;
  FD_WhichDLL := 2;
  Result := FCOM_SUCCESS;
end;

//The following routine check of the procedure addresses have been loaded
//from the DLL. If not, they are loaded. If not found, an error is returned.
//This method means that only those procedures use in the DLL are checked.
//So if it doesn't exist and we don't use it, nobody gets upset.
//The downside is that if the procedure is not there and we do use it,
//an error occurs at the time of use. This could potentially be in the
//middle of doing something causing an unexpected error.

//check if we found the procedure we want
function fcom_CallFryersLoaded:integer;
begin
  Result := FCOM_DllLoaded();
  if (Result <> FCOM_SUCCESS) then Exit;

  if (@fcdll_CallFryers = nil) then
  begin
    {We are only going to use the old CallFryers interface to allow}
    {compatibility to the old Fryers32 dll}
    @fcdll_CallFryers := GetProcAddress(FCOM_FryersLibHandle,'CallFryers');
    if (@fcdll_CallFryers = nil) then
    begin
      FCOM_ProcedureNumber := 1;
      {showmessage('Failed to lookup CallFryers');}
//      ErrorCode := NoFryers;
      Result := FCOM_ENTRY_NOT_FOUND;
      {halt(ErrorCode);}
      Exit;
    end;
  end;
  Result := FCOM_SUCCESS;
end;

function fcom_OpenPortLoaded:integer;
begin
  Result := FCOM_DllLoaded();
  if (Result <> FCOM_SUCCESS) then Exit;

  if (@fcdll_OpenPort = NIL) then
  begin
    @fcdll_OpenPort := GetProcAddress(FCOM_FryersLibHandle,'OpenPort');
    if (@fcdll_OpenPort = NIL) then
    begin
      FCOM_ProcedureNumber := 3;
      Result := FCOM_ENTRY_NOT_FOUND;
      Exit;
    end;
  end;//endif(OpenPortProc==NULL)
  Result := FCOM_SUCCESS;
end;

function fcom_ClosePortLoaded:integer;
begin
  Result := FCOM_DllLoaded();
  if (Result <> FCOM_SUCCESS) then Exit;

  if (@fcdll_ClosePort = NIL) then
  begin
    fcdll_ClosePort := GetProcAddress(FCOM_FryersLibHandle, 'ClosePort');
    if (@fcdll_ClosePort = NIL) then
    begin
      FCOM_ProcedureNumber := 4;
      Result := FCOM_ENTRY_NOT_FOUND;
      exit;
    end;
  end;//endif(ClosePortProc==NULL)
  Result := FCOM_SUCCESS;
end;

function fcom_GetPortStatusLoaded:integer;
begin
  Result := FCOM_DllLoaded();
  if (Result <> FCOM_SUCCESS) then Exit;

  if (@fcdll_GetPortStatus = NIL) then
  begin
    fcdll_GetPortStatus := GetProcAddress(FCOM_FryersLibHandle, 'GetPortStatus');
    if (@fcdll_GetPortStatus = NIL) then
    begin
      FCOM_ProcedureNumber := 5;
      Result := FCOM_ENTRY_NOT_FOUND;
      Exit;
    end;
  end;//endif(GetPortStatusProc==NULL)
  Result := FCOM_SUCCESS;
end;

function fcom_SendReadyLoaded:integer;
begin
  Result := FCOM_DllLoaded();
  if (Result <> FCOM_SUCCESS) then Exit;

  if (@fcdll_SendReady = NIL) then
  begin
    fcdll_SendReady := GetProcAddress(FCOM_FryersLibHandle, 'SendReady');
    if (@fcdll_SendReady = NIL) then
    begin
      FCOM_ProcedureNumber := 6;
      Result := FCOM_ENTRY_NOT_FOUND;
      Exit;
    end;
  end;//endif(SendReadyProc==NULL)
  Result := FCOM_SUCCESS;
end;

function fcom_SendCmdArrayLoaded:integer;
begin
  Result := FCOM_DllLoaded();
  if (Result <> FCOM_SUCCESS) then Exit;

  if (@fcdll_SendCmdArray = NIL) then
  begin
    fcdll_SendCmdArray := GetProcAddress(FCOM_FryersLibHandle, 'SendCmdArray');
    if (@fcdll_SendCmdArray = NIL) then
    begin
      FCOM_ProcedureNumber := 7;
      Result := FCOM_ENTRY_NOT_FOUND;
      Exit;
    end;
  end;//endif(SendCmdArrayProc==NULL)
  Result := FCOM_SUCCESS;
end;

function fcom_RspReadyLoaded:integer;
begin
  Result := FCOM_DllLoaded();
  if (Result <> FCOM_SUCCESS) then Exit;

  if (@fcdll_RspReady = NIL) then
  begin
    fcdll_RspReady := GetProcAddress(FCOM_FryersLibHandle, 'RspReady');
    if (@fcdll_RspReady = NIL) then
    begin
      FCOM_ProcedureNumber := 8;
      Result := FCOM_ENTRY_NOT_FOUND;
      Exit;
    end;
  end;//endif(RspReadyProc==NULL)
  Result := FCOM_SUCCESS;
end;

function fcom_GetRspArrayLoaded:integer;
begin
  Result := FCOM_DllLoaded();
  if (Result <> FCOM_SUCCESS) then Exit;

  if (@fcdll_GetRspArray = NIL) then
  begin
    fcdll_GetRspArray := GetProcAddress(FCOM_FryersLibHandle, 'GetRspArray');
    if (@fcdll_GetRspArray = NIL) then
    begin
      FCOM_ProcedureNumber := 9;
      Result := FCOM_ENTRY_NOT_FOUND;
      Exit;
    end;
  end;//endif(GetRspArrayProc==NULL)
  Result := FCOM_SUCCESS;
end;

function fcom_SetNoPollTimeoutLoaded:integer;
begin
  Result := FCOM_DllLoaded();
  if (Result <> FCOM_SUCCESS) then Exit;
  if (@fcdll_SetNoPollTimeout = NIL) then
  begin
    fcdll_SetNoPollTimeout := GetProcAddress(FCOM_FryersLibHandle, 'SetNoPollTimeout');
    if (@fcdll_SetNoPollTimeout = NIL) then
    begin
      FCOM_ProcedureNumber := 10;
      Result := FCOM_ENTRY_NOT_FOUND;
      Exit;
    end;
  end;//endif(SetNoPollTimeoutProc==NULL)
  Result := FCOM_SUCCESS;
end;

function fcom_GetNoPollTimeoutLoaded:integer;
begin
  Result := FCOM_DllLoaded();
  if (Result <> FCOM_SUCCESS) then Exit;

  if (@fcdll_GetNoPollTimeout = NIL) then
  begin
    fcdll_GetNoPollTimeout := GetProcAddress(FCOM_FryersLibHandle, 'GetNoPollTimeout');
    if (@fcdll_GetNoPollTimeout = NIL) then
    begin
      FCOM_ProcedureNumber := 11;
      Result := FCOM_ENTRY_NOT_FOUND;
      Exit;
    end;
  end;//endif(GetNoPollTimeoutProc==NULL)
  Result := FCOM_SUCCESS;
end;

function fcom_GetFcomVersionLoaded:integer;
begin
  Result := FCOM_DllLoaded();
  if (Result <> FCOM_SUCCESS) then Exit;

  if (@fcdll_GetFcomVersion = NIL) then
  begin
    fcdll_GetFcomVersion := GetProcAddress(FCOM_FryersLibHandle, 'GetFcomVersion');
    if (@fcdll_GetFcomVersion = NIL) then
    begin
      FCOM_ProcedureNumber := 12;
      Result := FCOM_ENTRY_NOT_FOUND;
      Exit;
    end;
  end;//endif(GetFcomVersionProc==NULL)
  Result := FCOM_SUCCESS;
end;

function fcom_MaxComPortLoaded:integer;
begin
  Result := FCOM_DllLoaded();
  if (Result <> FCOM_SUCCESS) then Exit;

  if (@fcdll_MaxComPort = NIL) then
  begin
    fcdll_MaxComPort := GetProcAddress(FCOM_FryersLibHandle, 'MaxComPort');
    if (@fcdll_MaxComPort = NIL) then
    begin
      FCOM_ProcedureNumber := 13;
      Result := FCOM_ENTRY_NOT_FOUND;
      Exit;
    end;
  end;//endif(MaxComPortProc==NULL)
  Result := FCOM_SUCCESS;
end;


function fcom_GetDebugStatusLoaded:integer;
begin
  Result := FCOM_DllLoaded();
  if (Result <> FCOM_SUCCESS) then Exit;

  if (@fcdll_GetDebugStatus = NIL) then
  begin
    fcdll_GetDebugStatus := GetProcAddress(FCOM_FryersLibHandle, 'GetDebugStatus');
    if (@fcdll_GetDebugStatus = NIL) then
    begin
      FCOM_ProcedureNumber := 14;
      Result := FCOM_ENTRY_NOT_FOUND;
      Exit;
    end;
  end;//endif(GetDebugStatusProc==NULL)
  Result := FCOM_SUCCESS;
end;

function fcom_SetPortModeLoaded:integer;
begin
  Result := FCOM_DllLoaded();
  if (Result <> FCOM_SUCCESS) then Exit;

  if (@fcdll_SetPortMode = NIL) then
  begin
    fcdll_SetPortMode := GetProcAddress(FCOM_FryersLibHandle, 'SetPortMode');
    if (@fcdll_SetPortMode = NIL) then
    begin
      FCOM_ProcedureNumber := 16;
      Result := FCOM_ENTRY_NOT_FOUND;
      Exit;
    end;
  end;//endif(SetPortMethod==NULL)
  Result := FCOM_SUCCESS;
end;

function fcom_GetPortModeLoaded:integer;
begin
  Result := FCOM_DllLoaded();
  if (Result <> FCOM_SUCCESS) then Exit;

  if (@fcdll_GetPortMode = NIL) then
  begin
    fcdll_GetPortMode := GetProcAddress(FCOM_FryersLibHandle, 'GetPortMode');
    if (@fcdll_GetPortMode = NIL) then
    begin
      FCOM_ProcedureNumber := 16;
      Result := FCOM_ENTRY_NOT_FOUND;
      Exit;
    end;
  end;//endif(GetPortMethod==NULL)
  Result := FCOM_SUCCESS;
end;


//----------------------------------------------------
//This is a wrapper to make the DLL call safe. It also eliminates the
//need to manage the DLL in your program since it is automatically managed here
//If this is a first time caller, the Fryers.DLL is searched for on the normal
//search path for the computer. If found, the driver is loaded into memory and
//the CallFryers interface is looked for. If found, the procedure pointer is
//updated and the call to the Fryers driver is made. Note: The call to initialize
//the Fryers dll will take up to 200ms to perform. Also if you change the baudrate
//there may be a 200ms delay as Windows does it's thing (I have no control over that).
//If you are doing an auto-seek for the port, it may take even longer as the
//drive will have to trundle thtough all the ports until a valid Frye Com port is found.
//Returns FRYERS_FILE_NOT_FOUND if the DLL was not found on the search path.
//returns FRYERS_ENTRY_NOT_FOUND if the CallFryers procedure call was not found.
//Note: the Fryers.dll on the first access will load the Fryers32.dll which it needs
//to provide the communication interface to Fonix equipment.
function FCOM_CallFryers(var IRegs : F_RegsType):integer;
begin
  Result := fcom_CallFryersLoaded();
  if (Result = FCOM_SUCCESS) then
  begin
    //CallFryers does not return anything other than in IRegs (all other calls return a result)
    fcdll_CallFryers(IRegs);
    Result := FCOM_SUCCESS;
  end;
end;


//---------------------------------------------------------------------------
//Open this com port for use with FryeCom
//ComPort is 0->99 (com1->com100)
function FCOM_OpenPort(ComPort:integer; PortControl:integer; StartBaud:integer; Callback:FC_tCallback):integer;
begin
  Result := fcom_OpenPortLoaded();
  if (Result = FCOM_SUCCESS) then
  begin
    Result := fcdll_OpenPort(ComPort,PortControl,StartBaud,Callback);
  end;
end;


//---------------------------------------------------------------------------
//Close this com port used by FryeCom
function FCOM_ClosePort(ComPort:integer):integer;
begin
  Result := fcom_ClosePortLoaded();
  if (Result = FCOM_SUCCESS) then
  begin
    Result := fcdll_ClosePort(ComPort);
  end;
end;


//---------------------------------------------------------------------------
//get the current status of the Fryers com port
function FCOM_GetPortStatus(ComPort:integer; var PortStatus:FC_tPortStatusRec):integer;
begin
  Result := fcom_GetPortStatusLoaded();
  if (Result = FCOM_SUCCESS) then
  begin
    Result := fcdll_GetPortStatus(ComPort, PortStatus);
  end;
end;


//-------------------------------------------------
//Check if FryeCom is ready to allow command to be sent to instrument
function FCOM_SendReady(ComPort:integer; Callback:FC_tCallback):integer;
begin
  Result := fcom_SendReadyLoaded();
  if (Result = FCOM_SUCCESS) then
  begin
    Result := fcdll_SendReady(ComPort,Callback);
  end;
end;


//-------------------------------------------------
//Send a command array to instrument via FryeCom
function FCOM_SendCmdArray(ComPort:integer; var Data:FC_tCmdArray):integer;
begin
  Result := fcom_SendCmdArrayLoaded();
  if (Result = FCOM_SUCCESS) then
  begin
    Result := fcdll_SendCmdArray(ComPort, Data);
  end;
end;

//-------------------------------------------------
//Check if instrument has returned a response
function FCOM_RspReady(ComPort:integer):integer;
begin
  Result := fcom_RspReadyLoaded();
  if (Result = FCOM_SUCCESS) then
  begin
    Result := fcdll_RspReady(ComPort);
  end;
end;


//-------------------------------------------------
//Get the response array returned form the instrument
function FCOM_GetRspArray(ComPort:integer; var Data:FC_tCmdArray):integer;
begin
  Result := fcom_GetRspArrayLoaded();
  if (Result = FCOM_SUCCESS) then
  begin
    Result := fcdll_GetRspArray(ComPort, Data);
  end;
end;

//-------------------------------------------------
//set a new Poll timeout value
function FCOM_SetNoPollTimeout(ComPort:integer; Value:integer):integer;
begin
  Result := fcom_SetNoPollTimeoutLoaded();
  if (Result = FCOM_SUCCESS) then
  begin
    Result := fcdll_SetNoPollTimeout(ComPort, Value);
  end;
end;

//-------------------------------------------------
//get the current Poll timeout value
function FCOM_GetNoPollTimeout(ComPort:integer; var Value:integer):integer;
begin
  Result := fcom_GetNoPollTimeoutLoaded();
  if (Result = FCOM_SUCCESS) then
  begin
    Result := fcdll_GetNoPollTimeout(ComPort, Value);
  end;
end;

//-------------------------------------------------
//get the version number of the FryeCom Driver
function FCOM_GetFcomVersion(var FcomVersion:integer):integer;
begin
  Result := fcom_GetFcomVersionLoaded();
  if (Result = FCOM_SUCCESS) then
  begin
    Result := fcdll_GetFcomVersion(FcomVersion);
  end;
end;

//-------------------------------------------------
//get the maximum possible comport for the FryeCom Driver
//returns 0 in MaxPort if none available (driver not found)
//result is SUCCESS or error number
function FCOM_MaxComPort(var MaxPort:integer):integer;
begin
  Result := fcom_MaxComPortLoaded();
  if (Result = FCOM_SUCCESS) then
  begin
    Result := fcdll_MaxComPort(MaxPort);
  end;
  if (Result <> SUCCESS) then MaxPort := 0;
end;

//---------------------------------------------------------------------------
//get the debug status of the Fryers com port
function FCOM_GetDebugStatus(ComPort:integer; var DebugStatus:FC_tDebugStatusRec):integer;
begin
  Result := fcom_GetDebugStatusLoaded();
  if (Result = FCOM_SUCCESS) then
  begin
    Result := fcdll_GetDebugStatus(ComPort, DebugStatus);
  end;
end;

//-------------------------------------------------
//set the port mode for the FryeCom Driver
function FCOM_SetPortMode(ComPort:integer; NewPortMode:integer):integer;
begin
  Result := fcom_SetPortModeLoaded();
  if (Result = FCOM_SUCCESS) then
  begin
    Result := fcdll_SetPortMode(ComPort, NewPortMode);
  end;
end;

//---------------------------------------------------------------------------
//get the port mode for the FryeCom Driver
function FCOM_GetPortMode(ComPort:integer; var CurrentPortMode:integer):integer;
begin
  Result := fcom_GetPortModeLoaded();
  if (Result = FCOM_SUCCESS) then
  begin
    Result := fcdll_GetPortMode(ComPort, CurrentPortMode);
  end;
end;


end.
