
//----------------------------------------------------
//FryeStr special strings interface unit
//Copyright 2010 Frye Electronics, Inc.
//Written by Michael Day as of 30 Aug 2010
//You may use this code for any lawful purpose without fees or royalties.
//No warranties are express or implied in the use of this code.
//----------------------------------------------------

unit FryeStr;

interface
uses Windows,SysUtils,FryeDefs;

type Dval=packed Record  Lw, Hw:word; End;
type Lval=packed Record b0,b1,b2,b3:byte; End;

//---------------------------------------------------------------------------
const FS_spc : char = ' ';                 //space char
const FS_Bspc : BYTE = BYTE(' ');          //byte declared space
const FS_sBLANK:str255 = sEND;      //blank string def (a string with just a zero in it)
const FS_sBlankMsg:str255  = '      '; //no message
const FS_sFailed:str255 = 'FAILED ';
const FS_sUnknown:str255 = 'UNKNOWN ';
const FS_sNoMic:str255  = 'NO MICROPHONE';
const FS_sCalibrationCable:str255 = 'CALIBRATION';
const FS_cHex : array[0..15] of char = '0123456789ABCDEF';

var FS_SpinCounter : integer = 0;
function FS_Spin(Count:integer):char;
function FS_SpinIt(var Count:integer):char;
function FS_Spinner:char;

function FS_BlankLabelLine(LabelLine:pchar):boolean;
function FS_IntToPChar(Num:integer; pS:pchar):pchar;
function FS_pCharLen(pStr:pchar):integer;
function FS_pCharCopy(pDest:pchar; pSrc:pchar):pchar;
function FS_pCharCat(pDest:pchar; pSrc:pchar):pchar;
function FS_pCharCopy3(pDest:pchar; pStr1:pchar; pStr2:pchar; pStr3:pchar):pchar;
function FS_pCharCat3(pDest:pchar; pStr1:pchar; pStr2:pchar; pStr3:pchar):pchar;
function FS_StringToPChar(pDest:pchar; Src:string; MaxSize:integer):pchar;
function FS_PCharToString(pStr:pchar):string;

function FS_ByteToHex(Value:integer):string;
function FS_WordToHex(Value:integer):string;
function FS_DWordToHex(Value:DWORD):string;
function FS_HexToDecimal(HexDigit:char):integer;
function FS_HexToNumber(HexStr:string):extended;
function FS_HexToValue(Size:WORD; HexStr:string; var Value:DWORD; DefaultValue:DWORD):integer;
function FS_HexToByte(HexStr:string; var ByteValue:BYTE; DefaultValue:BYTE):integer;
function FS_HexToWord(HexStr:string; var Value:WORD; DefaultValue:WORD):integer;
function FS_HexToDword(HexStr:string; var Value:DWORD; DefaultValue:DWORD):integer;
function FS_HexToBinary(Count:integer; HexStr:string; var Data; DefaultValue:BYTE):integer;
function FS_HexToByteArray(Count:integer; HexStr:string; var Data; DefaultValue:BYTE):integer;
function FS_BinaryToHex(Size:integer; var Data; var HexStr:string):integer;


procedure FS_TrimLeadingSpaces(var Str:string);
procedure FS_TrimLeadingZeros(var Str:string);
function FS_TrimTrailingSpaces(Str:string):string;
function FS_Justify(var sOutput:string; sInput:string; L:integer):string;

function FS_Znum(Num:integer; L:integer; var Str:string; sTag:string):string;
function FS_Inum(Num:integer; L:integer; var Str:string; sTag:string):string;
function FS_Rnum(Num:integer; D:integer; L:integer; var Str:string; sTag:string):string;
function FS_IntToStr(Num:integer):string;
function FS_StringToInt(Str:string; var Value:integer):boolean;
function FS_StringToInt16(Str:string; var Value:INT16):boolean;
function FS_TagFill(Str:string; FillChar:char; Size:integer):string;
function FS_ExtractString(Str:string; Loc:integer; Separator:char; var SubStr:string):integer;
function FS_SubStr(sSrc:string; var sDest:string; Start:integer; Count:integer):integer;
function FS_DateStr(var Date:FD_tCalDate; var Str:string; How:integer):string;
function FS_HexID(var Serial: array of BYTE; x:integer):string;
function FS_FileSizeStr(RawSize:DWORD):string;

function FS_UpperChar(C:char):char;
function FS_Upper(sSrc:string):string;
function FS_GetDeviceIDString(var DevID:FD_tDeviceID; TrimStr:boolean):string;
function FS_GetOptionsString(Options:DWORD; TrimStr:boolean):string;
function FS_GetVersionString(Version:integer; Decimal:integer):string;
function FS_GetFryeLanguageString(Language:integer; var Str:string):boolean;
function FS_GetFryeLanguageNumber(Str:string):integer;
function FS_GetVerLangString(Version:integer; Decimal:integer; Language:integer):string;


{=========================================}
{Pascal specific routines}
{=========================================}

function FS_HexB(B:byte):str255;
function FS_HexW(W:word):str255;
function FS_HexL(L:longint):str255;
function FS_IntStr(i:longint):str255;
function FS_LeftStr(i:integer; L:byte):str255;
function FS_RightStr(i:integer; L:byte):str255;
function FS_sTrim(S:str255):str255;
function FS_PackedFryeDateToStr(CalDate:longint; var Valid:boolean):str255;
function FS_UnpackedFryeDateToStr(CalDate:FD_tCalDate; var Valid:boolean):str255;
function FS_GetPackedFryeDateToStr(FryeDate:FD_tPackedCalDate):str255;
function FS_GetUnPackedFryeDateToStr(FryeDate:FD_tCalDate):str255;
function FS_ValidTimeStamp(Stamp:longint):boolean;
function FS_GetStampTimeStr(Stamp:longint; LongStr:boolean; var Valid:boolean):str255;
function FS_GetStampDateStr(Stamp:longint; var Valid:boolean):str255;
function FS_Str2D(value:integer):str255;
function FS_Str2Df(value:integer):str255;
function FS_Str3D(value:integer):str255;
function FS_DbStr100(Value:integer; ShowOff:boolean):str255;
function FS_DbStr(Value:integer; ShowOff:boolean):str255;
function FS_OnOff(value:integer):str255;
function FS_TemperatureStr(Temperature:integer):str255;
function FS_Deg64(value:integer):str255;

implementation



//---------------------------------------------------------------------------


//var fs_sDStr:array[0..999] of char;   //temp global string used to build date string
//var fs_sString:array[0..999] of char;

{-------------------------------------------------}
{core spin count converter}
function FS_Spin(Count:integer):char;
begin
  case Count and 3 of
   0:Result := '/';
   1:Result := '-';
   2:Result := '\';
   3:Result := '|';
  else Result := ' ';
  end;
end;
{calls spin with counter and incs the counter every other time}
function FS_SpinIt(var Count:integer):char;
begin
  Result := FS_Spin((Count shr 1) and 3);
  inc(FS_SpinCounter);
end;

{calls spinit with local counter - so caller doesn't have to manage it}
function FS_Spinner:char;
begin
  Result := FS_SpinIt(FS_SpinCounter);
end;

//{-----------------------------------------------------------------}
//Short version of IntToStr. No tags, no other calls, just convert the number.
function FS_IntToStr(Num:integer):string;
begin
  Result := IntToStr(Num);
end;

//---------------------------------------------------------------------------
//convert an integer to a pchar string
function FS_IntToPchar(Num:integer; pS:pchar):pchar;
var Str:Str255;
var i,L:integer;
begin
  Str := FS_IntToStr(Num);
  L := Length(Str);
  for i:=0 to L do
  begin
    pchar(pS)[i] := Str[i+1];
  end;
  pchar(pS)[L] := #0;
  Result := pchar(pS);
end;

//---------------------------------------------------------------------------
function FS_ByteToHex(Value:integer):string;
var HexStr:string;
begin
  SetLength(HexStr,2);
  HexStr[1] := FS_cHex[(Value shr 4) and $0f];
  HexStr[2] := FS_cHex[(Value shr 0) and $0f];
  Result := HexStr;
end;

//---------------------------------------------------------------------------
function FS_WordToHex(Value:integer):string;
var HexStr:string;
begin
  SetLength(Hexstr,4);
  HexStr[1] := FS_cHex[(Value shr 12) and $0f];
  HexStr[2] := FS_cHex[(Value shr 8) and $0f];
  HexStr[3] := FS_cHex[(Value shr 4) and $0f];
  HexStr[4] := FS_cHex[(Value shr 0) and $0f];
  Result := HexStr;
end;

//---------------------------------------------------------------------------
function FS_DWordToHex(Value:DWORD):string;
var temp1:string;
var temp2:string;
begin
  temp1 := FS_WordToHex(WORD((Value shr 16) and $0ffff)); //HexStr[0]);
  temp2 := FS_WordToHex((Value and $0ffff)); //HexStr[4]);
  Result := temp1 + temp2;
end;


//---------------------------------------------------------------------------
//returns -1 if bad hex char, or +n if ok.
function FS_HexToDecimal(HexDigit:char):integer;
var temp:integer;
begin
  if (HexDigit < '0') then
  begin
    Result := -1;
    Exit;
  end;
  if (HexDigit <= '9') then
  begin
    Result := ord(HexDigit) - ord('0');
    Exit;
  end;
  temp := ord(HexDigit) and $5f;
  if (temp < ord('A')) then
  begin
    Result := -1;
    Exit;
  end;
  if (temp <= ord('F')) then
  begin
    Result := temp - ord('A') + 10;
    Exit;
  end;
  Result := -1;
end;

//---------------------------------------------------------------------------
function FS_HexToNumber(HexStr:string):extended;
var i:integer;
var Number:integer;
var Error:integer;
var factor:extended;
var total:extended;
begin
  Error := 0;
  factor := 1.0;
  total := 0.0;
  for i:= length(HexStr) downto 1 do
  begin
    if (HexStr[i] >= '0') then
    begin
      Number := FS_HexToDecimal(HexStr[i]);
      if (Number >= 0) then
      begin
        total := total + (Number * factor);
        factor := factor * 16;
      end
      else
      begin
        Error := Error + 1;
      end;
    end;
  end;//endwhile
  if (Error > 0) then
    Result := -1.0
  else Result := total;
end;

//---------------------------------------------------------------------------
//generic hex to value conversion - used by HexToByte,HexToWord,HexToDword
//converts hex string in HexStr to a DWORD. If a bad hex value is given,
//that position is filled by the Default value.
//If not enough characters given, returns -1 and the value is zero.
//if bad hex char found, returns +n, and the determined number is returned.
//if all ok, returns a zero.
function FS_HexToValue(Size:WORD; HexStr:string; var Value:DWORD; DefaultValue:DWORD):integer;
var i:integer;
var Number:integer;
var factor:integer;
var total:integer;
var Error:integer;
begin
  factor := 1;
  total := 0;
  Error := 0;
  Value := 0;
  if (length(HexStr) < Size) then
  begin
    Result := -1;
    Exit;
  end;
  for i:=Size downto 1 do
  begin
    Number := FS_HexToDecimal(HexStr[i]);
    if (Number >= 0) then
    begin
      total := total + (Number * factor);
    end
    else
    begin
      total := DWORD(total) + DefaultValue;
      Error := Error+1;
    end;
    factor := factor * 16;
  end;//endfor
  Value := total;
  Result := Error;
end;

//---------------------------------------------------------------------------
//converts hex string in HexStr to a BYTE.
//returns zero result if successful
function FS_HexToByte(HexStr:string; var ByteValue:BYTE; DefaultValue:BYTE):integer;
var Number:DWORD;
var Error:integer;
begin
  Error := FS_HexToValue(2,HexStr,Number,DefaultValue);
  ByteValue := BYTE(Number);
  Result := Error;
end;

//---------------------------------------------------------------------------
//converts hex string in HexStr to a WORD.
//returns zero result if successful
function FS_HexToWord(HexStr:string; var Value:WORD; DefaultValue:WORD):integer;
var Number:DWORD;
var Error:integer;
begin
  Error := FS_HexToValue(4,HexStr,Number,DefaultValue);
  Value := WORD(Number);
  Result := Error;
end;

//---------------------------------------------------------------------------
//converts hex string in HexStr to a DWORD. If a bad hex value is given,
//that position is filled by the Default value.
//If not enough characters given, returns -1 and the value is zero.
//if bad hex char found, returns +n, and the determined number is returned.
//if all ok, returns a zero.
function FS_HexToDword(HexStr:string; var Value:DWORD; DefaultValue:DWORD):integer;
begin
  Result := FS_HexToValue(8,HexStr,Value,DefaultValue);
end;

//---------------------------------------------------------------------------
//converts hex string in HexStr to binary. If a bad hex value is given,
//that position is filled by the Default value.
//If not enough characters given, returns -1 and the data is not changed.
//Returns the number of errors encountered (0=all ok).
//Note: Count is the number of returned bytes, not the number of hex chars.
//If Count +n, the hex string length must match or be greater than the count
//If Count -n, the hex string result will be zero filled to the left if
//not enough data in HexStr to produce n bytes of data.
//If Count 0, no data is returned.
//Note: There must always be two hex nibbles per byte
//any odd count hex characters at the end (right) are discarded
function FS_HexToBinary(Count:integer; HexStr:string; var Data; DefaultValue:BYTE):integer;
var Number:BYTE;
var i:integer;
var HexIndex:integer;
var Error : integer;
type ptachar = ^tachar;
var pSC : ptachar; //^tachar;
begin
  Error := 0;
  if ((Count <= 0)or(@Data=NIL)) then
  begin
    Result := 0; //if count = 0 do nothing
    Exit;
  end;
  i := 0;
  while(Count > 0) do
  begin
    HexIndex := (Count-1)*2;
    pSC := ptachar(@HexStr[HexIndex]);
    Result := FS_HexToByte(pchar(pSC),Number,0);
    if (Result = 0) then
    begin
      tByteArray(Data)[i] := Number;
      i := i+1;
    end
    else
    begin
      tByteArray(Data)[i] := DefaultValue;
      i := i+1;
      Error := Error + 1;
    end;
    Count := Count -1;
  end;
  Result := Error;
end;

//---------------------------------------------------------------------------
//converts hex string in HexStr to a byte array. If a bad hex value is given,
//that position is filled by the Default value.
//If not enough characters given, returns -1 and the data is not changed.
//Returns the number of errors encountered (0=all ok).
//Note: Count is the number of returned bytes, not the number of hex chars.
//If Count 0, no data is returned.
//Note: There must always be two hex nibbles per byte
//any odd count hex characters at the end (right) are discarded

function FS_HexToByteArray(Count:integer; HexStr:string; var Data; DefaultValue:BYTE):integer;
var  Number:BYTE;
var  i:integer;
var  HexIndex:integer;
var  Error:integer;
var sTmp :string;
begin
  Error := 0;
  if ((Count <= 0)or(@Data=NIL)) then
  begin
    Result := 0; //if count = 0 do nothing
    Exit;
  end;
  i := 0;
  while(Count > 0) do
  begin
    HexIndex := i*2;
    sTmp := Copy(HexStr,HexIndex+1,2);
    Result := FS_HexToByte(sTmp,Number,0);
    if (Result = 0) then
    begin
      tByteArray(Data)[i] := Number;
      i := i + 1;
    end
    else
    begin
      tByteArray(Data)[i] := DefaultValue;
      i := i + 1;
      Error := Error + 1;
    end;
    Count := Count - 1;
  end;
  Result := Error;
end;

//---------------------------------------------------------------------------
//converts binary data of Size bytes to hex string in HexStr.
//If count = 0 returns -1, else returns 0
//Assumes that highest byte is to be placed on the left (lowest hex char)
function FS_BinaryToHex(Size:integer; var Data; var HexStr:string):integer;
var i:integer;
begin
  HexStr := FS_sBLANK;
  if (Size = 0) then
  begin
    Result := 0;
    Exit;
  end;
  if (Size < 0) then
  begin
    Result := -1;
    Exit;
  end;
  for i := Size-1 downto 0 do
  begin
    HexStr := HexStr + FS_cHex[(tByteArray(Data)[i] shr 4) and $0f];
    HexStr := HexStr + FS_cHex[tByteArray(Data)[i] and $0f];
  end;
  Result := 0;
end;

//---------------------------------------------------------------------------
//null protected string length
//Find a null byte in pchar string pointed at by pStr
//Returns index in array where the null was found
//Also returns zero if Data pointer is nil
function FS_pCharLen(pStr:pchar):integer;
var i : integer;
begin
  i := 0;
  if (pStr = NULL) then
  begin
    Result := 0;
    Exit;
  end;
  while(pStr[i] <> #0) do
  begin
    i := i+1;
  end;
  Result := i;
end;

//---------------------------------------------------------------------------
//Converts a pchar string to a Pascal string;
function FS_PCharToString(pStr:pchar):string;
begin
  Result := pStr;
end;


//---------------------------------------------------------------------------
//Converts a Pascal string to a pchar string
//MaxSize specifies the maximum possible number of characters that can be copied to pDest}
function FS_StringToPChar(pDest:pchar; Src:string; MaxSize:integer):pchar;
begin
  Result := StrPLCopy(pDest,Src,MaxSize);
end;


//---------------------------------------------------------------------------
//null protected pchar strcpy
//copy src to dest until null encountered.
//if either pointer is null, do nothing.
//Assumes enough space in dest is avail for copy.
//Returns destination pointer.
function FS_pCharCopy(pDest:pchar; pSrc:pchar):pchar;
begin
  Result := StrCopy(pchar(pDest),pSrc);
end;

//---------------------------------------------------------------------------
//null protected pchar strcat
//concatinate src to end of dest string until null in src encountered.
//if either pointer is null, do nothing. Assumes that enough space for
//copy to dest is avail. Returns destination pointer.
function FS_pCharCat(pDest:pchar; pSrc:pchar):pchar;
begin
  Result := StrCat(pchar(pDest),pSrc);
end;

//---------------------------------------------------------------------------
//copies three pchar strings ito a desitination string.
//returns a pointer to the destination string.
//If a string is null, it is not included.
//Note: Make sure Dest is big enough to hold all the strings
function FS_pCharCopy3(pDest:pchar; pStr1:pchar; pStr2:pchar; pStr3:pchar):pchar;
begin
  FS_pCharCopy(pDest,pStr1);
  FS_pCharCat(pDest,pStr2);
  FS_pCharCat(pDest,pStr3);
  Result := pDest;
end;

//---------------------------------------------------------------------------
//List StrCopy3 above, but concatinates to the end of an existing string
function FS_pCharCat3(pDest:pchar; pStr1:pchar; pStr2:pchar; pStr3:pchar):pchar;
begin
  FS_pCharCat(pDest,pStr1);
  FS_PcharCat(pDest,pStr2);
  FS_pCharCat(pDest,pStr3);
  Result := pDest;
end;

//{-----------------------------------------------------------------}
// Return a rounded integer "Num" as a fixed point number string.
// D specifies how many decimal points to return 0=none, 1=1.1, 2=2.22}
// If text is provided in S, the number will be added to the end of the string
// String is returned in S. The number in the string is right justified (left spaces)
// L specifies string S total length. If L<0, string is left justified (right spaces)
// If L=0 the exact length string is returned.
// Number passed is presumed to be Fn*100 (100x fixed point decimal number)
// Note: if the resulting string length is greater than L, the returned
// string will be longer than L. Make sure enough space has been allocated.
// If a Tag string is provided, it will be added to the end of the result.
function FS_Rnum(Num:integer; D:integer; L:integer; var Str:string; sTag:string):string;
var j,r:integer;
var n:div_t;
var sTmp:string;
//var sTmp2:string;
begin
  if (D < 0) then D := 0;
  if (D > 2) then D := 2;
  if (D=0) then
  begin
    //if D=0 we return only the integer portion rounded up from the fractional part
    n := FD_div(Num,100);
    if (n.rem <= -50) then
    begin
      n.quot := n.quot-1;
    end
    else if (n.rem >= 50) then
    begin
      n.quot := n.quot+1; //round it
    end;
    sTmp := FS_IntToStr(n.quot);  //if Num < 0 then v := Num+5-50 else v := Num+50;
  end
  else if (D=1) then
  begin
    //if D=1, we return one decimal position rounded up from the 100s position.
    n := FD_div(Num,10);   //pre-divide for rounding
    r := n.quot;
    if (n.rem <= -5) then
    begin
      r := r-1;
    end
    else if (n.rem >= 5) then
    begin
      r:=r+1; //round it
    end;
    n := FD_div(r,10);  //finish the divide
    if (r < 0) then
    begin
      //if negative number, add leading minus sign.
      sTmp := '-'+FS_IntToStr(abs(n.quot));
    end
    else
    begin
      //if positive number, just so a direct conversion
      sTmp := FS_IntToStr(n.quot);         //if i < 0 then v := i-5 else v := i+5;
    end;
    sTmp := sTmp + '.' + FS_IntToStr(abs(n.rem));  //get the fraction portion
  end
  else
  begin // D>=2
    //if D=2 or more, then we get two decimal places (no rounding needed)
    n := FD_div(Num,100);
    r := abs(n.rem);
    if (Num < 0) then
    begin
      sTmp := '-' + FS_IntToStr(abs(n.quot));
    end
    else
    begin
      sTmp := FS_IntToStr(n.quot); //positive number
    end;
    sTmp := sTmp + '.';   //add trailing decimal point
    if (r > 9) then
    begin
      //if 10 or more, we can do a direct ascii conversion
      sTmp := sTmp + FS_IntToStr(r);
    end
    else
    begin
      //if 9 or less, we add a 0 just behind the decimal point
      Stmp := sTmp+'0'+FS_IntToStr(r);
    end;
  end;

  if (L>0) then //right adj string if L=positive
  begin
    j := L - Length(sTmp);
    sTmp := StringOfChar(FS_spc,j) + sTmp;
  end

  else if (L<0) then  //Left adj string if L neg
  begin
    j := abs(L) - Length(sTmp);
    sTmp := sTmp + StringOfChar(FS_spc,j);
  end;
  //if a tag was provided, add it to the end of the string.
  Str := Str + sTmp + sTag;
  Result := Str;
end;

//{-----------------------------------------------------------------}
// Return an integer "Num" as a number string.
// If text is provided in S, the number will be added to the end of the string
// String is returned in S. The number in the string is right justified (left spaces)
// L specifies string S total length. If L<0, string is left justified (right spaces)
// If L=0 the exact length string is returned.
// Number passed is presumed to be a whole decimal number (N*1)
// Note: if the resulting string length is greater than L, the returned
// string will be longer than L. Make sure enough space has been allocated.
// If a Tag string is provided, it will be added to the end of the result.
function FS_Inum(Num:integer; L:integer; var Str:string; sTag:string):string;
var j:integer;
var sTmp:string;
begin
  sTmp := FS_IntToStr(Num);  //if Num < 0 then v := Num+5-50 else v := Num+50;
  //if L positive, right adjust the number in the string
  if (L>0) then
  begin
    j := L - Length(sTmp);
    sTmp := StringOfChar(FS_spc,j) + sTmp;
  end

  //if L negative, left adjust the number in the string
  else if (L<0) then
  begin
    j := abs(L) - Length(sTmp);
    sTmp := sTmp + StringOfChar(FS_spc,j);
  end;

  //if tag provided, add it to the end of the string
  Str := Str + sTmp + sTag;
  Result := Str;
end;

//{-----------------------------------------------------------------}
// Return an integer "Num" as a number string.
// String is returned in S. The number in the string is right justified (left spaces)
// L specifies string S total length. If L<0 zeros are padded in front.
// if L>0, spaces are padded in front. If L==0 nothing is added.
// Number passed is presumed to be a whole decimal number (N*1)
// Note: if the resulting string length is greater than L, the returned
// string will be longer than L. Make sure enough space has been allocated.
// **NOTE** If S is provided with a string, it will be added to the front
// of the result. If a Tag string is provided, it will be added to the
// end of the result. *CHANGE* The prestring feature is disabled due to
// too many problems with its use (garbage strings because they were not preinitialized
function FS_Znum(Num:integer; L:integer; var Str:string; sTag:string):string;
var j:integer;
var sTmp : string;
begin
  sTmp := FS_IntToStr(Num);  //if Num < 0 then v := Num+5-50 else v := Num+50;
  if (L>0) then
  begin
    j := L - Length(sTmp);
    sTmp := StringOfChar(FS_spc,j) + sTmp;
  end;
  if (L<0) then
  begin
    j := abs(L) - Length(sTmp);
    sTmp := sTmp + StringOfChar(FS_spc,j);
  end;
  Str := Str + sTmp + sTag;
  Result := Str;
end;

//------------------------------------------------------------------------------
//Converts a string to a decimal number. Leading spaces/ctrls are stripped
//Number is converted until no more numeric text found.
//If the number is too large, it will overflow.
//If nothing found, returns false, else returns true
function FS_StringToInt(Str:string; var Value:integer):boolean;
var i:integer;
var Size:integer;
var Valid:boolean;
begin
  Valid := false;
  Value := 0;
  Size := length(Str);
  for i:=1 to Size do
  begin
    if (Str[i] = #0) then break;
    if (Str[i] > '9') then break;
    if (Str[i] >= '0') then
    begin
      Value := (Value * 10) + (ord(Str[i]) and $0f);
      Valid := true;
    end;//endif(S)
  end;//endfor(i)
  Result := Valid;
end;

//------------------------------------------------------------------------------
//Just like StringToInt, but returns a 16 bit int.
function FS_StringToInt16(Str:string; var Value:INT16):boolean;
var Number:integer;
begin
  Result := FS_StringToInt(Str,Number);
  Value := INT16(Number);
end;

//------------------------------------------------------------------------------
// Return a justified string.
//   String is returned in sOutput.
//   L specifies string sOutput total length.
//   If L>0 the text in the string is right justified (spaces added to the left)
//   If L<0, string is left justified (spaces added to the right)
//   If L=0 the exact length string is returned.
//
// Note: if the resulting string length is greater than L, the returned
// string will be longer than L. Make sure enough space has been allocated.
//------------------------------------------------------------------------------
function FS_Justify(var sOutput:string; sInput:string; L:integer):string;
var  j:integer;
begin
  // Right justify string by adding spaces to left of string
  if (L>0) then
  begin
    j := L - Length(sInput);
    sOutput := StringOfChar(FS_spc,j) + sInput;
  end
  else if (L<0) then
  begin
    j := abs(L) - Length(sInput);
    sOutput := sInput + StringOfChar(FS_spc,j);
  end; //left adj?
  Result := sOutput;
end;

//---------------------------------------------------------------------------
//Trim leading zeros from a string
procedure FS_TrimLeadingZeros(var Str:string);
var i:integer;
var Size:integer;
begin
  i := 1;   //look for leading zeros
  Size := length(Str);
  while( (Str[i] = '0') and (i <= Size) ) do i := i + 1;
  Str := Copy(Str,i,Size);
end;

//---------------------------------------------------------------------------
//Trim leading spaces from a string (in-place strip)
procedure FS_TrimLeadingSpaces(var Str:string);
var i:integer;
var Size:integer;
begin
  i := 1;   //look for leading zeros
  Size := length(Str);
  while( (Str[i] = ' ') and (i <= Size) ) do i := i + 1;
  Str := Copy(Str,i,Size);
end;

//---------------------------------------------------------------------------
//trim trailing spaces and control chars from a string passed in Src
//Note: Dest array must be at least as big as Src array
function FS_TrimTrailingSpaces(Str:string):string;
var i : integer;
var sTmp:string;
begin
  i := length(Str);
  while((i > 0) and (Str[i] <= ' ')) do i := i - 1;
  if (i > 0) then
    Result := Copy(sTmp,1,i)
  else Result := FS_sBLANK;
end;

//---------------------------------------------------------------------------
//tag string with fill char to fill string to Size
//Note: Dest array must be at least as big as Src array plus count
function FS_TagFill(Str:string; FillChar:char; Size:integer):string;
var j :integer;
var sTmp:string;
begin
  j := Size - Length(Str);
  sTmp := Str+StringOfChar(FS_spc,j);
  Result := sTmp;
end;

//---------------------------------------------------------------------------
//This splits a sub string out of the provided str starting at location Loc
//until either the separator character is found, or the end of the string is found.
//returns a result of where the next search will start (separator+1).
//if Str is empty, or attempt to read past end of string, returns
//SubStr empty, and returned Location is set to end of the string.
function FS_ExtractString(Str:string; Loc:integer; Separator:char; var SubStr:string):integer;
var i : integer;
var Last,Size:integer;
begin
  Size := length(Str);
  Last := Size;
  for i := Loc to Size do
  begin
    if Str[i] = Separator then
    begin
      Last := i;
      break;
    end;
  end;
  SubStr := Copy(Str,Loc,Last-Loc);
  if Last >= Size then Last := Size else Last := Last+1;
  Result := Last;
end;


//---------------------------------------------------------------------------
//strip a substring from Src and place it into Dest.
//starts at Start location in Src string and copies Count characters
//returns actual size of the returned string
function FS_SubStr(sSrc:string; var sDest:string; Start:integer; Count:integer):integer;
//var Size:integer;
begin
  sDest := Copy(sSrc,Start,Count);
  Result := length(sDest);
end;

//-----------------------------------------------------------------------
//Convert Mic Cal date to a display string. rets null str if date not valid
// How = DATE_NORMAL (dd-mm-yy), DATE_SHORT (dd-mm-yy)
// DATE_COMPUTER (yyyy-mm-dd)
// If data is provided in Str, it will be left in front of the time string.
// Assumes DS_ReadMicCalibration() was called previously to update mic cal date.
//---------------------------------------------------------------------------
function FS_DateStr(var Date:FD_tCalDate; var Str:string; How:integer):string;
var sTag : string;
begin
  if ((Date.Year = 0)or(Date.Month = 0)or(Date.Day = 0)) then
  begin
    Result := Str;
    Exit;
  end;

  sTag := '/';
  if (How = DATE_COMPUTER) then
  begin
    FS_Znum(Date.Year,4,Str,sTag);
    FS_Znum(Date.Month,-2,Str,sTag);
    FS_Znum(Date.Day,-2,Str,FS_sBLANK);
  end
  else
  begin
    if (How = DATE_US_SHORT) then
      FS_Znum(Date.Year mod 100,-2,Str,sTag)
    else FS_Znum(Date.Year,4,Str,sTag);
    FS_Znum(Date.Month,-2,Str,sTag);
    FS_Znum(Date.Day,-2,Str,FS_sBLANK);
  end;
  Result := Str;
end;

//---------------------------------------------------------------------------
//Convert six byte serial number to ascii Hex
//If x>0 then "0x" is added to the start of the string.
function FS_HexID(var Serial : array of BYTE; x:integer):string;
var N:integer;
var sTmp:string;
begin
  if (x > 0) then
  begin
    sTmp := '0x';
  end;
  N := (Serial[5] shr 4) and $0F;
   if((N>0)or(x>=0)) then
   begin
     sTmp := sTmp + FS_cHex[N];
   end;
  N := Serial[5] and $0F;
   if((N>0)or(x>=0)) then
   begin
     sTmp := sTmp + FS_cHex[N];
   end;
  N := (Serial[4] shr 4) and $0F;
   if((N>0)or(x>=0)) then
   begin
     sTmp := sTmp + FS_cHex[N];
   end;
  N := Serial[4] and $0F;
   if((N>0)or(x>=0)) then
   begin
     sTmp := sTmp + FS_cHex[N];
   end;
  N := (Serial[3] shr 4) and $0F;
   if((N>0)or(x>=0)) then
   begin
     sTmp := sTmp + FS_cHex[N];
   end;
  N := Serial[3] and $0F;
   if((N>0)or(x>=0)) then
   begin
     sTmp := sTmp + FS_cHex[N];
   end;
  N := (Serial[2] shr 4) and $0F;
   if((N>0)or(x>=0)) then
   begin
     sTmp := sTmp + FS_cHex[N];
   end;
  N := Serial[2] and $0F;
   if((N>0)or(x>=0)) then
   begin
     sTmp := sTmp + FS_cHex[N];
   end;
  N := (Serial[1] shr 4) and $0F;
   if((N>0)or(x>=0)) then
   begin
     sTmp := sTmp + FS_cHex[N];
   end;
  N := Serial[1] and $0F;
   if((N>0)or(x>=0)) then
   begin
     sTmp := sTmp + FS_cHex[N];
   end;
  N := (Serial[0] shr 4) and $0F;
   if((N>0)or(x>=0)) then
   begin
     sTmp := sTmp + FS_cHex[N];
   end;
  sTmp := sTmp + FS_cHex[Serial[0] and $0F];
  Result := sTmp;
end;

//-------------------------------------------------------------------
// If data is provided in sBeg, it will be placed in front of the string.
// Assumes DS_ReadMicCalibration() was called previously to update mic cal date.
function FS_FileSizeStr(RawSize:DWORD):string;
var sTmp : string;
begin
  if (RawSize < 10000) then
  begin
    sTmp := FS_IntToStr(RawSize);
  end
  else if (RawSize < (1024*999)) then
  begin
    sTmp := FS_IntToStr(RawSize div 1024) + 'K';
  end
  else
  begin
    sTmp := FS_IntToStr(RawSize div (1000*1024)) + 'M';
  end;
  Result := sTmp;
end;


//-------------------------------------------------------------------
function FS_UpperChar(C:char):char;
begin
  if (C >= 'a') and (C <= 'z') then
  begin
    C := char(ord(C) and $BF);  //convert lower case letter to upper case
  end;
  Result := C;
end;

//-------------------------------------------------------------------
//Convert lower case string to upper case.
//If sDest is NULL, converts sSrc string in place.
//If sDest is not null, places new upper case string in sDest.
function FS_Upper(sSrc:string):string;
begin
  Result := UpperCase(sSrc);
end;

//---------------------------------------------------------------------------
//convert device ID number to hex string
function FS_GetDeviceIDString(var DevID:FD_tDeviceID; TrimStr:boolean):string;
var sTmp:string;
begin
  FS_BinaryToHex(6,DevID.Serial,sTmp);
  if (TrimStr = true) then FS_TrimLeadingZeros(sTmp);
  Result := sTmp;
end;

//---------------------------------------------------------------------------
//convert Options bit flags to a hex string
function FS_GetOptionsString(Options:DWORD; TrimStr:boolean):string;
var sTmp:string;
begin
  sTmp := FS_DwordToHex(Options);
  if (TrimStr = true) then FS_TrimLeadingZeros(sTmp);
  Result := sTmp;
end;


//---------------------------------------------------------------------------
//Version is the version number, Decimal is the position of the decimal point
//zero decimal point assumes non-decimal type version number
function FS_GetVersionString(Version:integer; Decimal:integer):string;
var sVer:string;
begin
  if (Decimal > 0) then
  begin
    FS_Rnum(Version, Decimal, 0, sVer, FS_sBLANK);  // decimal places=2
  end
  else
  begin
    FS_Inum(Version, 0, sVer, FS_sBLANK);  //non-decimal style version number
  end;
  Result := sVer;
end;

//---------------------------------------------------------------------------
//convert language number to language string
function FS_GetFryeLanguageString(Language:integer; var Str:string):boolean;
begin
  case (Language) of
    ENGLISH: begin
               Str := 'English';
               Result := true;
             end;
    FRENCH: begin
              Str := 'French';
              Result := true;
            end;
    GERMAN: begin
              Str := 'German';
              Result := true;
            end;
    SPANISH: begin
               Str := 'Spanish';
               Result := true;
             end;
    else begin
      Str := 'Unknown';
      Result := false;
    end;
  end;//endswitch(Language)
end;

//---------------------------------------------------------------------------
//convert language string (first char) to language number
function FS_GetFryeLanguageNumber(Str:string):integer;
var cLang:char;
var Stmp : string;
begin
   sTmp := UpperCase(Str);
   cLang := sTmp[1];
   case (cLang) of
     'E': Result := ENGLISH;
     'F': Result := FRENCH;
     'G': Result := GERMAN;
     'S': Result := SPANISH;
     'V': Result := ENGLISH;  //treat "V" the same as English for now
     else Result := UNKNOWN;
   end;//endswitch(sLanguage)
end;

//---------------------------------------------------------------------------
//Version is the version number, Decimal is the position of the decimal point
//zero decimal point assumes non-decimal type version number
//If Language > 0, tacks on the langauge letter at the end of the number
function FS_GetVerLangString(Version:integer; Decimal:integer; Language:integer):string;
//var pSC : ^tachar; //pointer to final target string
var sTmp:string;
var sVer:string;
begin
  if (Decimal > 0) then
  begin
    FS_Rnum(Version, Decimal, 0, sTmp, FS_sBLANK);  // decimal places=2
  end
  else
  begin
    FS_Inum(Version, 0, sTmp, FS_sBLANK);  //non-decimal style version number
  end;
  sVer := sTmp;
  if (Language > 0) then
  begin
    if (FS_GetFryeLanguageString(Language, sTmp) = true) then
    begin
      SetLength(sTmp,1); //only keep first letter
      sVer := sVer + sTmp;
    end;
  end;//endif(Language)
  Result := sVer;
end;

//-------------------------------------------------------------------
//returns true if the label line has no data
//Note: labelLines are pchar strings
function FS_BlankLabelLine(LabelLine:pchar):boolean;
var i:integer;
begin
  for i:=0 to 27 do
  begin
    if (ord(LabelLine[i]) <> $20) then
    begin
      Result := false;
      Exit;
    end;
    if (LabelLine[i] = #0) then break;
  end;
  Result := true;
end;


{========================================================}
{Pascal specific routines}
{========================================================}

{------------------------------------------------}
function onoff(value:integer):str255;
var temp : str255;
begin
  str(value,temp);
  case value of
    0 : onoff := 'OFF (0)';
    1 : onoff := 'ON  (1)';
  else
    onoff := temp;
  end;
end;

{convert a byte to hex}
function FS_HexB(B:byte):str255;
begin
  FS_HexB[1] := FS_cHex[B shr 4];
  FS_HexB[2] := FS_cHex[B and $f];
  FS_HexB[0] := #2;
end;

{convert a word to hex}
function FS_HexW(W:word):str255;
Begin
  FS_HexW[1] := FS_cHex[hi(W) shr 4];
  FS_HexW[2] := FS_cHex[hi(W) and $f];
  FS_HexW[3] := FS_cHex[lo(W) shr 4];
  FS_HexW[4] := FS_cHex[lo(W) and $f];
  FS_HexW[0] := #4;
end;

{convert a longint to hex}
function FS_HexL(L:longint):str255;
Begin
  FS_HexL := FS_HexW(Dval(L).Hw)+FS_HexW(Dval(L).Lw);
end;

{-----------------------------------------------------}
function FS_GetStampTimeStr(Stamp:longint; LongStr:boolean; var Valid:boolean):str255;
var hour,minute,second:word;
    S1,S2,S3:string[12];
    Temp : longint;
begin
  Valid := false;
  FS_GetStampTimeStr := '        ';
  if Stamp = 0 then Exit;
  Temp := Stamp and $1ffff;
  Second := Temp mod 60;
  Minute := (Temp div 60) mod 60;
  Hour := Temp div 3600;
{  asm
    mov ax,word ptr [Stamp]
    mov dx,word ptr [Stamp+2]
    and dx,$0001
    mov cx,60
    div cx
    mov [Second],dx
    mov dx,0
    div cx
    mov [Minute],dx
    mov [Hour],ax
  end;}
  if Minute > 59 then Exit;
  if Second > 59 then Exit;
  if Hour > 23 then Exit;

  S1 := FS_IntStr(Hour)+':';
  if length(S1) < 3 then S1 := '0'+S1;
  S2 := FS_IntStr(Minute);
  if length(S2) < 2 then S2 := '0'+S2;
  S3 := FS_IntStr(second);
  if length(S3) < 2 then S3 := ':0'+S3
  else S3 := ':'+S3;
  if not(LongStr) then
    FS_GetStampTimeStr := S1 + S2
  else
    FS_GetStampTimeStr := S1 + S2 + S3;
  Valid := true;
end;

{----------------------------------------------------}
function FS_ValidTimeStamp(Stamp:longint):boolean;
var t,Second,Hour,Minute:word;
    Temp : longint;
begin
  FS_ValidTimeStamp := false;
  if Stamp = 0 then Exit;
  if (Stamp shr 26) = 0 then Exit; {year}
  t := (Stamp shr 22) and $000f;
  if (t = 0) or (t > 12) then Exit; {month}
  t := (Stamp shr 17) and $001f;
  if (t = 0) or (t > 31) then Exit; {day}
  Temp := Stamp and $1ffff;
  Second := word(Temp mod 60);
  Minute := word(longint(Temp div 60) mod 60);
  Hour := word(Temp div 3600);

{  asm
    mov ax,word ptr [Stamp]
    mov dx,word ptr [Stamp+2]
    and dx,$0001
    mov cx,60
    div cx
    mov [Second],dx
    mov dx,0
    div cx
    mov [Minute],dx
    mov [Hour],ax
  end;}
  if Hour > 23 then Exit;
  if Minute > 59 then Exit;
  if Second > 59 then Exit;

  FS_ValidTimeStamp := true;
end;

{-----------------------------------------------------}
function FS_GetStampDateStr(Stamp:longint; var Valid:boolean):str255;
var Year,Month,Day:word;
    S1,S2 : string[12];
begin
  Valid := false;
  FS_GetStampDateStr := '        ';
  if Stamp = 0 then Exit;
  Year := (Stamp shr 26) + 1990;
  Month := (Stamp shr 22) and $000f;
  Day := (Stamp shr 17) and $001f;
  if (Day > 31) or (Day = 0) then Exit;
  if (Month > 12) or (Month = 0) then Exit;
  if Year < 1991 then Exit;

  S1 := FS_IntStr(Month)+'/';
  if length(S1) < 3 then S1 := '0'+S1;
  S2 := FS_IntStr(Day)+'/';
  if length(S2) < 3 then S2 := '0'+S2;
  FS_GetStampDateStr := S1 + S2 + copy(FS_IntStr(Year),3,2);
  Valid := true;
end;

{-------------------------------------------------}
function FS_PackedFryeDateToStr(CalDate:longint; var Valid:boolean):str255;
var Year,Month,Day:word;
    S1,S2 : string[12];
begin
  Valid := false;
  FS_PackedFryeDateToStr := '        ';
  Day := (CalDate shr 24) and $ff;
  Month := CalDate and $f;
  Year := (CalDate shr 4) and $fff;
  if CalDate = 0 then Exit;

  if (Day > 31) or (Day = 0) then Exit;
  if (Month > 12) or (Month = 0) then Exit;
  if Year = 0 then Exit;

  S1 := FS_IntStr(Month)+'/';
  if length(S1) < 3 then S1 := '0'+S1;
  S2 := FS_IntStr(Day)+'/';
  if length(S2) < 3 then S2 := '0'+S2;
  FS_PackedFryeDateToStr := S1 + S2 + copy(FS_IntStr(Year),3,2);
  Valid := true;
end;

{-------------------------------------------------}
function FS_UnpackedFryeDateToStr(CalDate:FD_tCalDate; var Valid:boolean):str255;
var Year,Month,Day:word;
    S1,S2 : string[12];
begin
  Valid := false;
  FS_UnpackedFryeDateToStr := '        ';
  Day := CalDate.Day;
  Month := CalDate.Month;
  Year := CalDate.Year;
  if (Year = 0) then Exit;

  if (Day > 31) or (Day = 0) then Exit;
  if (Month > 12) or (Month = 0) then Exit;
  if Year = 0 then Exit;

  S1 := FS_IntStr(Month)+'/';
  if length(S1) < 3 then S1 := '0'+S1;
  S2 := FS_IntStr(Day)+'/';
  if length(S2) < 3 then S2 := '0'+S2;
  FS_UnpackedFryeDateToStr := S1 + S2 + copy(FS_IntStr(Year),3,2);
  Valid := true;
end;

{-------------------------------------------------}
function FS_GetUnpackedFryeDateToStr(FryeDate:FD_tCalDate):str255;
var S:str255;
var valid:boolean;
begin
  S := FS_UnpackedFryeDateToStr(FryeDate,Valid);
  if length(FS_sTrim(S)) = 0 then
    S := '?';
  Result := S;
end;

function FS_GetPackedFryeDateToStr(FryeDate:FD_tPackedCalDate):str255;
var S:str255;
var valid:boolean;
begin
  S := FS_PackedFryeDateToStr(FryeDate,Valid);
  if length(FS_sTrim(S)) = 0 then
    S := '?';
  Result := S;
end;

{Convert longint to string}
function FS_IntStr(i:longint):str255;
var S : str255;
begin
  str(i,S);
  FS_IntStr := s;
end;

{Convert int to string right justified by L chars long total}
function FS_RightStr(i:integer; L:byte):str255;
var stemp:str255;
begin
  str(i,stemp);
  while length(stemp) < L do
    stemp := ' '+stemp;
  FS_RightStr:=stemp;
end;

{Convert int to string left justified by L chars long total}
function FS_LeftStr(i:integer; L:byte):str255;
var stemp:str255;
begin
  str(i,stemp);
  while length(stemp) < L do
    stemp := stemp+' ';
  FS_LeftStr:=stemp;
end;

{--------------------------------------}
function FS_sTrim(S:str255):str255;
var tS:str255;
begin
  tS := S;
  while (tS[0] > #0) and (ord(tS[1]) < $21) {"!"}
  do delete(tS,1,1);
  while (tS[length(tS)] > #0) and (ord(tS[1]) < $21) {"!"}
  do delete(tS,length(tS),1);
  FS_sTrim := tS;
end;

{------------------------------------------------}
function FS_Str2D(value:integer):str255;
var ts1,ts2,ts3 : str255;
    i,ts : integer;
begin
  ts3 := '';
  ts := (value{+5}) div 100;  {force roundup}
  str(ts,ts1);
  if ((value{+5}) < 0) and (ts = 0) then ts1 := '-0';
  ts := abs((value{+5}) mod 100) div 10;
  str(ts,ts2);
  if length(ts1) < 4 then
    for i := 1 to 4 - length(ts1) do
      ts3 := ts3+' ';
  FS_Str2D := ts3+ts1+'.'+ts2;
end;

{------------------------------------------------}
function FS_Str2Df(value:integer):str255;
var ts1,ts2,ts3 : str255;
    i,ts : integer;
begin
  ts3 := '';
  ts := (value{+5}) div 100;
  str(ts,ts1);
  if ((value{+5}) < 0) and (ts = 0) then ts1 := '-0';
  ts := abs((value{+5}) mod 100);  {force roundup}
  str(ts,ts2);
  if length(ts2) < 2 then ts2 := ts2+'0';
  if length(ts1) < 4 then
    for i := 1 to 4 - length(ts1) do
      ts3 := ts3+' ';
  FS_Str2Df := ts3+ts1+'.'+ts2;
end;

{------------------------------------------------------}
function FS_Str3D(value:integer):str255;
var ts1,ts2,ts3 : str255;
    i,ts : integer;
begin
  ts3 := '';
  ts := (value{+5}) div 100;  {force roundup}
  str(ts,ts1);
  if ((value{+5}) < 0) and (ts = 0) then ts1 := '-0';
  ts := abs((value{+5}) mod 100) {div 10};
  str(ts,ts2);
  if length(ts2) < 2 then ts2 := '0'+ts2;
  if length(ts1) < 4 then
    for i := 1 to 4 - length(ts1) do
      ts3 := ts3+' ';
  FS_Str3D := ts3+ts1+'.'+ts2;
end;

function FS_DbStr100(Value:integer; ShowOff:boolean):str255;
begin
  if (Value = 0) and ShowOff then
    FS_DbStr100 := 'OFF (0dB)'
  else
    FS_DbStr100 := FS_IntStr(Value div 100)+'dB';
end;

function FS_DbStr(Value:integer; ShowOff:boolean):str255;
begin
  if (Value = 0) and ShowOff then
    FS_DbStr := 'OFF (0dB)'
  else
    FS_DbStr := FS_Str2Df(Value)+'dB';
end;

{------------------------------------------------}
function FS_OnOff(value:integer):str255;
var temp : str255;
begin
  str(value,temp);
  case value of
    0 : FS_OnOff := 'OFF (0)';
    1 : FS_OnOff := 'ON  (1)';
  else
    FS_OnOff := temp;
  end;
end;

{------------------------------------------------}
{converts Frye Temperature (C*100) into a string}
function FS_TemperatureStr(Temperature:integer):str255;
var sTemp:string;
begin
  if (Temperature = INVALID_DATA16) then
    sTemp := '<unknown>'
  else if ((Temperature > 10000)or(Temperature < -10000)) then
    sTemp := '<error>'
  else
    FS_Znum(Temperature div 100, 0, sTemp,'C'); //make string
  Result := sTemp;
end;

{------------------------------------------------}
{This converts a degree*64 value into a decimal string}
function FS_Deg64(value:integer):str255;
var ts1,ts2,ts3 : str255;
    i,ts : integer;
begin
  ts3 := '';
  ts := value div 64;
  str(ts,ts1);
  if (value < 0) and (ts = 0) then ts1 := '-0';
  ts := trunc(abs(value mod 64) * (100 / 64));
  str(ts,ts2);
  if length(ts1) < 4 then
    for i := 1 to 4 - length(ts1) do
      ts3 := ts3+' ';
  Result := ts3+ts1+'.'+ts2;
end;


end.
