//---------------------------------------------------------------------------
//FryeTools toolkit file for applications using Fryers32.DLL or FryeCom.DLL
//Copyright 2010 Frye Electronics, Inc.
//Written by Michael Day as of 30 Aug 2010
//You may use this code for any lawful purpose without fees or royalties.
//This code is intended as an example showing how to communicate with Fonix
//equipment using the Fryers32 or FryeCom device driver on the Windows operating system.
//No warranties are express or implied in the use of this code.
//This will compile under any version of Borland Delphi (written using Delphi V3.00)
//---------------------------------------------------------------------------

unit FryeTools;
interface
uses Windows, SysUtils, FryeDefs, FryeStr;

//---------------------------------------------------------------------------

var FT_Crc16 : WORD = 0;              // global Crc16 var
const FT_BackSlash:char = char($5C); //'\\';

function FT_ConKeyWaiting:boolean; //console key waiting
function FT_ConGetKey:char;        //read console key

function FT_Timer:integer;           //system millisecond counter value
function FT_Tick(Tick:integer):integer; //user controlled resolution timer value
procedure FT_Delay(Value:integer);    //delay in ms
function FT_XDelay(var StartTime:integer; Delay:integer):boolean;
procedure FT_CopyFullID(var Dest:FD_tDeviceID; Src:FD_tDeviceID);
function FT_CompareFullID(var ID1; var ID2):boolean;
function FT_BuildPathName(FilePath:string; FileName:string):string;

procedure FT_RemoveLastDelimiter(var Str:string);
function FT_FileSize(PathName:string; var Size:DWORD):boolean;

function FT_CheckCrc8(Size:integer; var Data : array of BYTE):BYTE;
function FT_FastCrc16(Size:integer; var Data : array of BYTE; Crc16:WORD):WORD;
function FT_CheckFastCrc16(Size:integer; var Data : array of BYTE):WORD;
function FT_ChangeFileExt(FileName:string; NewExt:string):string;
function FT_RemoveFileExt(FileName:string):string;

function FT_ValidCalDate(var CalDate:FD_tCalDate):boolean;
function FT_UnpackCalDate(PackedCalDate:FD_tPackedCalDate; var CalDate:FD_tCalDate):boolean;
function FT_PackCalDate(var CalDate:FD_tCalDate):FD_tPackedCalDate;
function FT_UnpackCurveStamp(CrvStamp:FD_tCrvTimeStamp; var CrvTime:FD_tTime):boolean;

function Argval(var S; var Index:integer):longint;
function UnHexNib(C:char; var Value:word):boolean;
function UnhexWord(var S:str255; var Value:word):boolean;
function UnhexLong(var S:str255; var Value:longint):boolean;
function StrToWord(var pstr:str255; var Value:word):boolean;
function StrToLong(var pstr:str255; var Value:longint):boolean;

const Hex : array[0..15] of char = '0123456789ABCDEF';
function HexToWord(W:word):str255;


implementation
//uses Windows;

//---------------------------------------------------------------------------
const ft_Crc8Table : array[0..255] of BYTE = (
    0, 94,188,226, 97, 63,221,131,194,
    156,126, 32,163,253, 31, 65, 157,195, 33,127,252,162, 64, 30, 95,
    1,227,189, 62, 96,130,220, 35,125,159,193, 66, 28,254,160, 225,
    191, 93,  3,128,222, 60, 98, 190,224,  2, 92,223,129, 99, 61,124,
    34,192,158, 29, 67,161,255, 70, 24,250,164, 39,121,155, 197,132,
    218, 56,102,229,187, 89,  7, 219,133,103, 57,186,228, 6, 88, 25,
    71,165,251,120, 38,196,154, 101, 59,217,135,  4, 90, 184,230,167,
    249, 27, 69,198,152,122, 36, 248,166, 68, 26,153, 199, 37,123, 58,
    100,134,216, 91,  5,231,185, 140,210, 48,110, 237,179, 81, 15, 78,
    16,242,172, 47,113,147,205, 17, 79,173, 243,112, 46,204,146,211,
    141,111, 49,178,236, 14, 80, 175,241, 19, 77,206,144,114, 44,109,
    51,209,143, 12, 82,176,238, 50,108, 142,208, 83, 13,239,177,240,
    174, 76, 18,145,207, 45,115, 202, 148,118, 40,171,245, 23, 73,  8,
    86,180,234,105, 55,213,139, 87,  9,235,181, 54,104,138,212,149,
    203, 41,119,244,170, 72, 22, 233,183, 85, 11,136,214, 52,106, 43,
    117,151,201, 74, 20,246, 168, 116, 42,200,150, 21, 75,169,247,182,
    232, 10, 84,215,137, 107, 53);

//---------------------------------------------------------------------------
const ft_Crc16Table : array[0..255] of WORD = (
        $0000, $c0c1, $c181, $0140, $c301, $03c0, $0280, $c241,
        $c601, $06c0, $0780, $c741, $0500, $c5c1, $c481, $0440,
        $cc01, $0cc0, $0d80, $cd41, $0f00, $cfc1, $ce81, $0e40,
        $0a00, $cac1, $cb81, $0b40, $c901, $09c0, $0880, $c841,
        $d801, $18c0, $1980, $d941, $1b00, $dbc1, $da81, $1a40,
        $1e00, $dec1, $df81, $1f40, $dd01, $1dc0, $1c80, $dc41,
        $1400, $d4c1, $d581, $1540, $d701, $17c0, $1680, $d641,
        $d201, $12c0, $1380, $d341, $1100, $d1c1, $d081, $1040,
        $f001, $30c0, $3180, $f141, $3300, $f3c1, $f281, $3240,
        $3600, $f6c1, $f781, $3740, $f501, $35c0, $3480, $f441,
        $3c00, $fcc1, $fd81, $3d40, $ff01, $3fc0, $3e80, $fe41,
        $fa01, $3ac0, $3b80, $fb41, $3900, $f9c1, $f881, $3840,
        $2800, $e8c1, $e981, $2940, $eb01, $2bc0, $2a80, $ea41,
        $ee01, $2ec0, $2f80, $ef41, $2d00, $edc1, $ec81, $2c40,
        $e401, $24c0, $2580, $e541, $2700, $e7c1, $e681, $2640,
        $2200, $e2c1, $e381, $2340, $e101, $21c0, $2080, $e041,
        $a001, $60c0, $6180, $a141, $6300, $a3c1, $a281, $6240,
        $6600, $a6c1, $a781, $6740, $a501, $65c0, $6480, $a441,
        $6c00, $acc1, $ad81, $6d40, $af01, $6fc0, $6e80, $ae41,
        $aa01, $6ac0, $6b80, $ab41, $6900, $a9c1, $a881, $6840,
        $7800, $b8c1, $b981, $7940, $bb01, $7bc0, $7a80, $ba41,
        $be01, $7ec0, $7f80, $bf41, $7d00, $bdc1, $bc81, $7c40,
        $b401, $74c0, $7580, $b541, $7700, $b7c1, $b681, $7640,
        $7200, $b2c1, $b381, $7340, $b101, $71c0, $7080, $b041,
        $5000, $90c1, $9181, $5140, $9301, $53c0, $5280, $9241,
        $9601, $56c0, $5780, $9741, $5500, $95c1, $9481, $5440,
        $9c01, $5cc0, $5d80, $9d41, $5f00, $9fc1, $9e81, $5e40,
        $5a00, $9ac1, $9b81, $5b40, $9901, $59c0, $5880, $9841,
        $8801, $48c0, $4980, $8941, $4b00, $8bc1, $8a81, $4a40,
        $4e00, $8ec1, $8f81, $4f40, $8d01, $4dc0, $4c80, $8c41,
        $4400, $84c1, $8581, $4540, $8701, $47c0, $4680, $8641,
        $8201, $42c0, $4380, $8341, $4100, $81c1, $8081, $4040
);



//---------------------------------------------
//dummy procedure to do nothing.
procedure nop;
begin
  //do nothing
end;

{++++++++++++++++++++++++++++++++++++++++++}
{misc console input routines}
{Used for console applications}

{returns true if console key ready to be read}
{only a signel character is read at a time - crude, but works}
var ft_KeyData : integer = -1;
function FT_ConKeyWaiting:boolean;
var InputEvents : Windows.DWORD;
var InputEventsRead : Windows.DWORD;
var KeyBuf : Windows.TInputRecord;
begin
  if ft_KeyData < 0 then
  begin
    GetNumberOfConsoleInputEvents(GetStdHandle(STD_INPUT_HANDLE), InputEvents);
    if InputEvents <> 0 then
    begin
      ReadConsoleInput(GetStdHandle(STD_INPUT_HANDLE),KeyBuf,1,InputEventsRead);
      if (KeyBuf.EventType = KEY_EVENT) and (KeyBuf.Event.KeyEvent.bKeyDown=true) then
      begin
        ft_KeyData := integer(KeyBuf.Event.KeyEvent.AsciiChar);
      end;
    end;
  end;
  Result := ft_KeyData >= 0;
end;

{returns the console key char that was waiting to be read.}
{if no key is waiting, waits for one to happen.}
{Note: works with ascii characters only}
function FT_ConGetKey:char;
begin
  if ft_KeyData < 0 then
    while not(FT_ConKeyWaiting) do {nop};
  if ft_KeyData >= 0 then
    Result := char(ft_KeyData)
  else Result := #255;
  ft_KeyData := -1; //set Keydata back to -1 to indicate it is empty
end;


//---------------------------------------------
//This returns the system level millisecond timer value
function FT_Timer:integer;
begin
  Result := GetTickCount();
end;


//---------------------------------------------
//Like FT_Timer, except allows user to set the resolution (1ms minimum)
//If value is < 1 just returns the 1ms timer value.
function FT_Tick(Tick:longint):longint;
begin
  if Tick > 0 then
    Result := longint(GetTickCount()) div Tick
  else Result := GetTickCount();
end;


//---------------------------------------------
//This delays by the number of ms that was passed
procedure FT_Delay(Value:longint);
var StartTime:longint;
begin
  StartTime := GetTickCount();
  while(FOREVER) do
  begin
    if ((longint(GetTickCount()) - StartTime) > Value) then break;
  end;
end;

//----------------------------------------------
//process delay in ms - this starts a timer then exits. You can then go do something else
//and periodcically call this to see if your time is up.
//returns the current timer value. Delay is the amount of time to wait.
//StartTime is the original Starting time returned by the function when Delay=0.
//Example Usage:
// int StartTime;
// myXDelay(StartTime,0); //initialize counter
// while(true) {
//   if (myXDelay(StartTime,1000) == true) break; //wait for true
//   DoSomethingElse  //do something else while waiting
// }
function FT_XDelay(var StartTime:integer; Delay:integer):boolean;
begin
  if (Delay = 0) then
  begin
    StartTime := GetTickCount();
    Result := true;
    Exit;
  end;
  if ((longint(GetTickCount()) - StartTime) < Delay) then
  begin
    Result := false;
  end
  else
  begin
    Result := true;
  end;
end;

//---------------------------------------------------------------------------
//Copy a Dallas ID number to another location (eight bytes long)
procedure FT_CopyFullID(var Dest:FD_TdeviceID; Src:FD_tDeviceID);
begin
  Dest := Src;
end;

//---------------------------------------------------------------------------
//Compare two Dallas ID numbers (eight bytes long)
//Return TRUE if the same, or FALSE if different
function FT_CompareFullID(var ID1; var ID2):boolean;
const Size=8;
var i:integer;
begin
  for i:=0 to (Size-1) do
  begin
    if (tByteArray(ID1)[i] <> tByteArray(ID2)[i]) then
    begin
      Result := false;
      Exit;
    end;
  end;
  Result := true;
end;

//---------------------------------------------
//combines PathName with FileName to create a fully qualified PathName
//if backslash is missing from the path, adds it in.
//if Path is empty, just returns the FileName.
//returns a pointer to PathName as the function result.
function FT_BuildPathName(FilePath:string; FileName:string):string;
var Size:integer;
var FullPathName : string;
begin
  Size := Length(FilePath);
  if (Size = 0) then
  begin
    FullPathName := FileName; //scopy returns emtpy PathName if FileName is empty
  end
  else
  begin
    if (FilePath[Size] = FT_BackSlash) then ////"\\") then
    begin
      FullPathName := FilePath + FileName + FS_sBLANK;
    end
    else
    begin
      FullPathName := FilePath + FT_BackSlash + FileName; //"\\"
    end;
  end;
  Result := FullPathName;
end;

//---------------------------------------------
//removes the extent from a FileName (can be a fully qualified name)
//returns a pointer to DestName as the function result.
//function FT_RemoveFileExt(var DestName:string; FileName:string):string;
function FT_RemoveFileExt(FileName:string):string;
begin
  Result := FT_ChangeFileExt(Filename,'');
end;

//---------------------------------------------
//Changes the extent of a FileName (can be a fully qualified name)
//if DestName == null, or FileName == null, returns nothing.
//returns a pointer to DestName as the function result.
//function FT_ChangeFileExt(var DestName:str255; FileName:str255; NewExt:str255):string;
function FT_ChangeFileExt(FileName:string; NewExt:string):string;
begin
  Result := FT_ChangeFileExt(Filename,NewExt);
end;

//---------------------------------------------------------------------------
procedure FT_RemoveLastDelimiter(var Str:string);
var Last:integer;
begin
  Last := length(Str);
  if (Str[Last] = FT_BackSlash) then Delete(Str,Last,1);
end;

//---------------------------------------------
//if the file exists, returns true and the size of the file,
//if the file does not exist, returns false and zero size.
//Does not include hidden or system files.
var sr:TSearchRec;
var iAttributes:integer;
function FT_FileSize(PathName:string; var Size:DWORD):boolean;
begin
  //FS_PCharCopy(pPathName,PathName);
  iAttributes := faReadOnly or faArchive or faHidden;
  if (FindFirst(PathName, iAttributes, sr) = 0) then
  begin
    Size := sr.Size;
    Result := true;
  end
  else
  begin
    Size := 0;
    Result := false;
  end;
  FindClose(sr);
end;

//---------------------------------------------------------------------------
// Check the Crc8 of the serial number supplied Data array.
// Size = Array length in bytes. Data* = Array pointer.
// Assumes Crc8 of data is in the last byte of the array.
// If Crc8 is not the last byte in the array, returns Crc8 of the array
// else Returns 0 if Crc8=0 (last byte is a valid Crc8 of the data).
// Note: a Crc8 is much faster than a Crc16, but not as accurate
// If Data is all zeros, the crc will be zero.
//---------------------------------------------------------------------------
function FT_CheckCrc8(Size:integer; var Data : array of BYTE):BYTE;
var i:integer;
var Crc8:BYTE;
begin
  Crc8 := 0;          //Init the Crc8
  for i:=0 to (Size-1) do //Do all n bytes
  begin
    Crc8 := ft_Crc8Table[Crc8 xor Data[i]];
  end;
  Result := Crc8;
end;

//---------------------------------------------------------------------------
// This procedure calculates the Crc16 of the data byte X pass to it.
// The result is accumulated in global variable Crc16 which is a word type.
// Note: The Crc16 result from the Dallas part comes back inverted.
// The Polynomial function is: (X exp 16) + (X exp 15) + (X exp 2) + 1
//---------------------------------------------------------------------------
//calculate 16-bit CRC...  original code including lookup table
//directly derived from a mailing list posting by Mark Crispin via Aaron
function FT_FastCrc16(Size:integer; var Data : array of BYTE; Crc16:WORD):WORD;
var i:integer;
begin
  for i:= 0 to (Size-1) do
  begin
    Crc16 := WORD((Crc16 shr 8) xor ft_Crc16Table[(Crc16 xor Data[i]) and $ff]);
  end;
  Result := Crc16;
end;


//---------------------------------------------------------------------------
//calculate 16-bit CRC...  original code including lookup table
//directly derived from a mailing list posting by Mark Crispin via Aaron
function FT_CheckFastCrc16(Size:integer; var Data : array of BYTE):WORD;
var MyCrc16:WORD;
begin
  MyCrc16 := 0;
  Result := FT_FastCrc16(Size,Data,MyCrc16);
end;

//---------------------------------------------------------------------------
//Checks cal date. If good cal date returns TRUE, else returns FALSE
function FT_ValidCalDate(var CalDate:FD_tCalDate):boolean;
begin
  if ((CalDate.Month = $0f)or(CalDate.Day = $ff)or(CalDate.Year = 0)) then
    Result := false
  else Result := true;
end;



//---------------------------------------------------------------------------
//This unpacks the MicCal DWORD into the unpacked CalDate variable
//dd:ee:yyy:m where "yyy" = year, "m"=month, "dd"=day, "ee"=expire
//---------------------------------------------------------------------------
function FT_UnpackCalDate(PackedCalDate:FD_tPackedCalDate; var CalDate:FD_tCalDate):boolean;
begin
  CalDate.Year := WORD((PackedCalDate shr 4) and $0fff); //year of last cal
  CalDate.Month := BYTE(PackedCalDate and $000f);          //month of last calibration
  CalDate.Day := BYTE((PackedCalDate shr 24) and $00ff);    //day of last calibration
  CalDate.Expire := WORD((PackedCalDate shr 16) and $00ff); //months until expiration
  if (FT_ValidCalDate(CalDate) = false) then
  begin
    CalDate := FD_NoCalDate; //if unprogramed, dallas has -1 in it
    Result := false;
  end
  else
  begin
    Result := true;
  end;
end;

//---------------------------------------------------------------------------
//This packs the CalDate variable down to the MicCal DWORD
//dd:ee:yyy:m where "yyy" = year, "m"=month, "dd"=day, "ee"=expire
//---------------------------------------------------------------------------
function FT_PackCalDate(var CalDate:FD_tCalDate):FD_tPackedCalDate;
var PackedCalDate:FD_tPackedCalDate;
begin
  if ((CalDate.Year = 0)or(CalDate.Month = 0)or(CalDate.Day = 0)) then
  begin
    PackedCalDate := FD_NoPackedDate;
  end
  else
  begin
    PackedCalDate := ((CalDate.Year and $0fff) shl 4) +
                    (CalDate.Month and $000f) +
                    ((CalDate.Day and $00ff) shl 24) +
                    ((CalDate.Expire and $00ff) shl 16);
  end;
  Result := PackedCalDate;
end;

//---------------------------------------------------------------
//               Curve Frame Time Stamp Format
//----------- high word --------- ---------- low word -----------
//F E D C B A 9 8 7 6 5 4 3 2 1 0 F E D C B A 9 8 7 6 5 4 3 2 1 0
//|---------| |-----| |-------| |-------------------------------|
//   Year      Month     Day          Seconds in Day  H*M*S
//   (0-63)     (1-12)   (1-31)              (0-86399)
//   0=1990     1=Jan                   00:00:00 to 23:59:59
//---------------------------------------------------------------

//This unpacks a curve frame time stamp to the MSDOS/Windows date/time record format
//returns false is timestamp is invalid (0), else returns true.
function FT_UnpackCurveStamp(CrvStamp:FD_tCrvTimeStamp; var CrvTime:FD_tTime):boolean;
var Secs:integer;
begin
  CrvTime.Year := WORD(((CrvStamp shr 27) and $003F) + 1990);
  CrvTime.Month := BYTE((CrvStamp shr 23) and $000F);
  CrvTime.Day := BYTE((CrvStamp shr 18) and $001F);
  CrvTime.spare := 0;
  Secs := (CrvStamp and $1FFFF);
  CrvTime.Hour := BYTE(Secs div 3600);
  Secs := (Secs mod 3600);
  CrvTime.Minute := BYTE(Secs div 60);
  CrvTime.Second := BYTE(Secs mod 60);
  CrvTime.Millisecond := 0;
  if (CrvStamp = 0) then Result := false
  else Result := true;
end;


{-----------------------------------------------}

  {CONV ASCII # TO BINARY - my own version}
  function Argval(var S; var Index:integer):longint;
  type ByteArray = array[0..255] of byte;
  var Data : ByteArray absolute S;
  var Done : boolean;
  var Value : integer;
  var Work : longint;
  begin
    Done := false;
    Work := 0;
    Index := 0;
    if Data[0] = ord('-') then
      inc(Index);
    while not(Done) do
    begin
      Value := Data[Index]-ord('0');
      if (Value < 0) or (Value > 10) then
        done := true
      else
      begin
        Work := (Work*10)+Value;
        inc(Index);
      end;
    end;
    if Data[0] = ord('-') then
      Work := -(Work);
    ArgVal := Work;
  end;

{Convert a hex nibble into a binary value}
{returns false if failed}
function UnHexNib(C:char; var Value:word):boolean;
begin
  UnHexNib := false;
  C := upcase(C);
  if (C < '0') or (C > 'F') then Exit;
  if (C > '9') and (C < 'A') then Exit;
  if C > '9' then
    Value := (ord(C) - 7) and $f
  else
    Value := ord(C) and $f;
  UnHexNib := true;
end;

{convert a hex value into a binary word}
{returns fale if failed}
function UnhexWord(var S:str255; var Value:word):boolean;
var i,V1,V2 : word;
begin
  UnHexWord := false;
  Value := 0;
  i := 2;
  while i < length(S) do
  begin
     if not(UnHexNib(S[i],V1)) then Exit;
     if not(UnHexNib(S[i+1],V2)) then Exit;
     Value := (Value shl 8)+(V1 shl 4)+V2;
     inc(i,2);
     UnHexWord := true;
  end;
end;

{convert a hex value into a long int}
{returns false if failed}
function UnhexLong(var S:str255; var Value:longint):boolean;
var i,V1,V2 : word;
begin
  UnHexLong := false;
  Value := 0;
  i := 2;
  while i < length(S) do
  begin
    if not(UnHexNib(S[i],V1)) then Exit;
    if not(UnHexNib(S[i+1],V2)) then Exit;
    Value := (Value shl 8)+(V1 shl 4)+V2;
    inc(i,2);
    UnHexLong := true;
  end;
end;

{convert a string into a binary long int}
{can handle Pascal hex ($) or decimal strings}
{returns false if failed}
function StrToLong(var pstr:str255; var Value:longint):boolean;
var Index : integer;
begin
  StrToLong := false;
  Value := 0;
  if length(Pstr) = 0 then Exit;

  if Pstr[1] = '$' then
  begin
    StrToLong := UnhexLong(Pstr,Value);
  end
  else
  begin
    if length(Pstr) > 254 then dec(Pstr[0]);
    Pstr[length(Pstr)+1] := #0;
    Value := Argval(Pstr[1],Index);
    if Index = 0 then
      StrToLong := false
    else
      StrToLong := true;
  end;
end;

{convert a string into a binary word}
{can handle Pascal hex ($) or decimal strings}
{returns false if failed}
function StrToWord(var pstr:str255; var Value:word):boolean;
var Index : integer;
begin
  StrToWord := false;
  Value := 0;
  if length(Pstr) = 0 then Exit;

  if Pstr[1] = '$' then
  begin
    StrToWord := UnhexWord(Pstr,Value);
  end
  else
  begin
    if length(Pstr) > 254 then dec(Pstr[0]);
    Pstr[length(Pstr)+1] := #0;
    Value := Argval(Pstr[1],Index);
    if Index = 0 then
      StrToWord := false
    else
      StrToWord := true;
  end;
end;

function HexToWord(W:word):str255;
var S:str255;
begin
   S[0] := #4;
   S[1] := Hex[hi(W) shr 4];
   S[2] := Hex[hi(W) and $f];
   S[3] := Hex[lo(W) shr 4];
   S[4] := Hex[lo(W) and $f];
   HexToWord := S;
end;

end.
