
//Fryers interface unit for Borland Delphi/Pascal
//This will compile under Borland Delphi V3.0 or later
//Copyright 2010 Frye Electronics, Inc.
//Written by Michael Day as of 10 Oct 2010
//----------------------------------------------------
//If you have experience with the older MSDOS based Fryers TSR
//you will find this calling interface to the Fryers driver to be
//exactly the same. It was done this way on purpose to allow easy
//upgrading of old programs and to minimize the learning time
//to upgrading old systems up to Win32.

unit Fryers32;
interface
uses Windows,sysutils,Fryers;


//The call interface in the DLL looks like this
//typedef void __stdcall (*tCallFryers)(F_RegsType* IRegs);
//type tCallFryers = procedure(var FIregs:F_RegsType); stdcall;
var CallFryersProc : procedure(var FIregs:F_RegsType); stdcall;

const FryersLibHandle : integer = 0;
const FryersActive : boolean = false;
const FRYERS32_PortHandle : tHandle = 0; //Call FRYERS32_GetFryersVersion() to update this value
var FRYERS32_DllFilename : array[0..1999] of char = '';  //name of the DLL file being used

const FRYERS32_DLL_NAME : string = 'fryers32.dll';
const FRYECOM_DLL_NAME : string = 'fryecom.dll';

function FRYERS32_LastComPort():integer;
function FRYERS32_GetShortFryersVersion(var Version:integer):integer;
function FRYERS32_GetVersionString(var Vers:string):integer;
function FRYERS32_GetLongFryersVersion(var Version:integer):integer;

implementation

//----------------------------------------------------
//This is a wrapper to make the DLL call safe. It also eliminates the
//need to manage the DLL in your program since it is automatically managed here
//If this is a first time caller, the Fryers32.DLL is searched for on the normal
//search path for the computer. If found, the driver is loaded into memory and
//the CallFryers interface is looked for. If found, the procedure pointer is
//updated and the call to the Fryers driver is made. Note: The call to initialize
//the Fryers dll will take up to 200ms to perform. Also if you change the baudrate
//there may be a 200ms delay as Windows does it's thing (I have no control over that).
//Returns FRYERS_FILE_NOT_FOUND if the DLL was not found on the search path.
//returns FRYERS_ENTRY_NOT_FOUND if the CallFryers procedure call was not found.
function CallFryers(var IRegs : F_RegsType):integer;
begin
  FryersActive := true;
  if (FryersLibHandle = NULL) then
  begin
    CallFryersProc := NIL;
    //strcpy(DllFilename,FRYECOM_DLL_NAME);
    StrPLCopy(RS232_DllFilename,FRYECOM_DLL_NAME,sizeof(RS232_DllFilename));
    FryersLibHandle := LoadLibrary(RS232_DllFilename);
    if (FryersLibHandle = NULL) then
    begin
      //if fryecom not found, try for fryers32 instead
      StrPLCopy(RS232_DllFilename,FRYERS_DLL_NAME,sizeof(RS232_DllFilename));
      FryersLibHandle := LoadLibrary(RS232_DllFilename);
      if (FryersLibHandle = NULL) then
      begin
        FryersActive := false;
        Result := FRYERS_FILE_NOT_FOUND;
        Exit;
      end;
    end;
  end;
  if @CallFryersProc = nil then
  begin
    @CallFryersProc := GetProcAddress(FryersLibHandle,'CallFryers');
//    CallFryersProc := (tCallFryers)GetProcAddress(FryersLibHandle, "CallFryers");
    if @CallFryersProc = nil then
  //  if (CallFryersProc = NULL) then       //check if we found the procedure we want
    begin
      FreeLibrary(FryersLibHandle); //release the DLL
      FryersLibHandle := NULL;       //and kill the variable
      //strcpy(ErrorFilename,FRYERS32_DLL_NAME);
      FryersActive := false;
      Result := FRYERS_ENTRY_NOT_FOUND;
      Exit;
    end;//endif(CallFryersProc==NULL)
  end;//endif(CallFryersProc==NULL)
  CallFryersProc(IRegs);
  FryersActive := false;
  Result := FRYERS_SUCCESS;
end;

(*
//Don't use this, the DLL will automatically be unloaded when your program shuts down.
//----------------------------------------------
//For example only. Normally you should not call this.
//Windows till release the DLL properly when the program shuts down.
bool ReleaseFryers(void) {
  bool fFreeResult = FreeLibrary(FryersLibHandle);
  return(fFreeResult);
};
*)

//---------------------------------------------------------------------------
//get the last available Fryers com port
function FRYERS32_LastComPort():integer;
  var Ver:integer;
  var Max:integer;

  //var Result:integer;
  var R:F_RegsType;
begin
  Result := FRYERS32_GetLongFryersVersion(Ver);
  //if cmd failed, assume old style max port (9)
  if (Result <> FRYERS_SUCCESS) then
  begin
    Result := MAX_OLD_COM_PORT;
    Exit;
  end;
  //check if we have the newer Fryers32 version
  if (Ver >= FRYERS32_NEW_DLL_PORT_VERSION) then //Ver610
  begin
    //if later version,
    R.AX := $0FFFC;  //see if function 0xFFFC is available first
    R.DX := 0; //port# is not important
    Result := CallFryers(R);   //ax=maxport, bx=version, cx=portstate, dx=ffff(if cmd avail), di=baudrate, si=<reserved(0)>
    //verify that the cmd is valid
    if ( (smallint(R.DX) <> smallint($FFFF)) and (Result = FRYERS_SUCCESS) ) then
    begin
      Max := R.AX;
      if (Max > 256) then Max := MAX_NUM_PORTS;
    end
    else
    begin
      Max := MAX_NEW_COM_PORT;
    end;
    Result := Max;
    Exit;
  end
  //if really old version, we only have 9 ports available
  else if (Ver < FRYERS32_GOOD_DLL_PORT_VERSION) then //below Ver520
  begin
    Max := MAX_OLD_COM_PORT; //we have 9 comports allowed in this application if old fryers (V5.18 or lower)
    Result := Max;
    Exit;
  end
  //
  else //Assume Ver520 if we get here
  begin
    Max := MAX_99_COM_PORT; //we have 99 comports allowed in this application if good Fryers (V5.20 to 6.10)
    Result := Max;
    Exit;
  end;
end;

//-------------------------------------------------
// Example call to return Fryers short version number
// Here is an example of calling the Fryers driver.
// See the documentation and sample code for a detailed
//explaination on using the Fryers driver.
function FRYERS32_GetShortFryersVersion(var Version:integer):integer;
  var R : F_RegsType;
begin
  if (Version <> NULL) then Version := 0;
  R.AX := $0FFFF;
  R.DX := 0;
  R.CX := 0;
  Result := CallFryers(R);
  if (Result <> FRYERS_SUCCESS) then Exit;
  if ((R.DX and $0FFFF) <> $0FFFF) then
  begin
    Result := RS232_BAD_FRYERS_DRIVER;
    Exit;
  end;
  if (Version <> NULL) then Version := R.AX;
end;

//-------------------------------------------------
// Example call to return Fryers long version number
function FRYERS32_GetLongFryersVersion(var Version:integer):integer;
  var R : F_RegsType;
begin
  if (Version <> NULL) then Version := 0;
  R.AX := $0FFFE;
  R.DX := 0;
  R.CX := 0;
  R.SI := 0;
  R.DI := 0;
  Result := CallFryers(R);
  if (Result <> FRYERS_SUCCESS) then Exit;
  if ((R.DX and $0FFFF) <> $0FFFF) then
  begin
    Result := RS232_BAD_FRYERS_DRIVER;
    Exit;
  end;
  if (Version <> NULL) then Version := R.BX;
end;

//-----------------------------------------------------------------------
//returns empty string or "Unknown" if could not get valid version string
//returns version number as a result of the call, or failure code if negative number
//A specific com port it not required for this, so we can use com0.
//note the first byte of the version string is the string length.
//returns negative number on failure.
function FRYERS32_GetVersionString(var Vers:string):integer;
  var i:integer;
  var Count:integer;
  var R : F_RegsType;
begin
  if (Vers <> NULL) then Vers := '';   //initialize the Ansistring if available
  R.AX := $FFFE;
  R.DX := 0;        //ask for version information (can always call this in Windows)
  Result := CallFryers(R);
  if (Result <> FRYERS_SUCCESS) then //ack serious failure here
  begin
    Exit;   //probably no DLL
  end;
  //if AX<0 or DX != 0xFFFF, or AX<0x27 it is bad news. A good dll will pass all this.
  if ((R.AX <= 0)or ((integer(R.DX) and $FFFF) <> integer($FFFF)) or ((R.AX and $00ff) < $27) ) then
  begin
    Result := FRYERS_GENERIC_ERROR;   //This error should not happen, but just in case...
    Exit;                            //empty version string comes back with this error
  end;
  Result := R.BX;         //BX has long version number
  if (Vers = NULL) then    //if Vers = null, we are done
  begin
    Exit;      //just return the version number as the result
  end;
  Count := R.CX;          //CX has the string length in bytes
  for i:=1 to Count do
  begin
    R.AX := $FFFD;
    R.DX := 0;            //ask for version string bytes
    R.CX := i;
    CallFryers(R);
    if (R.AX < 0) then
    begin
      Vers := 'Unknown';             //pass back "Unknown" to identify this error
      Result := FRYERS_GENERIC_ERROR; //This error should not happen, but just in case...
      Exit;
    end;
    Vers := Vers+char(R.AL);             //Copy the string byte to the ansistring
  end;//endfor(i)
  //if successful, Result has the long version number and Vers has the version string
end;

//---------------------------------------------------------------------------
//Converts a Pascal string to a pchar string
//MaxSize specifies the maximum possible number of characters that can be copied to pDest}
function FS_StringToPChar(pDest:pchar; Src:string; MaxSize:integer):pchar;
begin
  Result := StrPLCopy(pDest,Src,MaxSize);
end;
//=====================================================
//If Fryers is not activated: 0xFFFF cmd returns:
//AX=short version number in hex (eg V5.20 returns as 0x52)
//BX=0
//CX=<reserved>
//DX=0xFFFF=Fryers present, else Fryers not found
//DI=0
//SI=version signon string pointer
//------------------------------------
//if Fryers is activated: 0xFFFF cmd returns:
//AX=short version number in hex (eg V5.20 returns as 0x52)
//BX=Windows Comport Handle. 0=none, nz=handle
//CX=<reserved>
//DX=0xFFFF=Fryers present, else Fryers not found
//SI=version signon string pointer
//DI=BaudClock (only valid for baudrates < 115200 high baudrates return 1).
//[Baudrate=115200/BaudClock]. Use 0xFFFE cmd to get real baudrate
//------------------------------------
//if Fryers is not activated: 0xFFFE cmd returns:
//AX=short version number in hex (eg V5.20 returns as 0x52)
//BX=long version number in binary (eg V5.20 = 520)
//CX=version signon string length in bytes
//DX=0xFFFF=Fryers present, else Fryers not found
//DI=<unchanged>
//SI=<unchanged>
//------------------------------------
//If Fryers is activated: 0xFFFE cmd returns:
//AX=short version number in hex (eg V5.20 returns as 0x52)
//BX=long version number in binary (eg V5.20 = 520)
//CX=version signon string length in bytes
//DX=0xFFFF=Fryers present, else Fryers not found
//DI=<unchanged>
//SI=<unchanged>
//------------------------------------
//if Fryers is not activated: 0xFFFD cmd returns:
//AX=signon version string character at string index in CX
//BX=<reserved>
//CX=Limited to valid signon version string index
//DX=<reserved>
//DI=<reserved>
//SI=<reserved>
//------------------------------------
//If Fryers is activated: 0xFFFD cmd returns:
//AX=signon version string character at string index in CX
//BX=<reserved>
//CX=Limited to valid signon version string index
//DX=<reserved>
//DI=TimeServiceThread handle for the selected comport
//SI=PortServiceThread handle for the selected comport
//====================================================
//Notes: Fryers commands 0xFFFFF, 0xFFFE and 0xFFFD are special
//they do not require a port selection and will not allocate
//a port object when called. They are also re-entrant.
//You can call perform the commands at anytime, anyplace in the code,
//or even in a callback function. The GetFryersVersion()
//code above can be useful to determine which version of Fryers
//is loaded (or if it is not available). The function is always
//operational and will return immediately with the results.
//The Windows Com Port Handle, TimeServiceHandle and PortServiceHandle
//handles are used during debugging to gain access to the internal variables
//The Windows FRYERS32.DLL has always supported all three commands,
//however 0xFFFD had a problem prior to V5.12 and should not be used
//with those early versions. The MSDOS versions prior to V3.00
//only supported 0xFFFF. See the Fryres documentation for details.

end.
//<eof>



