//---------------------------------------------------------------------------
//Label Input handling code
//This is kept separate to unclutter the main unit.
//Copyright 2010 Frye Electronics, Inc.
//Written by Michael Day as of 30 Aug 2010
//You may use this code for any lawful purpose without fees or royalties.
//This code is intended as an example showing how to communicate with Fonix
//equipment using the Fryers32 or FryeCom device driver on the Windows operating system.
//No warranties are express or implied in the use of this code.
//---------------------------------------------------------------------------

unit LabelUnit;

interface
uses SysUtils,FryeDefs,FComDefs,FippDefs,BitRev,FCErrMsg,FryeStr,FryeTools;

var LabelText : string; {temp working storage for label entry}
var CustomLab : string; {used for entering label text and file reads}

var BitMapType : word = 2;
var BitMapXpos : word = 0;
var BitMapYpos : word = 160;
var BitMapDest : word = 0;
var   BitMapSize : longint;
var LU_BmpPalette : BmpPaletteType;    {bmp palette for the label graphics}
var LU_WinBitmap : FIPP_tFryeBitmap;  {Windows bitmap array used for label graphics}
//var LU_FryeBitmap : FIPP_tFryeBitmap; {Frye bitmap array used for label graphics}

function LU_LabelIsBlank(Size:integer; LabelText:pchar):boolean;
procedure LU_EnterUserID(var UidText:string);
procedure LU_EnterStdLabel(var LabelText:string);    {enter a std label (cmd 6)}
procedure LU_EnterExtLabel(var LabelText:string);    {enter an extended label (cmd 66)}
function LU_GetLabelFile(Cmd:integer; Filename:str255):integer;

function LU_GetBitmapFile(Filename:str255):integer;
procedure LU_ConvertBitMapToSend(var FryeBitmap : FIPP_tFryeBitmap);

//Note: these are for use with FippCore. If you use FippUnit, these are managed already in the call.
procedure LU_ConvertLabelToSend(Size:integer; LabelData:pchar; CmdArray:FC_pCmdArray);
procedure LU_ConvertLabelFromRcv(Size:integer; LabelData:pchar; RspArray:FC_pRspArray);
procedure LU_ConvertUserIDToSend(ExtID:boolean; Size:integer; IDText:pchar; CmdArray:FC_pCmdArray);
procedure LU_GetUserIDFromRcv(ExtID:boolean; Size:integer; IDText:pchar; RspArray:FC_pRspArray);

implementation


{------------------------------------------------}
{check label and if it is blank (all nulls) returns true, else returns false}
function LU_LabelIsBlank(Size:integer; LabelText:pchar):boolean;
var i:integer;
begin
  LU_LabelIsBlank := false;
  for i := 0 to pred(Size) do
  begin
    //All nulls or underscores is a blank label
    if (LabelText[i] <> #0) and (LabelText[i] <> #32) then Exit;
  end;
  LU_LabelIsBlank := true;
end;

{-----------------------------------------------}
{enter a label line}
function LU_EntLabLn(S:Str255; L:integer):str255;
var s2 : str255;
begin
  write(S,#13);
  s2 := copy(s,1,Length(S)-succ(L));
  write(s2);
  readln(s2);
  s2 := s2+'___________________________';
  if L = 18 then
    s2 := copy(s2,1,18)
  else
    s2 := copy(s2,1,27);
  LU_EntLabLn := s2;
end;

{-----------------------------------------------}
{enter User ID Text}
procedure LU_EnterUserID(var UidText:string);
begin
  UidText := '';
  writeln('Enter Label Information');
  writeln;
  UidText :=           lu_EntLabLn('L1:___________________________|',27)+#0;
  UidText := UidText + lu_EntLabLn('L2:___________________________|',27)+#0;
end;

{-----------------------------------------------}
procedure LU_EnterStdLabel(var LabelText:string);  {enter a std label (cmd 6)}
begin
  LabelText := '';
  writeln('Enter Label Information');
  writeln;
  LabelText :=             lu_EntLabLn('       Date:__________________|',18);
  LabelText := LabelText + lu_EntLabLn('     Model#:__________________|',18);
  LabelText := LabelText + lu_EntLabLn('    Serial#:__________________|',18);
  LabelText := LabelText + lu_EntLabLn('      Owner:__________________|',18);
  LabelText := LabelText + lu_EntLabLn('   Comments:__________________|',18);
  LabelText := LabelText + lu_EntLabLn('C2:___________________________|',27);
  LabelText := LabelText + lu_EntLabLn('C3:___________________________|',27);
  LabelText := LabelText + lu_EntLabLn('C4:___________________________|',27);
end;

{-----------------------------------------------}
procedure LU_EnterExtLabel(var LabelText:string);  {enter an extended label (cmd 66)}
begin
  LabelText := '';
  writeln('Enter Label Information');
  writeln;
  LabelText :=             lu_EntLabLn('L1:___________________________|',27)+#0;
  LabelText := LabelText + lu_EntLabLn('L2:___________________________|',27)+#0;
  LabelText := LabelText + lu_EntLabLn('L3:___________________________|',27)+#0;
  LabelText := LabelText + lu_EntLabLn('L4:___________________________|',27)+#0;
  LabelText := LabelText + lu_EntLabLn('L5:___________________________|',27)+#0;
  LabelText := LabelText + lu_EntLabLn('L6:___________________________|',27)+#0;
  LabelText := LabelText + lu_EntLabLn('L7:___________________________|',27)+#0;
  LabelText := LabelText + lu_EntLabLn('L8:___________________________|',27)+#0;
end;


{------------------------------------------------}
{This reads a label file (ascii text) }
{and stores it in the Fipp object ready to be sent}
{Cmd is the type of label being read (standard or extended)}
function LU_GetLabelFile(Cmd:integer; Filename:str255):integer;
var i : integer;
var f : text;
var Stmp : Str255;

begin
  if FileSearch(Filename,'') = '' then
  begin
    writeln('File not found: ',Filename);
    Result := FCE_NoFileError;                    {rats, didn't find it}
    Exit;
  end;
  assign(f,Filename);
  reset(f);

  //fillchar(LabelText,sizeof(LabelText),0);
  //fillchar(CustomLab,sizeof(CustomLab),0);
  LabelText := '';
  CustomLab := '';
  if Cmd = 6 then             {if cmd 6 do old style label}
  begin
    while not(eof(f)) do
    begin
      for i := 1 to 5 do
      begin
        fillchar(Stmp[1],18,'_');
        readln(f,Stmp);
        Stmp[0] := #18;
        LabelText := LabelText+Stmp;
      end;
      for i := 6 to 8 do
      begin
        fillchar(Stmp[1],27,'_');
        readln(f,Stmp);
        Stmp[0] := #27;
        LabelText := LabelText+Stmp;
      end;
      LabelText := LabelText+#0;
    end;
    Result := SUCCESS;
  end
  else if Cmd = 66 then
  begin
    while not(eof(f)) do
    begin
      for i := 1 to 8 do             {otherwise do new style label}
      begin
        fillchar(Stmp[1],27,'_');
        readln(f,Stmp);
        Stmp := Stmp+#0;
        Stmp[0] := #28;
        LabelText := LabelText+Stmp;
      end;
      if not(eof(f)) then
      begin
        fillchar(Stmp[1],27,' ');    {try for first custom line}
        readln(f,Stmp);
        Stmp := Stmp+#0;
        Stmp[0] := #28;
        CustomLab := Stmp;
      end;
      if not(eof(f)) then
      begin
        fillchar(Stmp[1],27,' ');    {try for second custom line}
        readln(f,Stmp);
        Stmp := Stmp+#0;
        Stmp[0] := #28;
        CustomLab := CustomLab+Stmp;
      end;
    end;
    Result := SUCCESS;
  end
  else
  begin
    Close(f);
    writeln('Error: Command format error');
    Result := FCE_CommandFail;                    {rats, didn't find it}
    Exit;
  end;
  Close(f);
end;

{-------}
{This can convert std or ext label arrays.}
{This converts the label asciiZ (pchar) 2D string array from intel to motorola format}
{and pokes it into the command array.}
{Note: This does not set the cmd number. The caller must set the size and cmd number (6 or 66).}
{Size is the number of chars to be converted, LabelData is the place where the label is located.}
procedure LU_ConvertLabelToSend(Size:integer; LabelData:pchar; CmdArray:FC_pCmdArray);
var i:integer;
begin
  {make sure Size is valid}
  if (Size < 0) or (Size > FIPP_REG_EXT_LABEL_SIZEB) then
    Size := FIPP_REG_EXT_LABEL_SIZEB;
  i := 0;
  LabelData[pred(Size)] := #0; {force last byte to null}
  while i < Size do
  begin
    CmdArray.Data[(i shr 1)] := (ord(LabelData[i]) shl 8)+
                                  ord(LabelData[succ(i)]);
    inc(i,2);
  end;
end;

{-------}
{This converts the UserID asciiZ (pchar) 2D string array from intel to motorola format}
{and pokes it into the command array.}
{Note: This does not set the cmd number. The caller must set the size and cmd number (6 or 66).}
{Size is the number of chars to be converted, TextData is the place where the text is located.}
{If ExtID = true, Text is assumed to go at the end of an extended label cmd (66).
{If ExtID = false, text is assumed to be for a normal UserID cmd (166).}
procedure LU_ConvertUserIDToSend(ExtID:boolean; Size:integer; IDText:pchar; CmdArray:FC_pCmdArray);
var SrcIndex:integer;
var CmdIndex:integer;
begin
  {If extended label format, then put the user ID at the end of the custom label}
  {otherwise it goes at the start of the CmdArray.}
  if (ExtID = true) then
    CmdIndex := FIPP_REG_EXT_LABEL_SIZEB
  else CmdIndex := 0;
  {make sure Size is valid}
  if (Size < 0) or (Size > FIPP_MAX_USERID_TEXT) then
    Size := FIPP_MAX_USERID_TEXT;

  SrcIndex := 0;
  IDText[pred(Size)] := #0; {force last byte to null}
  while SrcIndex < Size do
  begin
    CmdArray.Data[(CmdIndex shr 1)] := (ord(IDText[SrcIndex]) shl 8)+
                                         ord(IDText[SrcIndex+1]);
    inc(CmdIndex,2);
    inc(SrcIndex,2);
  end;
end;

{-------}
{This converts the received label in Motorola format to Intel asciiZ (pchar) 2D string array}
{Size is total number of chars to be converted, LabelData is where to place the data}
procedure LU_ConvertLabelFromRcv(Size:integer; LabelData:pchar; RspArray:FC_pRspArray);
var i:integer;
begin
  {make sure Size is valid}
  if (Size < 0) or (Size > FIPP_REG_EXT_LABEL_SIZEB) then
    Size := FIPP_REG_EXT_LABEL_SIZEB;
  i := 0;
  while i < Size do
  begin
    LabelData[i] := char(RspArray.Data[(i shr 1)] shr 8);
    LabelData[succ(i)] := char(RspArray.Data[(i shr 1)] and $ff);
    inc(i,2);
  end;
  LabelData[pred(Size)] := #0; {force last byte to null}
end;

{This converts the received UserID in Motorola format to Intel asciiZ (pchar) 2D string array}
{Size is total number of chars to be converted, IDTest is where to place the data}
{If ExtID = true Text is assumed to be at the end of an extended label cmd (86).}
{If ExtID = false, text is assumed to be for a normal UserID cmd (167).}
procedure LU_GetUserIDFromRcv(ExtID:boolean; Size:integer; IDText:pchar; RspArray:FC_pRspArray);
var DestIndex:integer;
var RspIndex:integer;
begin
  if (Size < 0)or(Size > FIPP_MAX_USERID_TEXT) then Size := FIPP_MAX_USERID_TEXT;
  if (ExtID = true) then
    RspIndex := FIPP_REG_EXT_LABEL_SIZEB   {if via ExtLabel start at first char past ext label portion}
  else RspIndex := 0;                      {if via std User ID cmd 167, start at loc 0}
  DestIndex := 0;
  while DestIndex < Size do
  begin
    IDText[DestIndex] := char(RspArray.Data[(RspIndex shr 1)] shr 8);
    IDText[DestIndex+1] := char(RspArray.Data[(RspIndex shr 1)] and $ff);
    inc(RspIndex,2);
    inc(DestIndex,2);
  end;
  IDText[Size-1] := #0; {force last byte to null}
end;


{------------------------------------------------}
{2/27/03 -med fixed bug that caused bad format bmp }
{file (imageSize=0) to cause an exception}
{This reads a Windows BMP file and stores it in a temp bitmap structure.}
function LU_GetBitmapFile(Filename:str255):integer;
var Err:integer;
var fb : file;
label GetLabelFileExit;

begin
  Result := FCE_UnknownError;
  lu_BmpPalette.Item[0] := 0;  {Init default palette entries}
  lu_BmpPalette.Item[1] := $00ffffff;

  if FileSearch(Filename,'') = '' then
  begin
    writeln('File not found: ',Filename);
    Result := FCE_NoFileError;                    {rats, didn't find it}
    Exit;
  end;
  assign(fb,Filename);
  reset(fb,1);
  BlockRead(fb,BmpHeader,sizeof(BmpHeader),Err);
  if Err <> sizeof(BmpHeader) then
  begin
    writeln('Error: Bitmap format error (header)');
    Result := FCE_CommandFail;                    {rats, bad bitmap file }
    goto GetLabelFileExit;
  end;
  if BmpHeader.HeaderSize >= (sizeof(BmpHeader) + 8) then
  begin
    BlockRead(fb,lu_BmpPalette,sizeof(lu_BmpPalette),Err);
    if Err <> sizeof(lu_BmpPalette) then
    begin
      writeln('Error: Bitmap format error (palette)');
      Result := FCE_CommandFail;                    {rats, bad bitmap file }
      goto GetLabelFileExit;
    end;
  end;
  Seek(fb,BmpHeader.HeaderSize);
  BitMapSize := BmpHeader.biSizeImage;
  if BitMapSize = 0 then BitMapSize := BmpHeader.Width * BmpHeader.Height div 8;
  if ((BitMapSize div 2) > FIPP_MAX_FRYE_BITMAP_DATA) then //MaxLabelBitmap) then
  begin
    writeln('Error: Bitmap format error (BMP file too big)');
    Result := FCE_CommandFail;                    {rats, bad size }
    goto GetLabelFileExit;
  end;
  if ((BmpHeader.Bits <> 1) or (FileSize(fb) <> BmpHeader.FileSize)) then
  begin
    writeln('Error: Bitmap format error (it needs to be a monochrome BMP file)');
    Result := FCE_CommandFail;                    {rats, bad size }
    goto GetLabelFileExit;
  end;
  fillchar(LU_WinBitmap.Data,sizeof(LU_WinBitmap.Data),0);
  BlockRead(fb,LU_WinBitmap.Data,BitMapSize,Err);
  if Err <> BitMapSize then
  begin
    writeln('Error: Bitmap format error');
    Result := FCE_CommandFail;                    {rats, bad bitmap file }
    goto GetLabelFileExit;
  end;
  Result:= SUCCESS;

GetLabelFileExit:
  Close(fb);
end;

{====================================================================}
{This converts the bmp data (see above) into the Frye bitmap format}
{and stores it in the FryeBitmap object ready to be sent.}
procedure LU_ConvertBitmapToSend(var FryeBitmap : FIPP_tFryeBitmap);
var Data : word;
var h,w,i,x,y:integer;
var PixelsPerWord:integer;
var PacketSize : INT16;
var BmpOffset : integer;
begin
    w := BmpHeader.Width;
    h := BmpHeader.Height;
    FryeBitMap.Style := BitMapType;  {monochrome} {only monochrome currently supported}
    FryeBitMap.Xpos := BitMapXpos;  {xpos}
    FryeBitMap.Ypos := BitMapYpos;  {ypos}
    FryeBitMap.Width := BmpHeader.Width;
    FryeBitMap.Height := BmpHeader.Height;
    FryeBitMap.Page := BitMapDest; {Only the printer can be set with this command right now}
    FryeBitMap.Size := BitmapSize; {total bitmap size in bytes}
    //Fipp.LabelBitMap.Size := (w*h/8); //BitMapSize;  {image size in bytes}
    BmpOffset:= 8;     //offset in words into Bmp structure where data starts
    FryeBitMap.StartOfs := BmpOffset + 2; //offset in words into packet for Bmp data
    if ((BitmapSize div 2) > FIPP_MAX_FRYE_BITMAP_DATA) then
      PacketSize := FIPP_MAX_FRYE_BITMAP_DATA
    else PacketSize := WORD((BitmapSize div 2)+FryeBitMap.StartOfs);

    FD_WordCopy(FryeBitmap.Data,LU_WinBitmap,FryeBitmap.StartOfs); //copy header to send array
    i := 0;
    if BitMapType = 2 then
      PixelsPerWord := 16 {if monochrome, 16pixels per word}
    else PixelsPerWord := 4; {if color, 4 pixels per word}
    h := BmpHeader.Height;           {convert from bmp format to fp35 bitmap format}
    w := BmpHeader.Width div PixelsPerWord;
    if (h > 0) and (w > 0) then
    begin
      if (BmpHeader.Width mod PixelsPerWord) <> 0 then inc(w);
      for y := 0 to h-1 do
      begin
        for x := 0 to w-1 do
        begin
          Data := LU_WinBitMap.Data[((h-1-y)*w)+x];
          Data := ((BitReverse[lo(Data)]) + (BitReverse[hi(Data)] shl 8));
          {Need to read the palette to determine black and white pixel settings}
          if (LU_BmpPalette.Item[0] = 0) then
          //  Fipp.CmdArray.Data[i+BmpOffset] := not(Data) {pal[0]=black}
          //else Fipp.CmdArray.Data[i+BmpOffset] := Data;  {pal[0]=white}
            FryeBitMap.Data[i] := not(Data) {pal[0]=black}
          else FryeBitMap.Data[i] := Data;  {pal[0]=white}
          inc(i);
        end;
      end;
    end;

end;


(*
function TclsFipp.GetImageInfo:boolean;
begin
  Result := false;
  if not SendGetImageCmd(0,0,0,0,0) then Exit;
  if not(WaitForResponse) then Exit;
  if not GetResponse then Exit;

  ImgScreenBpp := Rary[3];  //bits per pixel (1 to 8) 1=mono, 4=16 color, 8=256color
  ImgScanLineSize := Rary[4];
  ImgScreenWidth := Rary[6];
  ImgScreenHeight := Rary[7];
  if (ImgScreenBpp = 2) then
    ImgBitMask := $000f
  else if (ImgScreenBpp = 4) then
    ImgBitMask := $0003
  else if (ImgScreenBpp = 8) then
    ImgBitMask := $0001
  else ImgBitMask := 1;
  if ImgBitMask < 1 then ImgBitMask := $000f;
  if (ImgScanLineSize < 40) then ImgScanLineSize := 40;
  ImgLinesPerChunk := 1000 div (ImgScanLineSize div 2);
  ImageAvailable := true;
  BuildImageSize(ImmediateUpdate);
  BuildMemoryImageMap;
  Result := true;
end;
*)
(*
function SendBitmapImage:boolean;
begin
  if GetImageInfo() == false Exit;
  SetNoPoll(GET_IMG_POLL_TIMEOUT);
  //get first image block
  xp := 0;
  yp := 0;
  yr := 0;
  ImgIndex := 0;
  if b > ImgScreenHeight then b := ImgScreenHeight;
  if not SendGetImageCmd(xp,yp,w,b,bpp) then Exit;
  while yr < ImgScreenHeight do
  begin
    br := b;
    yr := yp;
    //get image data from instrument
    if not(WaitForResponse) then Exit;
    if not GetResponse then Exit;
    begin
      goto ErrorExit;
    end;
    if (Rary[1] and $7fff) <> GetImageCmd then Exit;

    //send request for next block before we process this one
    //That way we can display this section while waiting for the next one.
    xp := 0;
    if (yp+b) < ImgScreenHeight then
    begin
      yp := yp+b;
      b := ImgLinesPerChunk;
      if (yp+b) > ImgScreenHeight then
        b := ImgScreenHeight - yp;
      if not SendGetImageCmd(xp,yp,w,b,bpp) then Exit;
    end;
    j := rary[10]+1; //location of first word in rary
    if j > 2000 then Exit; //bad img format

    style := Rary[3]; //get style of color (2=mono, 4=16color, 8=256color)
    cnt := Rary[9];  //remember! Pascal array starts at [1]
    cnt := cnt div 2;
    if cnt > 2000 then Exit; //bad image format
    for i := 0 to cnt-1 do
    begin
      ImgBuffer[ImgIndex+i] := Rary[j+i];
    end;
    ImgIndex := ImgIndex + cnt; //(Rary[9] div 2);
  end;
end;
*)

end.
