//---------------------------------------------------------------------------
//This unit manages reading and writing to comma delimited text files.
//---------------------------------------------------------------------------
#include <vcl.h>
#pragma hdrstop

#include <io.h>
#include "TextFile.h"

//---------------------------------------------------------------------------
#pragma package(smart_init)

TTextFile* TextFile1;

//---------------------------------------------------------------------------

int __fastcall TTextFile::Exists(AnsiString aFileName) {
  return(access(aFileName.c_str(), 0) == 0);
};
//  char Name[80];
//  STR_SCopy(Name,FileName.c_str());
//  return (access(Name, 0) == 0);

//----------------------------------------------------------------
//Open a text file for reading
//returns 0=ok, -1=error
int __fastcall TTextFile::Open(AnsiString aTextFileName) {
  try {
    StringList = new TStringList();
    StringList->LoadFromFile(aTextFileName);
    aReadBuf = "";
    aWriteBuf = "";
    LineIndex = 0;
    Written = false;
    aFileName = aTextFileName;
    return(0);
  }catch(...){
    delete StringList;
    return(-1);
  }
};

//----------------------------------------------------------------
//Open a new text file for writing
//returns 0=ok, -1=error
int __fastcall TTextFile::Create(AnsiString aTextFileName) {
  try {
    StringList = new TStringList();
    aReadBuf = "";
    aWriteBuf = "";
    LineIndex = 0;
    Written = false;
    aFileName = aTextFileName;
    return(0);
  }catch(...){
    delete StringList;
    return(-1);
  }
};

//----------------------------------------------------------------
//open a file to append to the end of the text file
//returns 0=ok, -1=error
int __fastcall TTextFile::Append(AnsiString aTextFileName) {
  try {
    StringList = new TStringList();
    StringList->LoadFromFile(aTextFileName);
    aReadBuf = "";
    aWriteBuf = "";
    LineIndex = StringList->Count;
    aFileName = aTextFileName;
    Written = false;
    return(0);
  }catch(...){
    delete StringList;
    return(-1);
  }
};

//----------------------------------------------------------------
//returns the number of lines in the String list buffer
//if not valid, returns -1
int __fastcall TTextFile::LineCount(void) {
  try{
    return(StringList->Count);
  }catch(...){
    return(-1);
  }
};

//----------------------------------------------------------------
//returns the current index into the String list buffer
//if not valid, returns -1
int __fastcall TTextFile::GetIndex(void) {
  try{
    return(LineIndex);
  }catch(...){
    return(-1);
  }
};

//----------------------------------------------------------------
//returns the current index into the String list buffer
//if not valid, returns -1
//if past eof, returns 1 and index is not set
int __fastcall TTextFile::SetIndex(int NewIndex) {
  try{
    if (NewIndex >= StringList->Count) return(1);
    LineIndex = NewIndex;
    return(NewIndex);
  }catch(...){
    return(-1);
  }
};

//----------------------------------------------------------------
//Clear cWrite Buffer used to prep buffer for creating a new
//buffer string if you are not sure if the buffer is empty.
//When you follow up with a WritelnIndex() the buffer contents will
//be placed at the indicated line.
//When you follow up with a Writeln(), the contents will be added to the
//end of the file. Writeln() will leave the buffer cleared when done.
//cWriteStr() and cWriteIt() are the only valid cWrite calls
//that work with this feature.
//WritelnInt() will discard the buffer contents first
//and leave the buffer cleared when done.
//WritelnStr() will add the passed string to the end of the existing
//buffer contents and leave the buffer cleared when done.
int __fastcall TTextFile::ClearWriteBuf(void) {
  try {
    aWriteBuf = "";
    return(0);
  }catch(...){
    return(-1);
  }
};

//----------------------------------------------------------------
//write comma delimted text
//Note: be sure to call Writeln() when done writing comma data on a line
//returns 0=ok, -1=error
int __fastcall TTextFile::cWriteStr(AnsiString aData) {
  try{
    Written = true;
    if (aWriteBuf.Length() == 0)
      aWriteBuf = aData;
    else aWriteBuf += ","+aData;
    return(0);
  }catch(...){
    return(-1);
  }
};

//----------------------------------------------------------------
//write comma delimted integer
//Note: be sure to call Writeln() when done writing comma data on a line
//returns 0=ok, -1=error
int __fastcall TTextFile::cWriteInt(int iData) {
  try{
    AnsiString aData = IntToStr(iData);
    Written = true;
    if (aWriteBuf.Length() == 0)
      aWriteBuf = aData;
    else aWriteBuf += ","+aData;
    return(0);
  }catch(...){
    return(-1);
  }
};

//----------------------------------------------------------------
//Write a buffer text to the file
//returns 0=ok, -1=error
int __fastcall TTextFile::Writeln(void) {
  try{
    Written = true;
    StringList->Add(aWriteBuf);
    aWriteBuf = "";
    return(0);
  }catch(...){
    return(-1);
  }
};

//----------------------------------------------------------------
//Write a text string to the file buffer
//returns 0=ok, -1=error
int __fastcall TTextFile::WriteStr(AnsiString aData) {
  try{
    Written = true;
    aWriteBuf += aData;
    return(0);
  }catch(...){
    return(-1);
  }
};

//----------------------------------------------------------------
//Write a buffer text to the file
//returns 0=ok, -1=error
int __fastcall TTextFile::WritelnInt(int iData) {
  try{
    aWriteBuf = IntToStr(iData);
    Written = true;
    StringList->Add(aWriteBuf);
    aWriteBuf = "";
    return(0);
  }catch(...){
    return(-1);
  }
};

//----------------------------------------------------------------
//Write a line of text to the file
//returns 0=ok, -1=error
int __fastcall TTextFile::WritelnStr(AnsiString aData) {
  try{
    Written = true;
    StringList->Add(aWriteBuf + aData);
    aWriteBuf = "";
    return(0);
  }catch(...){
    return(-1);
  }
};

//----------------------------------------------------------------
//Replace a buffer text line in the file. Does not affect anything else.
//(writes out previously written comma buffer.)
//returns 0=ok, -1=error, 1=index beyond end of buffer
int __fastcall TTextFile::WritelnIndex(int Index) {
  try{
    if (Index >= StringList->Count) {
      return(1);
    }
    Written = true;
    StringList->Strings[Index] = aWriteBuf;
    aWriteBuf = "";
    return(0);
  }catch(...){
    return(-1);
  }
};

//----------------------------------------------------------------
//Replace a buffer text line in the file. Does not affect anything else.
//returns 0=ok, -1=error, 1=index beyond end of buffer
int __fastcall TTextFile::WritelnIndexStr(int Index, AnsiString aData) {
  try{
    if (Index >= StringList->Count) {
      return(1);
    }
    Written = true;
    StringList->Strings[Index] = aData;
    return(0);
  }catch(...){
    return(-1);
  }
};

//----------------------------------------------------------------
//read comma delimited text from file
//Note: be sure to call Readln() first before the first comma read of a line.
//returns 0=ok, +1=eol, -1=error
int __fastcall TTextFile::cReadStr(AnsiString* aData) {
  int CommaPos;
  try{
    if (aReadBuf.Length() == 0) {
      *aData = aReadBuf; //return empty string
      return(1);
    }//endif(Length)
    CommaPos = aReadBuf.Pos(",");
    if (CommaPos == 0) {
      *aData = aReadBuf; //no comma, so get entire line
      aReadBuf = "";
      return(0);
    }else if (CommaPos == 1) {
      *aData = ""; //only comma found, so return nothing
      aReadBuf = aReadBuf.SubString(CommaPos+1,aReadBuf.Length());
      //aReadBuf = "";
      return(0);
    }else{ //found text so return it, then update read buffer
      *aData = aReadBuf.SubString(1,CommaPos-1);
      aReadBuf = aReadBuf.SubString(CommaPos+1,aReadBuf.Length());
      return(0);
    }
  }catch(...){
    return(-1);
  }//endif(Try)
};

//----------------------------------------------------------------
//read comma delimited integer. Does not update iData if not valid
//Note: be sure to call Readln() first before the first comma read of a line.
//returns 0=ok, -1=error
int __fastcall TTextFile::cReadInt(int* iData, int Default) {
  int Result;
  AnsiString aData = "0";
  try{
    Result = cReadStr(&aData);
    if (Result == 0) {
      *iData = StrToIntDef(aData,Default);
    }else{
      *iData = Default;
    }
    return(Result);
  }catch(...){
    return(-1);
  }
};


//----------------------------------------------------------------
//read comma delimted byte. Does not update iData if not valid
//Note: be sure to call Readln() first before the first comma read of a line.
//returns 0=ok, -1=error
int __fastcall TTextFile::cReadByte(BYTE* iData, BYTE Default) {
  int Result;
  AnsiString aData = "0";
  try{
    Result = cReadStr(&aData);
    if (Result == 0) {
      *iData = (BYTE)StrToIntDef(aData,Default);
    }else{
      *iData = Default;
    }
    return(Result);
  }catch(...){
    return(-1);
  }
};


//----------------------------------------------------------------
//read a line of text from the file into read buffer
//note: any old unprocessed data in ReadBuf is tossed
//returns 0=ok, +1=eof, -1=error
int __fastcall TTextFile::Readln(void) {
  try{
    if (LineIndex >= StringList->Count) {
      return(1);
    }
    aReadBuf = StringList->Strings[LineIndex];
    LineIndex++;
    return(0);
  }catch(...){
    return(-1);
  }
};

//----------------------------------------------------------------
//read a integer from indicated line in the file (rest of line is ignored)
//note: any old unprocessed data in ReadBuf is tossed
//index is left pointing to the next line
//returns 0=ok, +1=eof, -1=error
int __fastcall TTextFile::ReadlnIndexInt(int Index, int* iData, int Default) {
  try{
    if (Index >= StringList->Count) {
      *iData = Default;
      return(1);
    }
    aReadBuf = StringList->Strings[Index];
    *iData = StrToIntDef(aReadBuf,Default);
    aReadBuf = "";
    return(0);
  }catch(...){
    *iData = Default;
    return(-1);
  }
};

//----------------------------------------------------------------
//read a integer from next line in the file (rest of line is ignored)
//note: any old unprocessed data in ReadBuf is tossed
//index is left pointing to the next line
//returns 0=ok, +1=eof, -1=error
int __fastcall TTextFile::ReadlnInt(int* iData, int Default) {
  int Result;
  try{
    Result = Readln();
    if (Result == 0) {
      Result = cReadInt(iData,Default);
      aReadBuf = "";
    }
    return(Result); 
  }catch(...){
    iData = iDefault;
    return(-1);
  }
};

//----------------------------------------------------------------
//go to the indicated line then read the line of text from the
//file into read buffer
//note: any old unprocessed data in ReadBuf is tossed
//returns 0=ok, +1=eof, -1=error
int __fastcall TTextFile::ReadlnIndex(int Index) {
  try{
    if (Index >= StringList->Count) {
      return(1);
    }
    LineIndex = Index;
    return(Readln());
  }catch(...){
    return(-1);
  }  
};

//----------------------------------------------------------------
//read a specific line of text direct from the read buffer
//Does not affect comma read buffer or indexing
//returns 0=ok, +1=eof, -1=error
int __fastcall TTextFile::ReadlnIndexStr(int Index, AnsiString* aData) {
  try{
    if (Index >= StringList->Count) {
      *aData = "";
      return(1);
    }
    *aData = StringList->Strings[Index];
    return(0);
  }catch(...){
    *aData = "";
    return(-1);
  }
};

//----------------------------------------------------------------
//read a line of text from the file automatically increments to the next line
//returns 0=ok, +1=eof, -1=error
int __fastcall TTextFile::ReadlnStr(AnsiString* aData) {
  try{
    if (LineIndex >= StringList->Count) {
      return(1);
    }else if (aReadBuf.Length() > 0) { //if have a partial string, just use that
      *aData = aReadBuf;
      aReadBuf = "";
      LineIndex++;
      return(0);
    }else{
      aReadBuf = StringList->Strings[LineIndex];
      *aData = aReadBuf;
      aReadBuf = "";
      LineIndex++;
      return(0);
    }
  }catch(...){
    return(-1);
  }
};

//----------------------------------------------------------------
//Close the file. If was writing, buffer is flushed at this time
//returns 0=ok, +1=eof, -1=error
int __fastcall TTextFile::Close(void) {
  try{
    if (Written == true) { //save it if it was written to
      StringList->SaveToFile(aFileName);
    }
    return(0);
  }catch(...) {
    delete StringList;
    return(-1);
  }
};


