//---------------------------------------------------------------------------
//This unit manages reading and writing to comma delimited text files.
//---------------------------------------------------------------------------
unit TextFile;
interface
uses classes,sysutils;

//---------------------------------------------------------------------------

type TTextFile = Class
  private
    aReadBuf : AnsiString;
    aWriteBuf : AnsiString;
  public
    aFileName : AnsiString;
    StringList : TStrings;
    LineIndex:integer;
    Written : boolean;
    function Exists(aFileName:AnsiString):boolean;
    function OpenFile(aTextFileName:AnsiString):integer;
    function CreateFile(aTextFileName:AnsiString):integer;
    function AppendFile(aTextFileName:AnsiString):integer;
    function ClearWriteBuf:integer;
    function LineCount:integer;
    function GetIndex:integer;
    function SetIndex(NewIndex:integer):integer;

    function cWriteInt(iData:integer):integer;
    function cWriteStr(aData:AnsiString):integer;
  //Note: be sure to call Writeln() when done writing comma data on a line

    function WriteStr(aData:AnsiString):integer;
    function Writeln:integer;
    function WritelnInt(iData:integer):integer;
    function WritelnIndex(Index:integer):integer;
    function WritelnStr(aData:AnsiString):integer;
    function WritelnIndexStr(Index:Integer; aData:AnsiString):integer;

  //Note: be sure to call Readln() first before the first comma read of a line.
    function cReadByte(var iData:BYTE; iDefault:BYTE):integer;
    function cReadInt(var iData:Integer; iDefault:integer):integer;
    function cReadStr(var aData:AnsiString):integer;

    function Readln:integer;
    function ReadlnInt(var iData:integer; iDefault:integer):integer;
    function ReadlnIndex(Index:integer):integer;
    function ReadlnIndexInt(Index:integer; var iData:integer; iDefault:integer):integer;
    function ReadlnStr(var aData:AnsiString):integer;
    function ReadlnIndexStr(Index:integer; var aData:AnsiString):integer;
    function Close:integer;

    //-------------------------------
    constructor Create;
    destructor Destroy; override;
end;
//---------------------------------------------------------------------------

var TextFile1 : ^TTextFile;


implementation

//---------------------------------------------------------------------------

{===================================================================}
// Implementation for the class constructor
constructor TTextFile.Create;
begin
  inherited Create;
    //    InitializeCriticalSection(Csv);
end;

{===================================================================}
destructor TTextFile.Destroy;
begin
//  DeleteCriticalSection(Csv);
  inherited Destroy;
end;

{===================================================================}

//returns true if file exists, false if not
function TTextFile.Exists(aFileName:AnsiString):boolean;
begin
  Result := FileExists(aFilename);
end;

//----------------------------------------------------------------
//Open a text file for reading
//returns 0=ok, -1=error
function TTextFile.OpenFile(aTextFileName:AnsiString):integer;
begin
  try
    StringList := TStringList.Create;
    StringList.LoadFromFile(aTextFileName);
    aReadBuf := '';
    aWriteBuf := '';
    LineIndex := 0;
    Written := false;
    aFileName := aTextFileName;
    Result := 0;
    Exit;
  except
    StringList.free;
    Result := -1;
  end;
end;

//----------------------------------------------------------------
//Open a new text file for writing
//returns 0=ok, -1=error
function TTextFile.CreateFile(aTextFileName:AnsiString):integer;
begin
  try
    StringList := TStringList.Create;
    aReadBuf := '';
    aWriteBuf := '';
    LineIndex := 0;
    Written := false;
    aFileName := aTextFileName;
    Result := 0;
    Exit;
  except
    StringList.free;
    Result := -1;
  end;
end;

//----------------------------------------------------------------
//open a file to append to the end of the text file
//returns 0=ok, -1=error
function TTextFile.AppendFile(aTextFileName:AnsiString):integer;
begin
  try
    StringList := TStringList.Create;
    StringList.LoadFromFile(aTextFileName);
    aReadBuf := '';
    aWriteBuf := '';
    LineIndex := StringList.Count;
    aFileName := aTextFileName;
    Written := false;
    Result := 0;
    Exit;
  except
    StringList.free;
    Result := -1;
  end;
end;

//----------------------------------------------------------------
//returns the number of lines in the String list buffer
//if not valid, returns -1
function TTextFile.LineCount:integer;
begin
  try
    Result := StringList.Count;
    Exit;
  except
    Result := -1;
  end;
end;

//----------------------------------------------------------------
//returns the current index into the String list buffer
//if not valid, returns -1
function TTextFile.GetIndex:integer;
begin
  try
    Result := LineIndex;
    Exit;
  except
    Result := -1;
  end;
end;

//----------------------------------------------------------------
//returns the current index into the String list buffer
//if not valid, returns -1
//if past eof, returns 1 and index is not set
function TTextFile.SetIndex(NewIndex:integer):integer;
begin
  try
    if (NewIndex >= StringList.Count) then
    begin
      Result := 1;
      Exit;
    end;
    LineIndex := NewIndex;
    Result := NewIndex;
    Exit;
  except
    Result := -1;
  end;
end;

//----------------------------------------------------------------
//Clear cWrite Buffer used to prep buffer for creating a new
//buffer string if you are not sure if the buffer is empty.
//When you follow up with a WritelnIndex() the buffer contents will
//be placed at the indicated line.
//When you follow up with a Writeln(), the contents will be added to the
//end of the file. Writeln() will leave the buffer cleared when done.
//cWriteStr() and cWriteIt() are the only valid cWrite calls
//that work with this feature.
//WritelnInt() will discard the buffer contents first
//and leave the buffer cleared when done.
//WritelnStr() will add the passed string to the end of the existing
//buffer contents and leave the buffer cleared when done.
function TTextFile.ClearWriteBuf:integer;
begin
  try
    aWriteBuf := '';
    Result := 0;
    Exit;
  except
    Result := -1;
  end;
end;

//----------------------------------------------------------------
//write comma delimted text
//Note: be sure to call Writeln() when done writing comma data on a line
//returns 0=ok, -1=error
function TTextFile.cWriteStr(aData:AnsiString):integer;
begin
  try
    Written := true;
    if (Length(aWriteBuf) = 0) then
      aWriteBuf := aData
    else aWriteBuf := aWriteBuf+','+aData;
    Result := 0;
    Exit;
  except
    Result := -1;
  end;
end;

//----------------------------------------------------------------
//write comma delimted integer
//Note: be sure to call Writeln() when done writing comma data on a line
//returns 0=ok, -1=error
function TTextFile.cWriteInt(iData:integer):integer;
var aData : AnsiString;
begin
  try
    aData := IntToStr(iData);
    Written := true;
    if (Length(aWriteBuf) = 0) then
      aWriteBuf := aData
    else aWriteBuf := aWriteBuf+','+aData;
    Result := 0;
    Exit;
  except
    Result := -1;
  end;
end;

//----------------------------------------------------------------
//Write a buffer text to the file
//returns 0=ok, -1=error
function TTextFile.Writeln:integer;
begin
  try
    Written := true;
    StringList.Add(aWriteBuf);
    aWriteBuf := '';
    Result := 0;
    Exit;
  except
    Result := -1;
  end;
end;

//----------------------------------------------------------------
//Write a text string to the file buffer
//returns 0=ok, -1=error
function TTextFile.WriteStr(aData:AnsiString):integer;
begin
  try
    Written := true;
    aWriteBuf := aWriteBuf+aData;
    Result := 0;
    Exit;
  except
    Result := -1;
  end;
end;

//----------------------------------------------------------------
//Write a buffer text to the file
//returns 0=ok, -1=error
function TTextFile.WritelnInt(iData:integer):integer;
begin
  try
    aWriteBuf := IntToStr(iData);
    Written := true;
    StringList.Add(aWriteBuf);
    aWriteBuf := '';
    Result := 0;
    Exit;
  except
    Result := -1;
  end;
end;

//----------------------------------------------------------------
//Write a line of text to the file
//returns 0=ok, -1=error
function TTextFile.WritelnStr(aData:AnsiString):integer;
begin
  try
    Written := true;
    StringList.Add(aWriteBuf + aData);
    aWriteBuf := '';
    Result := 0;
    Exit;
  except
    Result := -1;
  end;
end;

//----------------------------------------------------------------
//Replace a buffer text line in the file. Does not affect anything else.
//(writes out previously written comma buffer.)
//returns 0=ok, -1=error, 1=index beyond end of buffer
function TTextFile.WritelnIndex(Index:integer):integer;
begin
  try
    if (Index >= StringList.Count) then
    begin
      Result := 1;
      Exit;
    end;

    Written := true;
    StringList.Strings[Index] := aWriteBuf;
    aWriteBuf := '';
    Result := 0;
    Exit;
  except
    Result := -1;
  end;
end;

//----------------------------------------------------------------
//Replace a buffer text line in the file. Does not affect anything else.
//returns 0=ok, -1=error, 1=index beyond end of buffer
function TTextFile.WritelnIndexStr(Index:integer; aData:AnsiString):integer;
begin
  try
    if (Index >= StringList.Count) then
    begin
      Result := 1;
      Exit;
    end;
    Written := true;
    StringList.Strings[Index] := aData;
    Result := 0;
    Exit;
  except
    Result := -1;
  end;
end;

//----------------------------------------------------------------
//read comma delimited text from file
//Note: be sure to call Readln() first before the first comma read of a line.
//returns 0=ok, +1=eol, -1=error
function TTextFile.cReadStr(var aData:AnsiString):integer;
var CommaPos:integer;
begin
  try
    if (Length(aReadBuf) = 0) then
    begin
      aData := aReadBuf; //return empty string
      Result := 1;
      Exit;
    end;//endif(Length)
    CommaPos := Pos('.',aReadBuf);
    if (CommaPos = 0) then
    begin
      aData := aReadBuf; //no comma, so get entire line
      aReadBuf := '';
      Result := 0;
      Exit;
    end
    else if (CommaPos = 1) then
    begin
      aData := ''; //only comma found, so return nothing
      aReadBuf := Copy(aReadBuf,CommaPos+1,Length(aReadBuf));
      //aReadBuf = "";
      Result := 0;
      Exit;
    end
    else //found text so return it, then update read buffer
    begin
      aData := Copy(aReadBuf,1,CommaPos-1);
      aReadBuf := Copy(aReadBuf,CommaPos+1,Length(aReadBuf));
      Result := 0;
      Exit;
    end;
  except
    Result := -1;
  end;//endif(Try)
end;

//----------------------------------------------------------------
//read comma delimited integer. Does not update iData if not valid
//Note: be sure to call Readln() first before the first comma read of a line.
//returns 0=ok, -1=error
function TTextFile.cReadInt(var iData:integer; iDefault:integer):integer;
var aData : AnsiString;
begin
  try
    aData := '0';
    Result := cReadStr(aData);
    if (Result = 0) then
    begin
      iData := StrToIntDef(aData,iDefault);
    end
    else
    begin
      iData := iDefault;
    end;
    Exit;
  except
    Result := -1;
  end;
end;


//----------------------------------------------------------------
//read comma delimted byte. Does not update iData if not valid
//Note: be sure to call Readln() first before the first comma read of a line.
//returns 0=ok, -1=error
function TTextFile.cReadByte(var iData:BYTE; iDefault:BYTE):integer;
var aData : Ansistring;
begin
  aData := '0';
  try
    Result := cReadStr(aData);
    if (Result = 0) then
    begin
      iData := BYTE(StrToIntDef(aData,iDefault));
    end
    else
    begin
      iData := iDefault;
    end;
    Exit;
  except
    Result := -1;
  end;
end;


//----------------------------------------------------------------
//read a line of text from the file into read buffer
//note: any old unprocessed data in ReadBuf is tossed
//returns 0=ok, +1=eof, -1=error
function TTextFile.Readln:integer;
begin
  try
    if (LineIndex >= StringList.Count) then
    begin
      Result := 1;
      Exit;
    end;
    aReadBuf := StringList.Strings[LineIndex];
    inc(LineIndex);
    Result := 0;
    Exit;
  except
    Result := -1;
  end;
end;

//----------------------------------------------------------------
//read a integer from indicated line in the file (rest of line is ignored)
//note: any old unprocessed data in ReadBuf is tossed
//index is left pointing to the next line
//returns 0=ok, +1=eof, -1=error
function TTextFile.ReadlnIndexInt(Index:integer; var iData:integer; iDefault:integer):integer;
begin
  try
    if (Index >= StringList.Count) then
    begin
      iData := iDefault;
      Result := 1;
      Exit;
    end;
    aReadBuf := StringList.Strings[Index];
    iData := StrToIntDef(aReadBuf,iDefault);
    aReadBuf := '';
    Result := 0;
    Exit;
  except
    iData := iDefault;
    Result := -1;
  end;
end;

//----------------------------------------------------------------
//read a integer from next line in the file (rest of line is ignored)
//note: any old unprocessed data in ReadBuf is tossed
//index is left pointing to the next line
//returns 0=ok, +1=eof, -1=error
function TTextFile.ReadlnInt(var iData:integer; iDefault:integer):integer;
begin
  try
    Result := Readln();
    if (Result = 0) then
    begin
      Result := cReadInt(iData,iDefault);
      aReadBuf := '';
    end;
  except
    iData := iDefault;
    Result := -1;
  end;
end;

//----------------------------------------------------------------
//go to the indicated line then read the line of text from the
//file into read buffer
//note: any old unprocessed data in ReadBuf is tossed
//returns 0=ok, +1=eof, -1=error
function TTextFile.ReadlnIndex(Index:integer):integer;
begin
  try
    if (Index >= StringList.Count) then
    begin
      Result := 1;
      Exit;
    end;
    LineIndex := Index;
    Result := Readln();
    Exit;
  except
    Result := -1;
  end;
end;

//----------------------------------------------------------------
//read a specific line of text direct from the read buffer
//Does not affect comma read buffer or indexing
//returns 0=ok, +1=eof, -1=error
function TTextFile.ReadlnIndexStr(Index:integer; var aData:AnsiString):integer;
begin
  try
    if (Index >= StringList.Count) then
    begin
      aData := '';
      Result := 1;
      Exit;
    end;
    aData := StringList.Strings[Index];
    Result := 0;
    Exit;
  except
    aData := '';
    Result := -1;
  end;
end;

//----------------------------------------------------------------
//read a line of text from the file automatically increments to the next line
//returns 0=ok, +1=eof, -1=error
function TTextFile.ReadlnStr(var aData:AnsiString):integer;
begin
  try
    if (LineIndex >= StringList.Count) then
    begin
      Result := 1;
      Exit;
    end
    else if (Length(aReadBuf) > 0) then //if have a partial string, just use that
    begin
      aData := aReadBuf;
      aReadBuf := '';
      inc(LineIndex);
      Result := 0;
      Exit;
    end
    else
    begin
      aReadBuf := StringList.Strings[LineIndex];
      aData := aReadBuf;
      aReadBuf := '';
      inc(LineIndex);
      Result := 0;
      Exit;
    end;
  except
    Result := -1;
  end;
end;

//----------------------------------------------------------------
//Close the file. If was writing, buffer is flushed at this time
//returns 0=ok, +1=eof, -1=error
function TTextFile.Close:integer;
begin
  try
    if (Written = true) then //save it if it was written to
    begin
      StringList.SaveToFile(aFileName);
    end;
    Result := 0;
    Exit;
  except
    StringList.free;
    Result := -1;
  end;
end;

end.

